// BigONSpaceTables.c: creates Tables with O(N) size
////////////////////////////////////////////////////////////////////////////////
//
//   Copyright (C) 2008   Georg S. Weber
//
//   This file is part of the Essen Modular package programs.
//
//   The Essen Modular package programs are
//   free software:
//   you can redistribute them and/or modify them
//   under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   The Essen Modular package programs are
//   distributed in the hope that they will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License along
//   with the Essen Modular package programs,
//   see the text file COPYING in this directory.
//   If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////
//
//   history / version (the version is YYYY-MM-DD of the last change)
//   2008-02-22: creation (gsw) of first version
//   2008-02-27: bulk of coding (gsw)
//   2008-03-03: added (0:0) as first symbol in list for fallback purposes (gsw)
//   2008-03-06: addition mod N now for x in [0 .. N] (gsw)
//
////////////////////////////////////////////////////////////////////////////////

#include "./BigONSpaceTables.h"
#include "./PrimeFactors.h"
#include "./EratosthenesFermat.h"

#include "./BigOofNSquareSpaceTables.h"

//local variables
static EF_sng sizeOfP1ZNZplusOne    = 0x0;  //used also as a flag that
                                            // tables are in use

static EF_sng currentNinUse         = 0x0;

static EF_sng p_N                   = 0x0;  //first prime factor of N
                                            //(needed in indexOfRepresentative)

static int8_t lenghPrimesCurrentNinUse = 0x0;

static EF_sng primesCurrentNinUse[MAX_PRIME_FACTORS][2];

static EF_sng * arrayForAdditionModN;

static EF_sng * arrayOfInversesModN;

static EF_sng * arrayOfRepresentatives;

static EF_sng * arrayOfVSMLowerEntries;

static EF_sng * arrayOfCoprimesAndNegatives;

static EF_sng * arrayOfHelperData;

static EF_sng * arrayOfXmodN;


//functions
uint8_t BigONSpaceTablesConstruction(EF_sng N)
{
    EF_sng i, j, indexRep;        //counter variables
    
    uint16_t primesSubset, complement;
    
    uint16_t primesSubsetBound;
    
    EF_sng temp;        //temporary variable
    
    if(0x0 != sizeOfP1ZNZplusOne)
    {
        // caller shall destruct old Tables first
        return(0x1);
    }
    
    if(N < 0x2)
    {
        //invalid N
        return(0x2);
    }
    
    currentNinUse = N;
    
    //malloc
    arrayForAdditionModN = (EF_sng *) malloc( (0x2 * currentNinUse + 0x1)
                                              * sizeof(EF_sng)
                                            );
    
    if(NULL == arrayForAdditionModN)
    {
        return(0x3);
    }
    
    //create list for addition
    for(i = 0x0; i <= currentNinUse; i++)
    {
        for(j = 0x0; j < currentNinUse; j++)
        {
            if( (i+j) < currentNinUse )
            {
                arrayForAdditionModN[i+j] = i+j;
            }
            else
            {   
                arrayForAdditionModN[i+j] = i+j-currentNinUse;
            }
        }
    }
    arrayForAdditionModN[currentNinUse + currentNinUse] = 0x0;
    //  well, the following code would have done the same
    //  (except for the two entries at index "2 * currentNinUse - 1" and above)
    //for(i = 0x0; i < currentNinUse; i++)
    //{
    //    arrayForAdditionModN[i]                 = i;
    //    arrayForAdditionModN[i + currentNinUse] = i;
    //}
    //  however the above explicit creation mimics the usage we have in mind
    
    //malloc
    arrayOfInversesModN = (EF_sng *) malloc(currentNinUse * sizeof(EF_sng));

    if(NULL == arrayOfInversesModN)
    {
        free(arrayForAdditionModN);
        
        return(0x4);
    }
    
    //create list of inverses
    EratosthenesFermat( currentNinUse, arrayOfInversesModN );
    
    
    lenghPrimesCurrentNinUse = PrimeFactorization( currentNinUse,
                                                   primesCurrentNinUse
                                                 );
    
    if( 0x0 >= lenghPrimesCurrentNinUse )
    {
        free(arrayOfInversesModN);
        
        free(arrayForAdditionModN);
        
        return(0x9);
    }
    
     //calculate sizeOfP1ZNZplusOne iteratively
    sizeOfP1ZNZplusOne = 0x1;
    for( i = 0x0; i < lenghPrimesCurrentNinUse; i++)
    {
        sizeOfP1ZNZplusOne *= primesCurrentNinUse[i][0] + 0x1;
        for( j = 0x0; j < primesCurrentNinUse[i][1] - 0x1; j++ )
        {
            sizeOfP1ZNZplusOne *= primesCurrentNinUse[i][0];
        }
    }
    sizeOfP1ZNZplusOne += 0x1;  //add one for zeroeth (invalid) entry (0 : 0)
    
    //calculate p_N, the first prime factor of N
    p_N = N;    //correct if N itself is a prime
    //adjust if necessary
    for(i = 0x1; i < currentNinUse; i++)
    {
        if(0x0 == arrayOfInversesModN[i])
        {
            p_N = i;
            
            break;
        }
    }
    
    
    //malloc for size of (almost) O(N)
    arrayOfHelperData = (EF_sng *) calloc( N<<lenghPrimesCurrentNinUse,
                                           sizeof(EF_sng)
                                         );
    
    if(NULL == arrayOfHelperData)
    {
        free(arrayOfInversesModN);
        
        free(arrayForAdditionModN);
        
        sizeOfP1ZNZplusOne = 0x0;
        
        return(10);
    }
    
    primesSubsetBound = 0x1<<lenghPrimesCurrentNinUse;
    for( i = 0x0; i < currentNinUse; i++)
    {
        //for primesSubset == 0, i.e. modulo 1, we have 0 == 1;
        //and we're interested in multiplicative structures here
        arrayOfHelperData[i] = 0x1;
    }
    for(primesSubset = 0x1; primesSubset < primesSubsetBound; primesSubset++)
    {
        temp = 0x1;
        for(i = 0x0; i < lenghPrimesCurrentNinUse; i++)
        {
            if( primesSubset & (0x1<<i) )
            {
                for( j = 0x0; j < primesCurrentNinUse[i][1]; j++)
                {
                    temp *= primesCurrentNinUse[i][0];
                }
            }
        }
        for(i = 0x0; i < N/temp; i++)
        {
            for(j= 0x0; j < temp; j++)
            {
                arrayOfHelperData[primesSubset*N + temp*i + j] = j;
            }
        }
    }                                                
    
    //malloc for size of (almost) O(N)
    arrayOfRepresentatives = (EF_sng *) malloc( sizeOfP1ZNZplusOne * 0x2
                                                * sizeof(EF_sng)
                                              );
    
    if(NULL == arrayOfRepresentatives)
    {
        free(arrayOfHelperData);
        
        free(arrayOfInversesModN);
        
        free(arrayForAdditionModN);
        
        sizeOfP1ZNZplusOne = 0x0;
        
        return(0x6);
    }

    //create list of representatives
    //(insert the invalid (0:0) as first entry in order to be able)
    //(to pass an invalid array index , namely 0, the index of (0:0))
    arrayOfRepresentatives[0x0]     = 0x0;
    arrayOfRepresentatives[0x1]     = 0x0;
    //create list of (true) representatives
    //extra code for the "first" case ("temp == 0x1", no primes in the product)
    //(needed e.g. to take (1:0) correctly into account)
    for(indexRep = 0x1; indexRep <= currentNinUse; indexRep++)
    {
        arrayOfRepresentatives[(indexRep<<0x1)      ] = 0x1;
        arrayOfRepresentatives[(indexRep<<0x1) + 0x1] = indexRep - 0x1;
    }
    
    //"middle" cases
    primesSubsetBound = (0x1<<lenghPrimesCurrentNinUse)-0x1;
    for(primesSubset = 0x1; primesSubset < primesSubsetBound; primesSubset++)
    {                                                
        temp        = 0x1;
        complement  = 0x0;
        for(i = 0x0; i < lenghPrimesCurrentNinUse; i++)
        {
            if( primesSubset & (0x1<<i) )
            {
                temp *= primesCurrentNinUse[i][0];
            }
            else
            {
                complement |= 0x1<<i;
            }
        }
        for(i = temp; i < currentNinUse; i += temp)
        {
            if( 0x1 == arrayOfHelperData[complement*currentNinUse + i] )
            {
                for(j = 0x1; j < currentNinUse; j++)
                {
                    if(0x1==arrayOfHelperData[primesSubset*currentNinUse + j])
                    {
                        arrayOfRepresentatives[(indexRep<<0x1)      ] =   i;
                        arrayOfRepresentatives[(indexRep<<0x1) + 0x1] =   j;
                        indexRep++;
                    }
                }
            }
        }
    }
    
    //extra code for the "last" case (temp == product of all the primes of N)
    //(needed e.g. to take (0:1) correctly into account)
    temp = 0x1;
    for(i = 0x0; i < lenghPrimesCurrentNinUse; i++)
    {
        temp *= primesCurrentNinUse[i][0];
    }
    for(i = 0x0; i < currentNinUse; i += temp)
    {
        arrayOfRepresentatives[(indexRep<<0x1)      ] =   i;
        arrayOfRepresentatives[(indexRep<<0x1) + 0x1] = 0x1;
        indexRep++;
    }
    
//    //malloc for arrayOfVSMLowerEntries (which remains uninitialized)
    arrayOfVSMLowerEntries = (EF_sng *) malloc( currentNinUse * 0x2
                                                * sizeof(EF_sng)
                                              );
    
    if(NULL == arrayOfVSMLowerEntries)
    {
        free(arrayOfRepresentatives);
            
        free(arrayOfHelperData);
        
        free(arrayOfInversesModN);
        
        free(arrayForAdditionModN);
        
        sizeOfP1ZNZplusOne = 0x0;
        
        return(0x7);
    }

    //malloc for arrayOfCoprimesAndNegatives (which remains uninitialized)
    arrayOfCoprimesAndNegatives = (EF_sng *) malloc( currentNinUse * 0x4
                                                     * sizeof(EF_sng)
                                                   );
    
    if(NULL == arrayOfCoprimesAndNegatives)
    {
        free(arrayOfVSMLowerEntries);
            
        free(arrayOfRepresentatives);
            
        free(arrayOfHelperData);
        
        free(arrayOfInversesModN);
        
        free(arrayForAdditionModN);
        
        sizeOfP1ZNZplusOne = 0x0;
        
        return(0x8);
    }

    //malloc for arrayOfXmodN (which remains uninitialized)
    arrayOfXmodN = (EF_sng *) malloc(currentNinUse * sizeof(EF_sng));

    if(NULL == arrayOfXmodN)
    {
        free(arrayOfCoprimesAndNegatives);
            
        free(arrayOfVSMLowerEntries);
            
        free(arrayOfRepresentatives);
            
        free(arrayOfHelperData);
        
        free(arrayOfInversesModN);
            
        free(arrayForAdditionModN);
        
        sizeOfP1ZNZplusOne = 0x0;
        
        return(11);
    }
    
    return(0x0);
}

void BigONSpaceTablesDestruction(void)
{
    if(0x0 != sizeOfP1ZNZplusOne)
    {
        //free memory
        free(arrayOfXmodN);
            
        free(arrayOfCoprimesAndNegatives);
            
        free(arrayOfVSMLowerEntries);
            
        free(arrayOfRepresentatives);
            
        free(arrayOfHelperData);
        
        free(arrayOfInversesModN);
            
        free(arrayForAdditionModN);
        
        sizeOfP1ZNZplusOne = 0x0;
    }
    
    return;
}

inline EF_sng addModN(EF_sng a, EF_sng b)
{
//    if( (0x0 != sizeOfP1ZNZplusOne)
//        &&
//        (0x0 <= a)
//        &&
//        (0x0 <= b)
//      )
//    {
//        a = a%currentNinUse;
//        b = b%currentNinUse;

        return(arrayForAdditionModN[a + b]);
//    }
//    else
//    {
//        return(0x0);
//    }
}

inline EF_sng listOfSumsModN(EF_sng ** ptrptr_list)
{
//    if(0x0 != sizeOfP1ZNZplusOne)
//    {
        *ptrptr_list = arrayForAdditionModN;
        
        return(0x2 * currentNinUse + 0x1);
//    }
//    else
//    {
//        *ptrptr_list = (EF_sng *)NULL;
//        
//        return(0x0);
//    }
}


inline EF_sng inverseModN(EF_sng a)
{
//    if( (0x0 != sizeOfP1ZNZplusOne)
//        &&
//        (0x0 <= a)
//      )
//    {
//        a = a%currentNinUse;
        
        return(arrayOfInversesModN[a]);
//    }
//    else
//    {
//        return(0x0);
//    }
}

inline EF_sng listOfInversesModN(EF_sng ** ptrptr_list)
{
//    if(0x0 != sizeOfP1ZNZplusOne)
//    {
        *ptrptr_list = arrayOfInversesModN;
        
        return(currentNinUse);
//    }
//    else
//    {
//        *ptrptr_list = (EF_sng *)NULL;
//        
//        return(0x0);
//    }
}

inline EF_sng listOfRepresentatives(EF_sng ** ptrptr_list)
{
//    if(0x0 != sizeOfP1ZNZplusOne)
//    {
        *ptrptr_list = (EF_sng *)arrayOfRepresentatives;
//    }
//    else
//    {
//        *ptrptr_list = (EF_sng *)NULL;
//    }
    
    return(sizeOfP1ZNZplusOne);
}

inline EF_sng indexOfRepresentativeAndTheFactor(EF_sng u,
                                                EF_sng v,
                                                EF_sng * ptr_factor
                                               )
{
    EF_sng u_inverse;
    EF_sng v_inverse;
//    if( (0x0 != sizeOfP1ZNZplusOne)
//        &&
//        (0x0 <= u)
//        &&
//        (0x0 <= v)
//      )
//    {
//        u = u%currentNinUse;
//        v = v%currentNinUse;
        
        u_inverse = arrayOfInversesModN[u];
        v_inverse = arrayOfInversesModN[v];
        
        if( 0x0 != u_inverse)
        {
            //convention as in William Steins book, algorithm 8.28:
            //(u:v) == (s u_0: s v_0)
            // here u_0 == 1, v_0 == v u_inverse, and s == u
            *ptr_factor = u;
            
            //the index of the representative (u_0:v_0) is precisely 1 + v_0
            return( 0x1 + (v * u_inverse) % currentNinUse);
        }
        else if( 0x0 != v_inverse)
        {                                            
            //u_0 == u v_inverse, v_0 == 1, s == v
            *ptr_factor = v;
            
            //the index of the representative (u_0:v_0) is
            //precisely 1 + currentNinUse + u_0 / p_N,
            //as long as N is some power of a single prime
            //TODO: as soon as N is allowed to have more than one prime factor,
            //TODO: change this!
            return(0x1 + currentNinUse + ((u * v_inverse) % currentNinUse)/p_N);
        }
        //TODO: as soon as N is allowed to have more than one prime factor,
        //TODO: complete this!
        else
        {
            *ptr_factor = 0x0;
        
            return(0x0);
        }
        
//    }
//    else
//    {
//        *ptr_factor = 0x0;
//        
//        return(0x0);
//    }
}


//inline EF_sng indexOfRepresentative(EF_sng u, EF_sng v)
//{
//    EF_sng dummy;   
//    
//    return(indexOfRepresentativeAndTheFactor(u, v, &dummy));
//}


//
// Given an upper left matrix entry 0 <= e11 < N, and a determinant detModN
// with 0 < detModN < N (detModN being invertible mod N), then:
//  produce for each upper right entry e12, with 0 <= e12 < N,
//  a somewhat "canonical" pair of lower matrix entries e21 and e22
//  such that e11*e22 - e12*e21 == detModN  (mod N), i.e. one matrix
//
//              ( e11  e12 )
//              ( e21  e22 )
//
//  (with the predescribed upper left entry e11 and determinant)
//  for each possible (not all are) upper right entry e12
//  (the "invalid" e12 being those with gcd(e11, N, e12) > 1).
//
// Place these pairs e21, e22 in an array of size 2 * N, at indices
// 2*e12 and 2*e12 + 1 respectively ( initializing with 0, 0 those
//  entries which do not correspond to a valid e12)
// and return that array
//
// Each matrix as above is a "virtual" start matrix for the Basmaji "A"
// series (with prescribed e11, e12), the whole such series consisting
// exactly of the matrices of the form (with 0 <= k < N):
//
//         (    e11          e12      )
//         ( e21 - k*e11  e22 - k*e12 )
//
// (Obviously, there are exactly N different possible choices for a
//  "canonical virtual start matrix" in each such series.)
// We don't ever need to calculate these whole series directly.
// It suffices to know a set of "canonical virtual start matrices"
// on the one hand, and on the other hand, we only need the following:
//
// Given A, a 2x2 matrix mod N (!!) with determinant equal to detModN,
// and a11 and a12 fix (xModN and +-yModN in our application).
// Then we do know a priori that the lower two entries (a21, a22)
// are for some 0 <= k < N of the above form (e21-k*e11, e22-k*e12).
// (proof: exercise. hint: do first the cases N being a prime, then
//         a prime power. For N having different prime factors, use
//         in the last step "Chinese Remaindering", that the General
//         Linear matrix group mod N is just the direct product of the
//         the matrix groups mod the powers of the prime factors of N)
// Now the other thing we need to be able to do fast is to retrieve
// this k, which in an obvious sense is the "index" of A in the
// respective Basmaji series.
// See below for the function that retrieves k without calculating
// the whole series.

inline EF_sng listOfVirtualStartMatricesLowerEntries(EF_sng e11ModN,
                                                     EF_sng detModN,
                                                     EF_sng ** ptrptr_list)
{
    EF_sng e12ModN;     //upper right entry
    EF_sng e21ModN;     //lower left  entry
    EF_sng e22ModN;     //lower right entry
    EF_sng inverse;     //e11 or e12 need not be invertible, but may
    EF_sng temp;        //temporary variable
    
    
    
//    if( (0x0 != sizeOfP1ZNZplusOne) || (e11ModN<0) || (detModN<0))
//    {
//        e11ModN = e11ModN%currentNinUse;
//        detModN = detModN%currentNinUse;

        //first check for valid detModN
        // (detModN has to be invertible)
        if( 0x00 == inverseModN(detModN) )
        {
            *ptrptr_list = (EF_sng *)NULL;
            
            return(0x0);
        }
        
        //one easy special case next
        if( 0x00 != (inverse = inverseModN(e11ModN) ) )
        {
            //all possible e12ModN are valid
            
            //"canonical" choice of lower left entry
            //e21ModN = 0x0;
            
            //our choice just made forces:
            e22ModN = (detModN * inverse)%currentNinUse;
            //REMARK: We don't do this often, so using the O(N^2) tables
            //        won't be of so much help. On the other side, it's
            //        not bad to have this module independent
            
            for(e12ModN = 0x0; e12ModN < currentNinUse; e12ModN++)
            {
                arrayOfVSMLowerEntries[(e12ModN<<0x1)      ] = 0x0;//e21ModN
                arrayOfVSMLowerEntries[(e12ModN<<0x1) + 0x1] = e22ModN;
            }
        }
        else
        {
            for(e12ModN = 0x0; e12ModN < currentNinUse; e12ModN++)
            {
                //further easy special cases
                if( 0x00 != (inverse = inverseModN(e12ModN) ) )
                {
                    //"canonical" choice of lower right entry
                    //e22ModN = 0x0;
            
                    //this choice just made forces:
                    e21ModN = (detModN * (currentNinUse-inverse))%currentNinUse;
                    
                    arrayOfVSMLowerEntries[(e12ModN<<0x1)      ] = e21ModN;
                    arrayOfVSMLowerEntries[(e12ModN<<0x1) + 0x1] = 0x0;//e22ModN
                }
                //check whether gcd(e11ModN, e12ModN, N) > 1,
                //which is equivalent to (e11ModN, e12ModN) not having a
                //representative in P^1(Z/NZ)
                else if( 0x0 == indexOfRepresentativeNSquareSpace(e11ModN, e12ModN) )
//                else if( 0x0 == indexOfRepresentative(e11ModN, e12ModN) )
                {
                    //e12ModN "invalid", initialize entries with zero
                    arrayOfVSMLowerEntries[(e12ModN<<0x1)      ] = 0x0;
                    arrayOfVSMLowerEntries[(e12ModN<<0x1) + 0x1] = 0x0;
                }
                else
                {
                    //brute force
                    //(we'll be done soon, however, since we know that
                    // among the (N-1)^2 pairs there are N solutions, so on
                    // "average", we have a hit before the Nth pair searched)
                    for(e21ModN = 0x1; e21ModN < currentNinUse; e21ModN++)
                    {
                        if( 0x0 == indexOfRepresentativeNSquareSpace(e11ModN, e21ModN) )
//                        if( 0x0 == indexOfRepresentative(e11ModN, e21ModN) )
                        {
                            //no search necessary for this e21ModN
                            continue;
                        }
                        
                        temp = (detModN + e12ModN * e21ModN)%currentNinUse;
                                                     
                        for(e22ModN = 0x1; e22ModN < currentNinUse; e22ModN++)
                        {
                            if(0x0 == indexOfRepresentativeNSquareSpace(e12ModN, e22ModN))
//                            if(0x0 == indexOfRepresentative(e12ModN, e22ModN))
                            {
                                //no search necessary for this e22ModN
                                continue;
                            }
                        
                            if(0x0 == indexOfRepresentativeNSquareSpace(e21ModN, e22ModN))
//                            if(0x0 == indexOfRepresentative(e21ModN, e22ModN))
                            {
                                //no search necessary for this e22ModN
                                continue;
                            }
                        
                            if( temp == (e11ModN * e22ModN)%currentNinUse )
                            {
                                //target hit
                                arrayOfVSMLowerEntries[(e12ModN<<0x1)
                                                      ] = e21ModN;
                                arrayOfVSMLowerEntries[(e12ModN<<0x1) + 0x1
                                                      ] = e22ModN;
                                e22ModN = currentNinUse;//break inner loop
                                e21ModN = currentNinUse;//break outer loop
                            }
                        }//inner loop
                    }//outer loop
                }//else brute force
            }//for(e12ModN  0x0; e12ModN < currentNinUse; e12ModN++)
        }//e11ModN invertible?

        *ptrptr_list = arrayOfVSMLowerEntries;
        
        return(0x2 * currentNinUse);
//    }
//    else
//    {
//        *ptrptr_list = (EF_sng *)NULL;
//        
//        return(0x0);
//    }
}



//
// Given 0<= a, b, ka, kb < N with gcd(a,b,N) = 1 and such that we know
// ka = k*a mod N, kb = k*b mod N for some k,
// retrieve the unique k with 0 <= k < N with this property
//
inline EF_sng retrieveK(EF_sng a, EF_sng b, EF_sng ka, EF_sng kb)
{
    //no sanity checks for the time being
    
    EF_sng inverse;     //a or b need not be invertible, but may
    EF_sng i;           //counter variable
    EF_sng ia = a;      //test variable for a
    EF_sng ib = b;      //test variable for b
    
    
    if( (0x0 == ka) && (0x0 == kb) )
    {
        return(0x0);
    }
    else if( (a == ka) && (b == kb) )
    {
        return(0x1);
    }
    else if( 0x0 != (inverse = inverseModN(a)) )
    {
        return( (ka * inverse)%currentNinUse );
        //TODO: use multiplication table?!
    }
    else if( 0x0 != (inverse = inverseModN(b)) )
    {
        return( (kb * inverse)%currentNinUse );
        //TODO: use multiplication table?!
    }
    else
    {
        //TODO: create look-up-table of size O(N) in composite case
        
        //brute force for the time being
        for( i = 0x2; i < currentNinUse; i++ )
        {
            if( (ia += a) > currentNinUse ) ia -= currentNinUse;
            if( (ib += b) > currentNinUse ) ib -= currentNinUse;
            //now   ia == i * a (mod N)   and   ib == i * b (mod N)
            if( (ia == ka) && (ib == kb) )  break;
        }
        return(i);
        //incidentally, on invalid input, we do get
        // k == currentNinUse
        // as output, which also is invalid, and thus could be used as a test
    }
}


inline EF_sng negateModN(EF_sng a)
{
//    if( (0x0 != sizeOfP1ZNZplusOne)
//        &&
//        (0x0 <= a)
//      )
//    {
//        a = a%currentNinUse;
        
        //differs from "currentNinUse - a" if and only if 0x0 == a
        return( arrayForAdditionModN[ currentNinUse - a ] );
//        return( addModN( 0x0, currentNinUse - a ) );
//    }
//    else                                           
//    {
//        return(0x0);
//    }
}


inline EF_sng listOfCoprimesAndNegatives(EF_sng xN, EF_sng ** ptrptr_list)
{
    EF_sng yN;                   //yN checked for being coprime to gcd(N, xN)
    EF_sng i    = 0x0;           //counter variable

//    if(0x0 != sizeOfP1ZNZplusOne)
//    {
        for( yN = 0x0; yN < currentNinUse; yN++)
        {
            if( 0x0 != indexOfRepresentativeNSquareSpace(xN, yN) )
//            if( 0x0 != indexOfRepresentative(xN, yN) )
            {
                arrayOfCoprimesAndNegatives[i++] = yN;
                arrayOfCoprimesAndNegatives[i++] = negateModN(yN);
                arrayOfCoprimesAndNegatives[i++] = yN *
                                                   ((currentNinUse<<0x1) + 0x1)
                                                     + 0x1;
                arrayOfCoprimesAndNegatives[i++] = negateModN(yN) *
                                                   ((currentNinUse<<0x1) + 0x1)
                                                     + 0x2 + currentNinUse;
            }
        }
        
        *ptrptr_list = (EF_sng *)arrayOfCoprimesAndNegatives;
//    }
//    else
//    {
//        *ptrptr_list = (EF_sng *)NULL;
//    }
    
    return(i);
}


inline int8_t PrimeFactorizationCurrentN(EF_sng factors[][0x2])
{
    int8_t i;
    
    for( i = 0x0; i < lenghPrimesCurrentNinUse; i++)
    {
        factors[i][0x0] = primesCurrentNinUse[i][0x0];
        factors[i][0x1] = primesCurrentNinUse[i][0x1];
    }
    
    return lenghPrimesCurrentNinUse;
}


inline EF_sng listOfXmodN(EF_sng primesSubset, EF_sng ** ptrptr_list)
{
    EF_sng i = 0x0;
    EF_sng j = 0x0;
    EF_sng temp = 0x1;
    
    
    *ptrptr_list = (EF_sng *)arrayOfXmodN;
    
    if(0x0 == primesSubset)
    {
        for(i = 0x0; i < currentNinUse; i++)
        {
            if( 0x0 != arrayOfInversesModN[i] )
            {
                arrayOfXmodN[j++] = i;
            }
        }
    }
    else if( (0x1<<lenghPrimesCurrentNinUse) - 0x1 == primesSubset )
    {
//        for(i = 0x0; i < lenghPrimesCurrentNinUse; i++)
//        {
//            if( primesSubset & (0x1<<i) )
//            {
//                temp *= primesCurrentNinUse[i][0];
//            }
//        }
        for(i = 0x0; i < lenghPrimesCurrentNinUse; i++)
        {
            temp *= primesCurrentNinUse[i][0x0];
        }
        for(i = 0x0; i < currentNinUse; i += temp)
        {
            arrayOfXmodN[j++] = i;
        }
    }
    else if(0x1 == primesSubset)
    {
        //fake for "middle" cases: all at once
        for(i = 0x0; i < currentNinUse; i++)
        {
            arrayOfXmodN[i] = arrayOfInversesModN[i];
        }
        for(i = 0x0; i < lenghPrimesCurrentNinUse; i++)
        {
            temp *= primesCurrentNinUse[i][0x0];
        }
        for(i = 0x0; i < currentNinUse; i += temp)
        {
            arrayOfXmodN[i] = 0x1;  //dummy non-zero value
        }
        //now exactly the entries for the "middle" cases are zero
        for(i = 0x0; i < currentNinUse; i++)
        {
            if( 0x0 == arrayOfXmodN[i] )
            {
                arrayOfXmodN[j++] = i;
            }
        }
    }
    else
    {
        //fake for "middle" cases: zero length for the others
        ;
    }
    
    return(j);
}
