/* ************************************************************
   ** GroupDetermination.mg                                  **
   **                                                        **
   ** Computes the image of a modular Galois representation. **
   ** Currently only works in characteristic 2.              **
   **                                                        **
   ** Gabor Wiese                                            **
   ** version of 01/01/2008                                  **
   **                                                        **
   ************************************************************ */

import "Structure.mg" : ModF, WholeLevel;

// returns a tuple <S,F>, where S is the set of orders
// of Frobenius elements, and F is the field generated
// by the relevant Frobenius traces (i.e. those in L).
getOrders := function (mf, L)
  deg := 1;
  S := {1};
  M := MatrixAlgebra(GF(Characteristic(mf),FieldDegree(mf)),2);

  for j := 1 to #L do
      x := Coefficient(mf,L[j]);
      Include (~S, Order(M![0,1,1,x]));
      if x ne 0 then
        r := Degree(MinimalPolynomial(x));
        deg := Lcm(deg,r);
      end if;
  end for;

  return < S,GF(2,deg) >;
end function;


intrinsic Group ( mf :: Rec : bound_factor := 2 ) -> MonStgElt, MonStgElt
{Given an eigenform in characteristic 2, returns two strings L,U.
L contains the name of a lower bound for
the group generated by the associated Galois representation.
Similarly, U gives an upper bound.
The maximum field degree allowed is specified by the option maxDegree.
All admissible primes up to bound_factor * HeckeBound are taken
into account.}

  if (mf`Characteristic ne 2) or ((assigned mf`Character) and not (IsTrivial(mf`Character)))then
    vprint Weight1: "Group determination only possible in characteristic 2 for the trivial character.";
    return "?","?";
  end if;

  b := Ceiling(HeckeBound(mf) * bound_factor);
  L := [i : i in [2..b] | IsPrime(i) and 
              (((mf`Characteristic * mf`Level) mod i) ne 0)];
  T := getOrders(mf, L);

  S := T[1];
  F := T[2];
  q := #F;

  if IsEmpty(S) then
    st := "?";
  else

    divqp := 1;
    divqm := 1;

    for s in S do
      if s ne 1 then
        if ((q-1) mod s) eq 0 then
          divqm := Lcm(divqm, s);
        elif ((q+1) mod s) eq 0 then
          divqp := Lcm(divqp, s);
        end if;
      end if;
    end for;

    if (divqp eq 1) and (divqm eq 1) then
      st := "E";
    elif (divqm eq 1) then
      st := "C_"*Sprint(divqp);
    elif (divqp eq 1) then
      st := "C_"*Sprint(divqm);
    elif (divqp ne 1) and (divqm ne 1) then

      r := Valuation(q,2);
      D := Divisors(r);
      s := 1; found := false;
      while (s le #D) and (not found) do
        if ((2^(2*(D[s]))-1) mod (divqp * divqm)) eq 0 then
          found := true;
        else
          s := s + 1;
        end if;
      end while;

      st := "SL(2,"*Sprint(2^(D[s]))*")";

    end if;
  end if;

  return st, "SL(2,"*Sprint(2^mf`FieldDegree)*")";
end intrinsic;

