/*
********************************************
* Creation.mg                              *
*                                          *
* Creation functions                       *
* for commutative p-adic algebras.         *
*                                          *
* Gabor Wiese, version of 24/01/2014       *
********************************************
*/

pAdicAlgebraFormat := recformat <
  p		: RngIntElt, 	// the p of Z_p
  R		: Any,		// the p-adic ring of given precision
  dim		: RngIntElt,	// the dimension of the residual algebra
  deg		: RngIntElt,	// degree of the matrix algebra (i.e. number of rows)
  Amod		: Any,		// the full matrix algebra over the residue field
  Aadic		: Any,		// the full matrix algebra over R
  basis		: SeqEnum,	// matrices over Z s.t. the reductions generate an algebra of dimension dim

  adic_coords   : Any,		// coordinate function wrt basis
  mod_coords    : Any,		// coordinate function wrt basis mod p

  ipmod		: Any,		// a complete set of orthogonal idempotents for Amod (as an algebra)
  ipadic	: Any		// the lifts of ipmod to idempotents in Aadic
>;


// this is the creation for matrix algebras over Z
intrinsic pAdicAlgebra (L :: SeqEnum, p :: RngIntElt : prec := 30) -> Any
{Input: L a list of commuting matrices with integer entries; p a prime.
Output: pAdicAlgebraFormat record structure for the Z_p-algebra generated by the matrices in L. Basis and idempotents are automatically computed.}
  require L ne [] : "First argument must not be the empty list.";
  require Nrows(L[1]) eq Ncols(L[1]) : "Matrices must be square.";
  require IsPrime(p) : "Second argument must be a prime number.";
  require BaseRing(L[1]) eq Integers() : "The base ring of matrices must be the integers.";

  AF := rec< pAdicAlgebraFormat |  >;

  AF`deg := Nrows(L[1]);
  AF`p := p;
  AF`R := pAdicRing(p,prec);

  // compute basis using Nakayama's lemma
  AF`basis := pAdicBasis(L,p);
  AF`dim := #AF`basis;

  AF`Amod := MatrixAlgebra(GF(p),AF`deg);
  AF`Aadic := MatrixAlgebra(AF`R,AF`deg);

  _compute_mod_coords(~AF);

  return AF;
end intrinsic;

// this is the creation for matrix algebras over Z_p (needed in the decomposition!)
intrinsic pAdicAlgebra (L :: SeqEnum) -> Any
{Input: L a list of commuting matrices with entries in a p-adic ring.
Output: pAdicAlgebraFormat record structure for the Z_p-algebra generated by the matrices in L. Basis and idempotents are automatically computed.}
  require L ne [] : "First argument must not be the empty list.";
  require Nrows(L[1]) eq Ncols(L[1]) : "Matrices must be square.";
  require Type(BaseRing(L[1])) eq RngPad : "The base ring of matrices must be a p-adic ring.";

  AF := rec< pAdicAlgebraFormat |  >;

  AF`R := BaseRing(L[1]);
  AF`deg := Nrows(L[1]);
  AF`p := Prime(AF`R);

  // compute basis using Nakayama's lemma
  AF`basis := pAdicBasis(L);

  AF`dim := #AF`basis;
  AF`Amod := MatrixAlgebra(GF(AF`p),AF`deg);
  AF`Aadic := MatrixAlgebra(AF`R,AF`deg);

  _compute_mod_coords(~AF);

  return AF;
end intrinsic;



/* create a matrix algebras from the pAdicAlgebra structure */

intrinsic pAdicMatrixAlgebra ( AF :: Rec ) -> AlgMat, Map
{Input: A p-adic algebra in pAdicAlgebraFormat.
Output: The corresponding matrix algebra.}
  A,B := sub<AF`Aadic | [AF`Aadic!b : b in AF`basis]>;
  return A,B;
end intrinsic;

intrinsic modpMatrixAlgebra ( AF :: Rec ) -> AlgMat, Map
{Input: A p-adic algebra in pAdicAlgebraFormat.
Output: The corresponding matrix algebra over the residue field.}
  A,B := sub<AF`Amod | [AF`Amod!b : b in AF`basis]>;
  return A,B;
end intrinsic;



/* precision handling of p-adic matrix algebras */

intrinsic ChangePrecision ( A :: AlgMat, prec :: RngIntElt ) -> AlgMat
{Input: Matrix algebra over p-adic ring; an integer prec.
Output: Same matrix algebra over p-adic ring with precision prec..}
  R := CoefficientRing(A);
  p := Prime(R);
  d := Degree(A);

  An := MatrixAlgebra(pAdicRing(p,prec),d);
  newGen := [An!g : g in Generators(A)];

  return MatrixAlgebra(newGen);
end intrinsic;

intrinsic ChangePrecision ( A :: AlgMatElt, prec :: RngIntElt ) -> AlgMat
{Input: Matrix in matrix algebra over p-adic ring; an integer prec.
Output: Same matrix in matrix algebra over p-adic ring with precision prec..}

  R := CoefficientRing(A);
  p := Prime(R);
  d := Nrows(A);

  An := MatrixAlgebra(pAdicRing(p,prec),d);

  return An!A;
end intrinsic;

