/*
********************************************
* AffineAlgebras.mg                        *
*                                          *
* Create an affine algebra from a          *
* commutative matrix algebra or a          *
* commutative associative algebra.         *
*                                          *
* Gabor Wiese, version of 31/12/2007       *
********************************************
*/


forward HomogeneousBasis;
HomogeneousBasis := function (R, n)
  if n eq 1 then 
    return [[R.i : i in [1..Rank(R)]]];
  else
    H := HomogeneousBasis(R,n-1);
    h := H[#H];
    l := [];
    for g in h do 
      for i in [1..Rank(R)] do
        s := g*R.i;
        if not (s in l) then
          Append(~l,s);
        end if;
      end for;
    end for;
    return Append(H,l);
  end if;
end function;


intrinsic AffineAlgebra (A :: AlgMat : try_minimal := true) -> RngMPolRes
{Turn the local algebra A into an affine algebra over the residue field.}
  return AffineAlgebra(AffineAlgebraTup(A : try_minimal := try_minimal));
end intrinsic;

intrinsic AffineAlgebra (A :: AlgAss : try_minimal := true) -> RngMPolRes
{Turn the local algebra A into an affine algebra over the residue field.}
  return AffineAlgebra(AffineAlgebraTup(A : try_minimal := try_minimal));
end intrinsic;

intrinsic AffineAlgebraTup (A :: AlgMat : try_minimal := true ) -> Tup
{Given a local algebra A, return a tuple <k,e,n,R>, consisting of the
residue field k of A, the embedding dimension e, the nilpotency order n
and relations R.}
  return AffineAlgebraTup(Algebra(A) : try_minimal := try_minimal);
end intrinsic;

intrinsic AffineAlgebraTup (A :: AlgAss : try_minimal := true) -> Tup
{Given a local algebra A, return a tuple <k,e,n,R>, consisting of the
residue field k of A, the embedding dimension e, the nilpotency order n
and relations R.}

  // determine the maximal ideal
  max := MaximalIdeals(A);
  error if #max ne 1, "Error. The algebra is not local!"; 
  m := max[1];

  // base change the algebra to its residue field and
  // take one conjugate factor.
  if Dimension(A) - Dimension(m) ne 1 then
    A := Algebra(AlgebraDecompositionUpToConjugation(MatrixAlgebra(A))[1]);
    max := MaximalIdeals(A);
    error if #max ne 1, "Error. The algebra is not local!"; 
    m := max[1];
  end if;


  // determine base field
  k := CoefficientRing(A);

  // determine the embedding dimension e
  e := Dimension(m) - Dimension(m^2);

  if e eq 0 then
    return <k,0,0,<>>;
  end if;

  // determine the nilpotency order n
  // i.e. the maximum st m^n ne 0
  n := 0; mi := m;
  while Dimension(mi) ne 0 do
    mi := mi*m;
    n := n+1;
  end while;

  // determine generators of m

  qu,phi := quo<m|m^2>;              
  psi := Inverse(phi);               
  gen := [psi(ba) : ba in Basis(qu)];

  // identify the algebra with a vector space V
  V := VectorSpace(k,Degree(Vector(ElementToSequence(gen[1]))));

  // get the relations
  L := []; // will contain a basis of m
  L1 := [];// will contain the elements of V corresponding to L 
  M := <>; // will contain relations in a form to be stored
  M1 := [];// will contain relations as polynomials
  R := PolynomialRing(k,e);
  S := PolynomialRing(A,e);
  hp := HomogeneousBasis(R,n+1);

  if try_minimal then
    M1 := hp[n+1];
  end if;

  for d := 1 to n do  // d stands for degree
    for i := 1 to #hp[d] do
      f := hp[d][i];
      if #L1 ne 0 then  
        W := VectorSpaceWithBasis(L1);
      else
        W := sub<V|>;
      end if;
      v := V!Vector(ElementToSequence(m!Evaluate(S!f,gen)));

      if not (v in W) then
        Append(~L1,v);
        Append(~L,<d,i>);
      else
        co := Coordinates(W,v);

        r := R!0;
        for j := 1 to #co do
          r := r+ co[j]*hp[L[j][1]][L[j][2]];
        end for;
        rel := f-r;

        if not try_minimal then
          Append(~M,<d,i,co>);
        else
          if not (rel in ideal<R | M1>) then
            Append(~M,<d,i,co>);
            Append(~M1,rel);
          end if;
        end if;
      end if;
    end for;
  end for;

  return <k,e,n,M>;
end intrinsic;


intrinsic AffineAlgebra (form :: Rec) -> RngMPolRes
{Given a modular form record, return the corresponding Hecke
algebra as an affine algebra.}
  return AffineAlgebra (<GF(form`Characteristic,form`AlgebraFieldDegree),
                 form`EmbeddingDimension,form`NilpotencyOrder, form`Relations>);
end intrinsic;

intrinsic AffineAlgebra (A :: Tup) -> RngMPolRes
{Turns a tuple <k,e,n,R>, consisting of a field k, two integers
e,n (the embedding dimension and the nilpotency order) and relations R, 
into an affine algebra.}
  k := A[1];
  e := A[2];
  n := A[3];
  M := A[4];

  if e eq 0 then
    return k;
  end if;

  R := PolynomialRing(k,e);
  hp := HomogeneousBasis(R,n+1);
  W := [<c[1],c[2]> : c in M];
  L := [];
  for d := 1 to n do  // d stands for degree
    for i := 1 to #hp[d] do
      if not (<d,i> in W) then
        Append(~L,<d,i>);
      end if;
    end for;
  end for;

  idealgen := [];
  for m in M do
    co := m[3];
    r := R!0;
    for i := 1 to #co do
      r := r+ co[i]*hp[L[i][1],L[i][2]];
    end for;
    Append(~idealgen,hp[m[1]][m[2]]-r);
  end for;
  idealgen := idealgen cat hp[n+1];
  return quo<R|idealgen>;
end intrinsic;


