/*
********************************************
* Idempotents.mg                           *
*                                          *
* Computes idempotents of commutative      *
* finite dimensional algebras over fields. *
*                                          *
* Gabor Wiese, version of 19/12/2007       *
********************************************
*/

BinaryExpansion := function (n)
  L := [];
  while n gt 0 do
    Append(~L,n mod 2);
    n := n div 2;
  end while;
  return L;
end function;

intrinsic AllIdempotents (L :: SeqEnum ) -> SeqEnum
{For a list L of orthogonal idempotents (not 0,1),
this function returns all idempotents that can be generated from L (including 0,1)}
  output := [];
  for i := 0 to 2^(#L)-1 do
    b := BinaryExpansion(i);
    e := Parent(L[1])!0;
    for j := 1 to #b do
      if b[j] eq 1 then
        e := e + L[j];
      end if;
    end for;
    Append(~output,e);
  end for;

  return output;
end intrinsic;

intrinsic AddUp (L :: SeqEnum) -> Any
{This function adds the elements of the sequence L.}
  sum := 0;
  for e in L do
    sum := sum + e;
  end for;
  return sum;
end intrinsic;

IdempotentsOneMatrix := function (M)
  f := MinimalPolynomial(M);
  
  // factor the minimal polynomial into prime powers
  // sort them such that x^n is the last factor, if it occurs
  fac := Factorisation(f);
  polys := []; store_x := Parent(f)!1;
  for g in fac do
    if Evaluate(g[1],0) ne 0 then
      Append(~polys,g[1]^(g[2]));
    else
      store_x := g[1]^(g[2]);
    end if;
  end for;
  if store_x ne Parent(f)!1 then 
    Append(~polys,store_x); 
  end if;

  output := [];
  for i := 1 to #polys-1 do
    g := f div polys[i];
    M1 := Evaluate(g,M);
    h := MinimalPolynomial(M1);
    seq := Coefficients(h);
    while seq[1] eq 0 do
      seq := seq[2..#seq];
    end while;
    h := 1 - Polynomial(seq)/seq[1];
    Append(~output, Evaluate(h,M1));
  end for;
  Append(~output,Parent(M)!1-Parent(M)!(AddUp(output)));

  return output;
end function;

intrinsic IsIdempotent (e :: Any) -> BoolElt
{Tests whether e is an idempotent.}
  return e^2 eq e;
end intrinsic;

intrinsic AreOrthogonalIdempotents (L :: Any) -> BoolElt
{Tests whether the elements of a given sequence of idempotents L
form a set of orthogonal idempotents.}
  output := true;
  for e in L do
    output := output and IsIdempotent(e);
    for f in L do
      output := output and ((e*f eq 0) or (e eq f));
    end for;
  end for;
  return output;
end intrinsic;

intrinsic GetOrthogonalIdempotents (L :: SeqEnum) -> SeqEnum
{For a list L of idempotents, this function
returns an orthogonal set of idempotents spanning L.}

  if #L eq 0 then
    print "Error. L is empty.";
    return L;
  end if;

  A := Parent(L[1]);
  output := [A!1];

  while #L ne 0 do
    a := L[1];
    L := L[2..#L];
    if not (a in [A!0,A!1]) then
      R := output;
      output := [];
      for e in R do
        if e*a eq 0 then
          if e ne A!0 then Append(~output, e); end if;
        else
          Append(~output, e*a);
          if e ne e*a then Append(~output, e-e*a); end if;
        end if;
      end for;
    end if;
  end while;

  return output;
end intrinsic;

intrinsic CompleteOrthogonalIdempotents ( L :: SeqEnum ) -> SeqEnum
{For a list L of commuting algebra elements over a field this function
returns a compelete set of orthogonal idempotents for all the elements in L.
Note that if L forms a basis of an algebra, then the output is a complete
set of orthogonal idempotents of this algebra. If, however, L only
forms a generating set, then it is not guaranteed that the output
is a complete set.}

  K := CoefficientRing(L[1]);
  A := MatrixAlgebra(K,NumberOfRows(L[1]));
  L := [A!l : l in L];

  output := [A!1];
  for l in L do
    R := IdempotentsOneMatrix(l);
    S := output;
    output := [A!1];
    for r in R do
      for s in S do
        Append(~output,r*s);
      end for;
    end for;
    // remove redundancies
    output := GetOrthogonalIdempotents(output);
  end for;

  return output;
end intrinsic;

intrinsic CompleteOrthogonalIdempotents ( A :: AlgMat ) -> SeqEnum
{For a matrix algebra A over a field this function
returns a compelete set of orthogonal idempotents for A.}
  return CompleteOrthogonalIdempotents(Basis(A));
end intrinsic;


// *************** idempotents from given decomposition **************+

intrinsic Idempotents ( D :: Tup ) -> SeqEnum, Tup
{Given a decomposition D, calculate a list of the 
corresponding idempotents as matrices.
The function also returns base change matrices describing the
algebra decomposition in terms of block matrices.}
  local i,L,n,F,j,z,C1,C,alg,V,W;

  n := NumberOfColumns (D[1][1]);
  F := BaseRing(D[1][1]);
  alg := MatrixAlgebra (F,n);
  C := []; 
  L := [];

  z := 1;
  for i := 1 to #D do
    L[i] := alg!0;
    for j := 1 to NumberOfRows(D[i][1]) do
      C[z] := D[i][1][j];
      L[i] := L[i] + MatrixUnit (alg,z,z);
      z := z + 1;
    end for;
  end for;

  V := VectorSpaceWithBasis(C);
  W := Generic(V);
  C := alg!(ExtendBasis(V,W));

  C1 := C^(-1);
  for i := 1 to #D do
    L[i] := C1*L[i]*C;
  end for;

  return L,<C,C1>;
end intrinsic;

