/*****************************************
 * FB_Rankin-Cohen.m                     *
 * part of                               *
 * FastBases Magma Package               *
 * Author: Gabor Wiese                   *
 * 2 June 2021                           *
 *****************************************/


intrinsic RC (f,g,k,l,n) -> Any
{Compute the n-th Rankin-Cohen bracket of the power series f and g, where k is the weight of f and l is the weight of g.}
  R := Parent(f);
  h := R!0;
  for j := 0 to n do
    h := h + (-1)^j * FB_binomial(n+k-1,j) * FB_binomial(l+n-1,n-j) * qDerivative(f,n-j) * qDerivative(g,j);
  end for;
  return h;
end intrinsic;


intrinsic FB_RC_even (k,n,bound) -> Any
{Computes the n-th Rankin-Cohen bracket [G_*, theta]_n such that the output has EVEN weight k and level 4.}

  assert ((Integers()!(k-1/2)) mod 2) eq 0;

  // compute the weight k1 of the Eisenstein series E so that [E,theta]_n has weight k
  k1 := Integers()!(k-1/2-2*n);

  G4 := FB_eisenstein(k1,bound);
  T := FB_theta(bound);

  return RC(G4,T,k1,1/2,n);
end intrinsic;


intrinsic FB_RC_odd (k,n,index,bound) -> Any
{Computes the n-th Rankin-Cohen bracket [G_*, theta]_n such that the output has ODD weight k and level 4; index must be chosen to be 0 or 1, where 0 corresponds to the Eisenstein series with character conductors 1,-4 and 1 to the one with conductors -4,1.}

  assert ((Integers()!(k-1/2) mod 2)) eq 1;

  // compute the weight k1 of the Eisenstein series E so that [E,theta]_n has weight k
  k1 := Integers()!(k-1/2-2*n);

  if index eq 0 then
    G4 := FB_eisenstein(k1,1,-4,bound);
  else
    G4 := FB_eisenstein(k1,-4,1,bound);
  end if;
  T := FB_theta(bound);
  return RC(G4,T,k1,1/2,n);
end intrinsic;


intrinsic FB_RC_forms_plus (k,number,bound) -> Any
{Computes the first number of modular forms in the plus-space of level 4 and weight k that can be obtained from RC brackets of Eisenstein series with the standard theta function. We choose the same ordering as in the pari/gp code. This requires k-1/2 to be even.}
  assert IsEven(Floor(k-1/2));

  bas := [];
  for a := 0 to number-1 do
    Append(~bas,FB_RC_even(k,a,bound));
  end for;    

  return bas;  
end intrinsic;

intrinsic FB_RC_forms_full (k,number,bound) -> Any
{Computes the first number of modular forms in level 4 and weight k that can be obtained from RC brackets of Eisenstein series with the standard theta function. We choose the same ordering as in the pari/gp code. This requires k-1/2 to be odd.}
  assert IsOdd(Floor(k-1/2));

  bas := [];
  for a := 0 to number-1 do
    Append(~bas,FB_RC_odd(k,a div 2,a mod 2,bound));
  end for;  

  return bas;  
end intrinsic;

intrinsic FB_RC_basis_full (k, bound) -> Any
{Computes the RC-basis of the full space of weight k and level Gamma_0(4) up to bound. This requires k-1/2 to be odd. Currently, it is not checked that it's a basis.}
  l := Floor(k-1/2);     // this is what is called k in Kohnen's paper: k = l+1/2
  assert ((l+1/2) eq k); // only defined for k = l + 1/2 with l odd.
  assert IsOdd(l);

  di := 1 + Floor(l/2);  // compute the dimension of the target space


  return FB_RC_forms_full(k,di,bound);
end intrinsic;


intrinsic FB_RC_basis_plus (k, bound) -> Any
{Computes the RC-basis of the plus space of weight k and level Gamma_0(4) up to bound. This requires k-1/2 to be even. Currently, it is not checked that it's a basis.}
  l := Floor(k-1/2);      // this is what is called k in Kohnen's paper: k = l+1/2
  assert ((l+1/2) eq k);  // only defined for k = l + 1/2 with l even.  
  assert IsEven(l);
  assert l ge 4;          // it doesn't work with l eq 2 because we'd run into the weight 2 Eisenstein series, but cannot use F2 as it's not in the plus space
  
  if l ge 6 then
    di := DimensionCuspFormsGamma0(1,l) + DimensionCuspFormsGamma0(1,l-2) + 2; // compute the dimension of the target space
  else // l eq 2 or 4
    di := 1;
  end if;
  return FB_RC_forms_plus(k,di,bound);
end intrinsic;



intrinsic FB_RC_lin_comb (k,coefs,bound) -> Any
{coefs is a vector of numbers (reals, in a number field, etc.). Computes the linear combination with coefficients coefs of the modular forms in level 4 and weight k that can be obtained from RC brackets of Eisenstein series with the standard theta function. We choose the same ordering as in the pari/gp code.}
  K := Parent(coefs[1]);
  R<q> := PowerSeriesRing(K,bound);
  f := R!0;

  if ((Integers()!(k-1/2) mod 2)) eq 0 then
    // even case
    for a := 1 to #coefs do
      f := f + coefs[a] * R!FB_RC_even(k,a,bound);
    end for;    
  else
    // odd case
    for a := 1 to #coefs do
      f := f + coefs[a] * R!FB_RC_odd(k,(a+1) div 2,(a+1) mod 2,bound);
    end for;  
  end if;  

  return f;  
end intrinsic;

