/*****************************************
 * FB_standard_forms.m                   *
 * part of                               *
 * FastBases Magma Package               *
 * Author: Gabor Wiese                   *
 * 5 December 2019                       *
 *****************************************/

intrinsic FB_theta (bound) -> Any
{Creates the standard theta series of weight 1/2 and level 4 as a power series up to bound.}
  R<q> := PowerSeriesRing(Rationals(),bound);
  m := Ceiling(Sqrt(bound));
  T := R!1;
  for n := 1 to m do
    T := T+2*q^(n^2);
  end for;
  return T;
end intrinsic;

intrinsic FB_eisenstein (k,chi1,chi2,bound) -> Any
{Compute the standard weight k Eisenstein series associated with the characters chi1, chi2 (conductors of Kronecker characters only at the moment) up to bound.}
  eps1 := KroneckerCharacter(chi1);
  eps2 := KroneckerCharacter(chi2);

  L:=[];
  for a := 1 to bound do
    D := Divisors(a);
    r := 0;
    for d in D do r := r + eps1(d) * eps2(a div d)* d^(k-1); end for;
    Append(~L,r);
  end for;

  if chi2 ne 1 then
    L1:= [0] cat L;
  else
    G := DirichletGroup(Conductor(eps1),Rationals()); 
    L1:= [-BernoulliNumber(k,G!eps1)/(2*k)] cat L;
  end if;    
  R<q> := PowerSeriesRing(Rationals(),bound);
  return R!L1;
end intrinsic;


intrinsic FB_eisenstein (k,bound) -> Any
{Compute the standard weight k Eisenstein series degenerated to level 4 up to bound.}
  L:=[];
  for a := 1 to bound do
    if a mod 4 eq 0 then
      Append(~L,DivisorSigma(k-1,(a div 4)));
    else
      Append(~L,0);
    end if;
  end for;
  L1:= [-BernoulliNumber(k)/(2*k)] cat L;
  R<q> := PowerSeriesRing(Rationals(),bound);
  return R!L1;
end intrinsic;

intrinsic FB_F2 (bound) -> Any
{Compute the standard weight 2 Eisenstein series in level 4 up to bound.}
  L:=[0];
  for a := 1 to bound do
    if (a mod 2) eq 1 then
      Append(~L,DivisorSigma(1,a));
    else
      Append(~L,0);
    end if;
  end for;
  R<q> := PowerSeriesRing(Rationals(),bound);
  return R!L;
end intrinsic;


// Cohen's Eisenstein series in the plus-space
intrinsic FB_H5_2 (bound) -> Any
{Compute Cohen's Eisenstein series H_5/2 up to bound. Formula from [Coh75].}
  T := FB_theta(bound);
  return T^5 - 20*T*FB_F2(bound);
end intrinsic;

intrinsic FB_H7_2 (bound) -> Any
{Compute Cohen's Eisenstein series H_7/2 up to bound. Formula from [Coh75].}
  T := FB_theta(bound);
  F2 := FB_F2(bound);
  return T^7 - 14*T^3*F2;
end intrinsic;

intrinsic FB_H11_2 (bound) -> Any
{Compute Cohen's Eisenstein series H_11/2 up to bound. Formula from [Coh75].}
  T := FB_theta(bound);
  F2 := FB_F2(bound);
  return T^11 - 22*T^7*F2 + 88*T^3*F2^2;
end intrinsic;

