/* ************************************************************
   **                                                        **
   ** Dihedral.mg                                            **
   **                                                        **
   ** Handling of dihedral forms.                            **
   **                                                        **
   ** Gabor Wiese                                            **
   ** version of 01/01/08                                    **
   **                                                        **
   ************************************************************ */

import "Structure.mg" : ModularFormFormat, AlgebraData;

intrinsic GetLegendre (N :: RngIntElt, K :: FldFin ) -> GrpDrchElt
{For an odd integer N this function returns the element
of the DirichletGroup(Abs(N),K) (with K a finite field of char ne 2) which
corresponds to the Legendre symbol p |-> (+-N/p).
The sign in front of N is chosen so that the number is congruent to 1 mod 4. }
  G := DirichletGroup(Abs(N),K);

  if Characteristic(K) eq 2 then return G!1; end if;

  // choose N such that the quadratic field K 
  // is unramified at 2
  if (N mod 4) eq 3 then
    N := -N;
  elif ((N mod 4) eq 0) or ((N mod 4) eq 2) then
    print "N even is not supported.";
    return G!1;
  end if;

  E := [ e : e in Elements(G) | (Order(e) eq 2)];

  p := 3;
  while #E gt 1 do
    if (N mod p) ne 0 then
      E1 := [];

      for e in E do
        if K!(LegendreSymbol(N,p)) eq e(p) then
          Append(~E1,e);
        end if;
      end for;

      E := E1;
    end if;
    p := NextPrime(p);
  end while;

  return E[1];
end intrinsic;

// modified Legendre symbol
LS := function(a,b)
  if b eq 2 then 
     if IsSplit(2,MaximalOrder(QuadraticField(a)))
       then return 1; 
       else return -1;
     end if;
  else 
    return LegendreSymbol(a,b);
  end if;
end function;


intrinsic DihedralFieldDegree (n :: Any, p :: Any) -> Any
{For an integer n>0 and a prime p this function returns the
smallest d such that the traces of D_n as a
subgroup of GL_2(F_p^bar) all lie in F_(p^d).}
  // compute the smallest power p^r such that p^r+1 or p^r-1 is divisible by n
  i := 1;
  while ((((p^i) +1) mod n) ne 0) and ((((p^i) -1) mod n) ne 0) do
    i := i + 1;
  end while;

  return i;
end intrinsic;


intrinsic DihedralForms (N :: RngIntElt : 
     bound := 100,              
     ListOfPrimes := [], 
     completely_split := true,
     odd_only := true,
     quad_disc := 0,
     all_conjugacy_classes := true   
   ) -> Rec
{Gets all modular forms in level N over a finite field of characteristic p that
come from dihedral representations which arise from the quadratic
field K=Q(sqrt(+-quad_disc)) by induction of an unramified character of K.
If quad_disc is 0, then N is used instead.
The sign in front of quad_disc is chosen so that the number is congruent to 1
mod 4.  If the option completely_split is set, only those
representations are returned which are completely split at p.  If the
option ListOfPrimes is set, only those primes are considered as
characteristic.  If it is the empty set, all primes up to the bound
are taken into consideration.
If if the odd_only is true, only odd representations will be returned.
If all_conjugacy_classes is true, all conjugacy classes of the character
of the quadratic field are used, otherwise, only one is taken.}

  if ListOfPrimes eq [] then
    ListOfPrimes := [p : p in [1..bound] | IsPrime(p)];
  end if;

  level := N;
  if quad_disc ne 0 then
    N := quad_disc;
  end if;

  // choose N such that the quadratic field K 
  // is unramified at 2
  if (N mod 4) eq 3 then
    N := -N;
  elif ((N mod 4) eq 0) or ((N mod 4) eq 2) then
    print "N even is not yet programmed.";
    return <>;
  end if;

  K := NumberField(Polynomial([-N,0,1]));
  O := MaximalOrder(K);
  CL,phi := ClassGroup(O);
  psi := Inverse(phi);

  output := [];

  for g in Subgroups(CL) do
    H,alpha := quo<CL | g`subgroup>;
    h := #H;
    if (h ne 1) and IsCyclic(H) then
      for p in ListOfPrimes do
        if IsPrime(p) and 
           ((N mod p) ne 0) and
           ((h mod p) ne 0) and 
           ((not completely_split) or
           (IsSplit(p,O) and
           (Order(alpha(psi(Factorisation(p*O)[1][1]))) eq 1))) then

          d := DihedralFieldDegree(h,p);
          eps := GetLegendre(N,GF(p));

          // possibly make modulus bigger
          eps := DirichletGroup(level, BaseRing(eps))!eps; 

          // run through all Gal(Fbar/F_p)-conjugacy classes 
          // of h-th roots of unity; but never take both zeta and its inverse
          factors := Factorisation(PolynomialRing(GF(p))!CyclotomicPolynomial(h));
          if not all_conjugacy_classes then
            factors := [factors[1]];
          end if;
          SetOfZetas := []; SetOfFactors := [];
          for fac in factors do
            zeta := -Evaluate(Factorisation(PolynomialRing(GF(p,Degree(fac[1])))!(fac[1]))[1][1],0);
            if not (MinimalPolynomial(zeta^(-1)) in SetOfFactors) then
              Append(~SetOfZetas,zeta);
              Append(~SetOfFactors,fac[1]);
            end if;
          end for;

          for zeta in SetOfZetas do
            beta := map < Integers() -> PolynomialRing(GF(p)) | 
                  l :-> MinimalPolynomial( (GF(p,d)!((LS(N,l) + 1) div 2)) * 
                 (zeta^(ElementToSequence(alpha(psi(Factorisation(l*O)[1][1])))[1] ) 
                    + 
                  zeta^(- ElementToSequence(alpha(psi(Factorisation(l*O)[1][1])))[1]) )) >; 

            form := rec< ModularFormFormat |
              Character := eps,
              Weight := p,
              CoefficientFunction := beta,
              ImageName := "D_\{"*Sprint(h)*"\}"
            >;

            if (not odd_only) or (Evaluate(eps,-1) eq -1) then
              Append(~output, form);
            end if;

          end for;
        end if;
      end for;
    end if;
  end for;

  return output;
end intrinsic;

