/* ************************************************************
   **                                                        **
   ** HeckeAlgebra.mg                                        **
   **                                                        **
   ** Computation of Hecke Algebras.                         **
   **                                                        **
   ** Gabor Wiese                                            **
   ** version of 01/01/08                                    **
   **                                                        **
   ************************************************************ */

import "Structure.mg" : ModularFormFormat, AlgebraData;

// *********** Hecke bound **************************

intrinsic HeckeBound ( N :: RngIntElt, k :: RngIntElt ) -> RngIntElt
{Computes the Hecke bound for level N and weight k.}
  local B,L;

  B := k * N  / 12; 
  L := Factorization (N);

  for i := 1 to #L do
    B := B * (1 + (1/L[i][1])); 
  end for;

  return Ceiling(B);  
end intrinsic;

intrinsic HeckeBound ( eps :: GrpDrchElt, k :: RngIntElt ) -> RngIntElt
{Computes the Hecke bound for the character eps in weight k.}
  return HeckeBound (Modulus(eps),k);
end intrinsic;


// -------------- Hecke algebra computation ------------------------

// Removes the first occurance of the entry p from the list,
// and then includes p at the n-th place in the list.
// If the list is too short, just append p at the end.
function PutAt (list, p, n)
  i := Index(list,p);
  if i ne 0 then
    outlist := Remove(list,i);
  else
    outlist := list;
  end if;

  if #outlist lt (n-1) then
    return Append(outlist, p);
  else
    return outlist[1..(n-1)] cat [p] cat outlist[n..#outlist];
  end if;
end function;


// returns true iff the degree of all algebras is factor times their dimension.
function DimensionTest (OpList : factor := 2)
  vprint HeckeAlgebra: "Dimension testing...";
  output := true;
  i := 1;
  while output and (i le #OpList) do
    deg := NumberOfRows(OpList[i][1]);
    dim := Dimension(MatrixAlgebra(OpList[i]));
    output := output and (factor*dim eq deg);
    i := i + 1;
  end while;

  return output;
end function;

intrinsic HeckeAlgebras (eps :: GrpDrchElt, weight :: RngIntElt :
    UserBound := 0,        // overwrite HeckeBound by UserBound, 0 means usual HeckeBound
    first_test := 3,       // when is the first dim test performed
    test_interval := 1,    // after how many primes the dim test is performed
    when_test_p := 3,      // in which step test T_p: 0 means usual point
    when_test_bad := 4,    // in which step test T_l with l dividing the level
    test_sequence := [],   // test these operators first
    dimension_factor := 2, // dimension test factor
    ms_space := 0,         // 1 for plus, -1 minus space
    cuspidal := true,      // use cuspical subspace
    DegreeBound := 0,      // degree bound
    OperatorList := [],    // list of precomputed Hecke operators, OperatorList[l] is the l-th op.
    over_residue_field := true, // change the output to residue field
    try_minimal := true,   // should a minimal presentation of the affine algebras be tried?
    force_local := false   // compute T_p's until local
  ) -> SeqEnum, SeqEnum, ModSym, Tup, Tup
{Computes all local Hecke algebras (up to Galois conjugacy) in the
specified weight for the given Dirichlet character.  The function
returns 5 values A,B,C,D,E.  A contains a list of records of type
AlgebraData describing the local Hecke algebra factors. B is a list
containing the local Hecke algebra factors as matrix algebras.  C is
the space of modular symbols used in the computations.  D is a tuple
containing the base change tuples describing the local Hecke
factors. Its knowledge is necessary in order to compute matrices
representing Hecke operators in the local factor.  Finally, E contains
a tuple consisting of all computed Hecke operators for each local
factor of the Hecke algebra.  For a closer description and the usage
of the option, please consult the manual.}

  form := rec< ModularFormFormat |
    Character := eps,
    Weight := weight
  >;

  return HeckeAlgebras(form : 
    UserBound := UserBound,
    first_test := first_test,
    test_interval := test_interval,
    when_test_p := when_test_p,
    when_test_bad := when_test_bad,
    test_sequence := test_sequence,
    dimension_factor := dimension_factor,
    ms_space := ms_space,
    cuspidal := cuspidal,
    DegreeBound := DegreeBound,
    OperatorList := OperatorList,
    over_residue_field := over_residue_field,
    try_minimal := try_minimal,
    force_local := force_local);
end intrinsic;


intrinsic HeckeAlgebras ( t :: Rec :
    UserBound := 0,        // overwrite HeckeBound by UserBound, 0 means usual HeckeBound
    first_test := 3,       // when is the first dim test performed
    test_interval := 1,    // after how many primes the dim test is performed
    when_test_p := 3,      // in which step test T_p: 0 means usual point
    when_test_bad := 4,    // in which step test T_l with l dividing the level
    test_sequence := [],   // test these operators first
    dimension_factor := 2, // dimension test factor
    ms_space := 0,         // 1 for plus, -1 minus space
    cuspidal := true,      // use cuspical subspace
    DegreeBound := 0,      // degree bound
    OperatorList := [],    // list of precomputed Hecke operators, OperatorList[l] is the l-th op.
    over_residue_field := true, // change the output to residue field
    try_minimal := true,   // should a minimal presentation of the affine algebras be tried?
    force_local := false   // compute T_p's until local
  ) -> SeqEnum, SeqEnum, ModSym, Tup, Tup
{Computes all local Hecke algebras (up to Galois conjugacy)
corresponding to the specified modular form t which is supposed to be
of type ModularFormFormat.  The function returns 5 values A,B,C,D,E.
A contains a list of records of type AlgebraData describing the local
Hecke algebra factors. B is a list containing the local Hecke algebra
factors as matrix algebras.  C is the space of modular symbols used in
the computations.  D is a tuple containing the base change tuples
describing the local Hecke factors. Its knowledge is necessary in
order to compute matrices representing Hecke operators in the local
factor.  Finally, E contains a tuple consisting of all computed Hecke
operators for each local factor of the Hecke algebra.  For a closer
description and the usage of the option, please consult the manual.}

  // Initialise some values.
  eps := t`Character;
  weight := t`Weight;
  p := Characteristic(CoefficientRing(eps));
  N := Modulus(eps);

  // Is the Hecke bound overwritten?
  // Let HB contain the bound for the computation.
  if UserBound gt 0 then
    HB := UserBound;
    vprint HeckeAlgebra: "Warning. Using user given bound instead of Hecke bound!";
  else
    HB := HeckeBound(N,weight);
  end if;

  // Is a ModularForm given?
  if assigned t`CoefficientFunction then
    have_mf := true;
    ModularForm := t`CoefficientFunction;
  else
    have_mf := false;
  end if;

  // Let M contain the modular symbols space specified 
  // by the respective options.
  vprint HeckeAlgebra: "Getting modular symbols...";
  if ms_space eq 0 then
    M := ModularSymbols(eps,weight);
  else
    M := ModularSymbols(eps,weight,ms_space);
  end if;
  if cuspidal then
    vprint HeckeAlgebra: "Getting cuspidal subspace.";
    M := CuspidalSubspace(M);
  end if;

  // Let TestList contain prime numbers indicating the
  // sequence in which Hecke operators will be computed.
  TestList := test_sequence;
  // Add all primes smaller than HB.
  l := 2;
  while l le HB do
    if not (l in TestList) then Append(~TestList,l); end if;
    l := NextPrime(l);
  end while;
  // Add p at the right place.
  if when_test_p gt 0 then
    TestList := PutAt(TestList,p,when_test_p);
  end if;
  // Add all "bad" primes at the right places.
  if when_test_bad gt 0 then
    for l in PrimeDivisors(N) do
      // Discard all bad primes beyond the real Hecke bound.
      if l le HeckeBound(N,weight) then
        TestList := PutAt(TestList,l,when_test_bad);
      end if;
    end for;
  end if;

  // Initialise data for the computation.
  Id := HeckeOperator(M,1);
  calc_step := 1;
  BC := <<Id,Id>>;
  OpList := <[Id]>;
  stop_now := false;

  // Do the computation.
  while (calc_step le #TestList) and (not stop_now) do 
    // Use the algorithm for the operator l.
    l := TestList[calc_step];

    // If a list of operators is given, take the operator from it.
    if IsDefined(OperatorList,l) then
      vprint HeckeAlgebra: "Take Hecke operator",l,"from list.";
      T := OperatorList[l];
    else
      vprint HeckeAlgebra: "Compute Hecke operator",l;
      T := HeckeOperator(M,l);
    end if;

    // Restrict the operator to all factors found so far,
    // and decompose further.
    NewBC := <>;
    NewOpList := <>;

    for i := 1 to #BC do
      bc := BC[i];
      S := BaseChange(T,bc);
      S := MatrixAlgebra(CoefficientRing(S),NumberOfRows(S))!S;
      mipo := MinimalPolynomial(S);

      // If a modular form was given, get minimal polynomial from it.
      // Otherwise, factor the minimal polynomial of the operator.
      if (not have_mf) or (N*p mod l) eq 0 then
        vprint HeckeAlgebra: "Factor minimal polynomial.";
        FactorMipo := FactorisationDeg(mipo, DegreeBound);
      else
        pol := Factorisation(ModularForm(l));
        FactorMipo := [];

        for po in pol do
          j := 1;
          while (mipo mod (po[1]^j)) eq 0 do
            j := j + 1;
          end while;

          Append(~FactorMipo,<po[1],j-1>);
        end for; 
      end if;

      // For every factor of the minimal polynomial of the operator,
      // resp. the minimal polynomial of the coefficient of the given modular form,
      // restrict the modular symbols space to the corresponding primary space.
      for fac in FactorMipo do
        vprint HeckeAlgebra: "Compute base change.";

        W := Kernel(Evaluate(fac[1]^(fac[2]),S));
        if Dimension(W) ne 0 then
          bcmat := BaseChangeMatrices(W);
          Append(~NewBC,BaseChange(bc, bcmat));
          Append(~NewOpList,[BaseChange(h,bcmat) : h in OpList[i]] cat [BaseChange(S,bcmat)]);
        end if;
      end for;

    end for;

    BC := NewBC;
    OpList := NewOpList;
   
    // Dimension testing
    if     ((calc_step mod test_interval) eq 0)
       and (calc_step ge first_test) then

      // Is the stop criterion satisfied?
      stop_now := DimensionTest(OpList : factor := dimension_factor);

      // If force_local then only stop if all algebras are local.
      if force_local then
        // Check whether all algebras are local.
        ij := 1;
        while stop_now and (ij le #OpList) do
          stop_now := #MaximalIdeals(MatrixAlgebra(OpList[ij])) eq 1; 
          ij := ij + 1;
        end while;
      end if;
    end if;

    // Make the next step.
    calc_step := calc_step + 1;
  end while;

  // Attention: the algebras need not yet be local. That will be
  // taken care of now.

  vprint HeckeAlgebra: "Doing final localisations...";

  NewBC := <>;
  NewOpList := <>;
  AlgebraList := [];

  for i := 1 to #OpList do
    if over_residue_field then
      bctup := DecompositionUpToConjugation(OpList[i]);
    else
      _,bctup := Localisations(OpList[i]);
    end if;

    for bcmat in bctup do
      Append(~NewBC,BaseChange(BC[i],bcmat));
      ol := [BaseChange(h,bcmat) : h in OpList[i]];
      Append(~NewOpList,ol);
      Append(~AlgebraList,MatrixAlgebra(ol));
    end for;
  end for;

  BC := NewBC;
  OpList := NewOpList;

  vprint HeckeAlgebra: "Creating output...";

  // create output
  DataList := [];
  for a in AlgebraList do
    b := AffineAlgebraTup(a : try_minimal := try_minimal);
    outEl := rec < AlgebraData |
      Level := N,
      Weight := weight,
      Characteristic := p,
      BaseFieldDegree := Degree(CoefficientRing(eps)),

      CharacterOrder := Order(eps),
      CharacterConductor := Conductor(eps),
      CharacterIndex    := Index(Elements(Parent(eps)), eps),

      AlgebraFieldDegree := Degree(b[1]),
      ResidueDegree := Degree(CoefficientRing(a)) * 
                       (Dimension(a) - Dimension(MaximalIdeals(a)[1])),
      Dimension := Dimension(a),
      GorensteinDefect := GorensteinDefect(a),

      EmbeddingDimension := b[2],
      NilpotencyOrder    := b[3],
      Relations          := b[4],

      NumberGenUsed     := calc_step - 1
    >;

    if assigned t`ImageName then outEl`ImageName := t`ImageName; end if;
    if assigned t`Polynomial then outEl`Polynomial := t`Polynomial; end if;

    Append(~DataList, outEl);
  end for;

  return DataList, AlgebraList, M, BC, OpList; 
end intrinsic;

