/* ************************************************************
   **                                                        **
   ** A5.mg                                                  **
   **                                                        **
   ** Handling of A5-forms.                                  **
   **                                                        **
   ** Gabor Wiese                                            **
   ** version of 01/01/08                                    **
   **                                                        **
   ************************************************************ */

import "Structure.mg" : ModularFormFormat, AlgebraData;


// test for wild ramification
IsWR := function (L)
  outp := false;

  for l in L do
    p := l[1];
    for r in l[2] do
      outp := outp or ((r[1] mod p) eq 0);
    end for;
  end for;

  return outp;
end function;

// for a ramification tuple <e,f> return the
// mod 2 conductor and the order of twist.

getCond := function (t)
  s := SequenceToSet(t[2]);
  if <5,1> in s then
    if (t[1]-1) mod 5 eq 0 then
      return <t[1],1,5>;
    else
      return <t[1],2,1>;
    end if;
  elif (<3,1> in s) then
    if (t[1]-1) mod 3 eq 0 then
      return <t[1],1,3>;
    else
      return <t[1],2,1>;
    end if;
  else
    return <t[1],1,1>;
  end if;
end function;

getChar := function (cond)
  eps := DirichletGroup(1,GF(2))!1;
  N := 1; deg := 1;
  for c in cond do
    if c[3] eq 1 then
      G := DirichletGroup(c[1],GF(2));
      eps1 := G!1;
    elif c[3] eq 3 then
      G := [e : e in Elements(DirichletGroup(c[1],GF(4))) | Order(e) eq 3];
      if deg eq 1 then deg := 2; end if;
      if #G eq 0 then 
        eps1 := DirichletGroup(c[1],GF(2,deg))!1;
      else
        eps1 := G[1];
      end if;
    elif c[3] eq 5 then
      G := [e : e in Elements(DirichletGroup(c[1],GF(16))) | Order(e) eq 5];
      deg := 4;
      if #G eq 0 then 
        eps1 := DirichletGroup(c[1],GF(2,deg))!1;
      else
        eps1 := G[1];
      end if;
    end if;

    N := N * c[1];
    eps := DirichletGroup(N,GF(2,deg))!eps * DirichletGroup(N,GF(2,deg))!eps1;
  end for;

  return eps;
end function;

getCoeffPoly := function(l,eps,f)
  K := NumberField(f);
  O := MaximalOrder(K);
  R<x> := PolynomialRing(CoefficientRing(eps));

  F := {Degree(P[1]) : P in Factorisation(l*O)};
  if 5 in F then
    return x^2 + Evaluate(eps,l)*x + Evaluate(eps,l)^2;
  elif 3 in F then
    return x + Evaluate(eps,l);
  else
    return x;
  end if;
end function;

intrinsic A5Form (f :: RngUPolElt) -> Rec
{Returns the A5 form in characteristic 2 and weight 2 of smallest predicted level
corresponding to the polynomial f. No checks about f are performed.} 
  K := NumberField(f);
  O := MaximalOrder(K);
  D := Factorisation(Discriminant(O));

  // get ramification type
  L := [];
  for d in D do
    p := d[1];
    F := Factorisation(p*O);
    Append(~L,<p,[<P[2], Degree(P[1])> : P in F]>);
  end for;

  // test for wild ramification
  error if IsWR(L), "Error. Wild ramification is not supported."; 

  // compute the conductor
  cond := [getCond(l) : l in L];
  co := 1;
  for c in cond do co := co * (c[1]^(c[2])); end for;

  // compute the character
  eps := getChar(cond);
  // give it the good modulus
  eps := DirichletGroup(co,CoefficientRing(eps))!eps;
 
  // compute the coefficient function
  coeff := map< Integers() -> PolynomialRing(CoefficientRing(eps)) |
                l :-> getCoeffPoly(l,eps,f) >;

  form := rec< ModularFormFormat |
    Character := eps^2,
    Weight := 2,
    CoefficientFunction := coeff,
    ImageName := "A_5",
    Polynomial := f
  >;
  return form;
end intrinsic;

