/* ************************************************************
   ** EigenformsWt1.mg                                       **
   **                                                        **
   ** Gabor Wiese                                            **
   ** version of 01/01/2008                                  **
   **                                                        **
   ************************************************************ */

import "Structure.mg" : ModF, WholeLevel;

// *************** weight 1 functions ***************************

intrinsic EigenformWt1 ( W :: Rec, SoEV :: Rec ) -> SeqEnum
{Given a system of eigenvalues SoEV, computes a list of 
eigenform belonging to it in weight 1.
So the list is either empty or contains precisely one element.}
  require Set(Names(Format(SoEV))) eq Set(Names(ModF)) : "Argument 1 must have recformat ModF.";
  p := SoEV`Characteristic;
  N := SoEV`Level;
  if assigned SoEV`Character then
    epsp := Evaluate(SoEV`Character,p);
  else
    epsp := 1;
  end if;

  // first do the fast test, and hope it is conclusive
  Tp := HeckeOperator(W,SoEV,p);
  mipo := MinimalPolynomial(Tp);
  m := Factorisation(mipo);
  F := ext< GF(p,SoEV`FieldDegree) | Lcm ([ Degree(a[1]) : a in m ]) >;
  m := Factorisation(PolynomialRing(F)!mipo);
  if #m eq 2 then
    wt1 := true;
    conclusive := true;
  else
    if (Evaluate(m[1][1],0))^2 ne epsp then
      wt1 := false;
      conclusive := true;
    else
      if m[1][2] eq 1 then
        wt1 := false;
        conclusive := true;
      else
        conclusive := false;
      end if;
    end if;
  end if;

  AComputed := false;
  if not conclusive then
    // Here we are local. Can we do with a smaller bound?
    b := HeckeBound(N, p+2);
    T1 := HeckeOperator(W,SoEV,1);
    gens := [HeckeOperator(W,SoEV,l) : l in [2..b] | IsPrime(l) and (p ne l)];
    A := sub<Parent(T1) | T1, gens>;
    AComputed := true;
    
    wt1 := not (Tp in A);
  end if;

  if not wt1 then 
    return [];
  else
    // compute the algebras A and the full one T in weight p
    if not AComputed then
      // Here we are local. Can we do with a smaller bound?
      b := HeckeBound(N, p+2);
      T1 := HeckeOperator(W,SoEV,1);
      gens := [HeckeOperator(W,SoEV,l) : l in [2..b] | IsPrime(l) and (p ne l)];
      A := sub<Parent(T1) | T1, gens>;
      AComputed := true;
    end if;
    Tp := HeckeOperator(W,SoEV,p);
    T := sub<Parent(T1) | T1, Tp, gens>;

    // get a basis for T/A
    bas := [T!b : b in Basis(A)];
    bas1 := []; // will contain indices of the operators in the basis

    i := 1;
    while (#bas lt Dimension(T)) do
      t := HeckeOperator(W,SoEV,i);
      if IsIndependent(Append(bas,t)) then 
        bas := Append(bas,t);
        bas1:= Append(bas1,i);
      end if;
      if i eq 1 then i := p; else i := i + p; end if;
    end while;

    // prepare for the computation of the weight 1 operators
    d := Nrows(T1);
    MS := KMatrixSpace(CoefficientRing(T1), d, d);
    MS := KMatrixSpaceWithBasis([MS!b : b in bas]);
    MA1 := MatrixAlgebra(CoefficientRing(T1), #bas1);

    // create the operator F
    F := MA1!0;
    for i := 1 to #bas1 do 
      F[i] := Vector(Coordinates(MS, MS!HeckeOperator(W,SoEV,bas1[i] div p))
            [(#bas-(#bas1)+1)..#bas]);
    end for;
    F := Transpose(F);

    // create the weight 1 eigenform
    mf := rec< ModF | 
      Level             := SoEV`Level,
      Characteristic    := SoEV`Characteristic,
      FieldDegree       := SoEV`FieldDegree,
      Dimension         := #bas1,
      Weight            := 1,
      Basis             := SoEV`Basis,
      F                 := F,
      MS                := MS,
      BasisWt1          := bas1
    >;
    if assigned SoEV`Character then
      mf`Character := SoEV`Character;
    end if;
    StoreHeckeOperators(W, ~mf);
    StoreCoefficients(~mf);
    StoreHeckeAlgebra(~mf);
    return [mf];
  end if;
end intrinsic;

intrinsic EigenformsWt1 ( W :: Rec, SoEV :: Rec ) -> SeqEnum
{Given a system of eigenvalues SoEV, computes a list of 
eigenform belonging to it in weight 1.
So the list is either empty or contains precisely one element.}
  // The same as EigenformWt1, just with an additional "s",
  // since I always write it. The answer, however, will only
  // contain 1 element, or be empty.
  return EigenformWt1 (W,SoEV);
end intrinsic;

intrinsic EigenformsWt1 ( W :: Rec, L :: SeqEnum ) -> SeqEnum
{Given a list L of systems of eigenvalues, computes a list of 
eigenform belonging to it in weight 1.}
  output := [];
  for l in L do
    output := output cat EigenformWt1(W,l);
  end for;
  return output;
end intrinsic;

intrinsic EigenformsWt1 ( W :: Rec ) -> SeqEnum
{Given a whole level W, computes a list of 
eigenform belonging to it in weight 1.}
  output := [];
  for l in W`SoEVs do
    output := output cat EigenformWt1(W,l);
  end for;
  return output;
end intrinsic;
