/* ************************************************************
   **  Hecke.mg                                              **
   **                                                        **
   **  Hecke operators, algebras and coefficients            **
   **  for weight 1 computations                             **
   **                                                        **
   **  Gabor Wiese                                           **
   **  version of 01/01/2008                                 **
   **                                                        **
   ************************************************************ */

import "Structure.mg" : ModF, WholeLevel;

// *********** Hecke bound **************************

intrinsic HeckeBound ( N :: RngIntElt, k :: RngIntElt ) -> RngIntElt
{Computes the Hecke bound for level N and weight k.}
  local B,L;

  B := k * N  / 12; 
  L := Factorization (N);

  for i := 1 to #L do
    B := B * (1 + (1/L[i][1])); 
  end for;

  return Ceiling(B);  
end intrinsic;

intrinsic HeckeBound ( eps :: GrpDrchElt, k :: RngIntElt ) -> RngIntElt
{Computes the Hecke bound for the character eps in weight k.}
  return HeckeBound (Modulus(eps),k);
end intrinsic;

intrinsic HeckeBound ( mf :: Rec ) -> RngIntElt
{Computes the Hecke bound for the level and weight of
the eigenform mf.}
  return HeckeBound (mf`Level, mf`Weight);
end intrinsic;



// *********** Hecke algebras **************************************

intrinsic HeckeAlgebra ( mf :: Rec : bound_factor := 1 ) -> AlgMat
{Computes the matrix algebra generated all Hecke operators.}
  if assigned mf`HeckeAlgebra then
    return mf`HeckeAlgebra;
  else
    StoreHeckeAlgebra(~mf);
    return mf`HeckeAlgebra;
  end if;
end intrinsic;

intrinsic StoreHeckeAlgebra ( ~mf :: Rec : bound_factor := 1 )
{Computes the matrix algebra generated all Hecke operators.}
  p := mf`Characteristic;
  N := mf`Level;
  // This is the good bound in the global case. 
  // However, it seems we can do with much less after localisation.
  b := Ceiling(HeckeBound(N, p) * bound_factor);
  T1 := HeckeOperator(mf,1);
  gens := [HeckeOperator(mf,l) : l in [2..b] | IsPrime(l) ];
  
  H := sub<Parent(T1) | T1, gens>;

  mf`HeckeAlgebra := sub<Generic(H) | Basis(H)>;
  return;
end intrinsic;



// ************ Hecke operators ***************************

forward getit;
getit := function (Tp,p,r,k,N,eps)
  if r eq 0 then return Parent(Tp)!1; end if;
  if r eq 1 then return Tp; end if;
  TT := Tp * getit(Tp,p,r-1,k,N,eps);
  if (N mod p) ne 0 then
    if Type(eps) eq GrpDrchElt then
      TT := TT - Evaluate(eps,p) * p^(k-1) * getit(Tp,p,r-2,k,N,eps);
    else
      TT := TT - p^(k-1) * getit(Tp,p,r-2,k,N,eps);
    end if;
  end if;
  return TT;
end function;

intrinsic HeckeOperator ( R :: Rec, i :: RngIntElt ) -> Mtrx
{Returns the ith Hecke operator for the specified level.}
  // decide first whether R is a "whole level" or a "eigenform"
  if "tp" in Names(R) then
    // it seems to be of type "whole level"
    WL := R;
    indexl := [a[1] : a in WL`HeckeList];
    f := Factorisation(i);

    alg := Parent(WL`HeckeList[1][2]);
    T := alg!1;
    for a in f do
      ind := Index(indexl,a[1]);
      if ind eq 0 then
        Tp := WL`tp(a[1]);
      else
        Tp := WL`HeckeList[ind][2];
      end if;

      if assigned WL`Character then
        T := T * getit (Tp,a[1],a[2],WL`k,WL`N, WL`Character);
      else
        T := T * getit (Tp,a[1],a[2],WL`k,WL`N, -1);
      end if;
    end for;

    return T;
  else
    // R seems to be an eigenform
    mf := R;
    require assigned mf`HeckeList : "A list of Hecke operators must have been computed.";
    indexl := [a[1] : a in mf`HeckeList];
    f := Factorisation(i);
    indexf := [a[1] : a in f];
    require Set(indexf) subset Set(indexl) :
      "All Hecke operators for primes dividing the second argument must have been computed.";

    if assigned mf`Weight then
      k := mf`Weight;
    else
      k := mf`Characteristic;
    end if;

    alg := Parent(mf`HeckeList[1][2]);
    T := alg!1;
    for a in f do
      ind := Index(indexl,a[1]);
      Tp := mf`HeckeList[ind][2];
      if assigned mf`Character then
        T := T * getit (Tp,a[1],a[2],k,mf`Level,mf`Character);
      else
        T := T * getit (Tp,a[1],a[2],k,mf`Level,-1);
      end if;
    end for;
    return T;
  end if;
end intrinsic;

forward getOp1;
getOp1 := function (i,WL,mf)
  N := mf`Level;
  bas1 := mf`BasisWt1;
  F := mf`F;
  d := #bas1;
  MS := mf`MS;

  dim := Dimension(MS);
  alg := Parent(F);
  K := CoefficientRing(F);
  p := Characteristic(K);

  if assigned mf`Character then
    epsp := Evaluate(mf`Character,p);
  else
    epsp := 1;
  end if;

  if i eq 1 then
    return alg!1;
  end if;

  if IsPrime(i) then
    t := alg!0;
    hop := BaseChange(HeckeOperator(WL,i),mf`Basis);
    for j := 1 to d do 
       t[j] := Vector(Coordinates(MS, MS!((MS.(dim-d+j))*hop))
              [(dim-d+1)..dim]);
    end for;
    t := Transpose(t);

    if i eq p then 
      return t + epsp * F;
    else
      return t;
    end if;
  else
    f := Factorisation(i);

    if #f eq 1 then
      // a prime power
      tp  := getOp1(f[1][1], WL, mf);
      tp1 := getOp1(f[1][1]^(f[1][2]-1), WL, mf);

      if (N mod f[1][1]) ne 0 then
        tp2 := getOp1 (f[1][1]^(f[1][2]-2), WL, mf);
        if assigned mf`Character then
          return tp * tp1 - Evaluate(mf`Character, f[1][1])*tp2;
        else
          return tp * tp1 - tp2;
        end if;
      else
        return tp * tp1;
      end if;
    else
      t := getOp1 (f[1][1]^(f[1][2]), WL,mf);
      for j := 2 to #f do
        t := t * getOp1 (f[j][1]^(f[j][2]), WL,mf);
      end for;
      return t;
    end if;
  end if;
end function;

intrinsic HeckeOperator ( WL :: Rec, mf :: Rec, i :: RngIntElt ) -> Mtrx
{Returns the ith Hecke operator for the specified level.}
  if assigned mf`Weight then
    if mf`Weight eq 1 then
      return ChangeTypeToAlgebra(getOp1(i,WL, mf));
    end if;
  end if;

  return ChangeTypeToAlgebra(BaseChange(HeckeOperator(WL,i),mf`Basis));
end intrinsic;


// ******************* Coefficients *********************************

intrinsic Coefficient ( WL :: Rec, mf :: Rec, i :: RngIntElt ) -> Any
{Returns the ith coefficient.}
  if assigned mf`CoefficientList then
    indexl := [a[1] : a in mf`CoefficientList];
    a := Index(indexl,i);
    if a ne 0 then
      c := mf`CoefficientList[a][2];
      if Type(c) eq AlgMatElt then
        return Trace(c);
      else
        return c;
      end if;
    end if;
  end if;

  return SetToSequence(Eigenvalues(HeckeOperator(WL,mf,i)))[1][1];
end intrinsic;

intrinsic Coefficient ( mf :: Rec, i :: RngIntElt ) -> Any
{Returns the ith coefficient.}
  if assigned mf`CoefficientList then
    indexl := [a[1] : a in mf`CoefficientList];
    a := Index(indexl,i);
    if a ne 0 then
      c := mf`CoefficientList[a][2];
      if Type(c) eq AlgMatElt then
        return Trace(c);
      else
        return c;
      end if;
    end if;
  end if;

  return SetToSequence(Eigenvalues(HeckeOperator(mf,i)))[1][1];
end intrinsic;


// ********* Storing Hecke operators *********************************

intrinsic StoreHeckeOperator (~WL , i )
{Stores the prime operators necessary for the computation
of the ith Hecke operator in memory.}
  require "SoEVs" in Names(WL) :
    "Argument 1 must be a whole level.";

  indexl := [a[1] : a in WL`HeckeList];
  f := Factorisation(i);
  for p in f do
    a := Index(indexl,p[1]);
    if a eq 0 then
      T := WL`tp(p[1]);
      Append(~WL`HeckeList, <p[1],T>);
    end if;
  end for;
end intrinsic;

intrinsic StoreHeckeOperators (~WL , a, b )
{Stores the prime operators necessary for the computation
of the Hecke operator in the range [a..b] in memory.}
  for i := a to b do
    StoreHeckeOperator(~WL,i);
  end for;
end intrinsic;

intrinsic StoreHeckeOperators( WL :: Rec, ~mf :: Rec )
{Stores those prime Hecke operators of the eigenform mf
in the attribute HeckeList, which are stored for the 
whole level WL specified.}
  if not (assigned mf`HeckeList) then mf`HeckeList := []; end if;
  indexlM := [a[1] : a in mf`HeckeList];
  indexl  := [a[1] : a in WL`HeckeList];
  for i in indexl do
    if not (i in indexlM) then
      Append(~mf`HeckeList, <i, HeckeOperator(WL, mf, i)>);
    end if;
  end for;
end intrinsic;

intrinsic StoreHeckeOperator( WL :: Rec, ~mf :: Rec, i )
{Stores the prime Hecke operators of the eigenform mf
in the attribute HeckeList, which are necessary for the
computation of the ith Hecke operator.}
  if not (assigned mf`HeckeList) then mf`HeckeList := []; end if;
  indexlM := [a[1] : a in mf`HeckeList];
  indexl  := [a[1] : a in WL`HeckeList];
  f := Factorisation(i);
  for p in f do
    if not (p[1] in indexlM) then
      Append(~mf`HeckeList, <i, HeckeOperator(WL, mf, p[1])>);
    end if;
  end for;
end intrinsic;

intrinsic StoreHeckeOperators( WL :: Rec, ~mf :: Rec, a, b )
{Stores the prime Hecke operators of the eigenform mf
in the attribute HeckeList, which are necessary for the
computation of the Hecke operators in the range [a..b].}
  for i := a to b do
    StoreHeckeOperator(WL, ~mf, i);
  end for;
end intrinsic;
 

// ********* Storing Coefficients *********************************

intrinsic StoreCoefficients( ~mf :: Rec )
{Stores those prime coefficients of the eigenform mf
in the attribute CoefficientList, for which the corresponding
Hecke operators are stored.}
  if not (assigned mf`CoefficientList) then mf`CoefficientList := []; end if;
  indexlM := [a[1] : a in mf`CoefficientList];
  indexl  := [a[1] : a in mf`HeckeList];
  for i in indexl do
    if not (i in indexlM) then
      Append(~mf`CoefficientList, <i, Coefficient(mf, i)>);
    end if;
  end for;
end intrinsic;

