/* ************************************************************
   ** SystemsOfEigenvalues.mg                                **
   **                                                        **
   ** Systems of eigenvalues for weight one compuations.     **
   **                                                        **
   ** Gabor Wiese                                            **
   ** version of 01/01/2008                                  **
   **                                                        **
   ************************************************************ */

import "Structure.mg" : ModF, WholeLevel;

// *********** A priori exclusion of systems of evs **************

invPol := function (f, epsp)
  F := SplittingField(f);
  prF<X> := PolynomialRing(F);
  mF := Factorisation(prF!f);
  g := prF!1;
  for h in mF do
    g := g * (X + epsp*((Evaluate(h[1],0))^(-1)))^(h[2]);
  end for;

  return Parent(f)!g;
end function;


doF2 := function( M, epsp : utc := true ) 
  pr<x> := PolynomialRing(CoefficientRing(M));
  mf := Factorisation(MinimalPolynomial(M));
  monSeq := [ mf[i][1] : i in [1..#mf]]; // the prime factors

  wt1polList := <>;
  notL := [];
  for i := 1 to #mf do
    if not (i in notL) then
      g := mf[i];
      if (g[1] eq pr!x) then
        // case 0
        // print g[1], " case 0: weight p";
      elif (Degree(g[1]) eq 1) and ((Evaluate(g[1],0))^2 eq epsp) then
        // case 1
        if g[2] gt 1 then
          Append(~wt1polList, g[1]^(g[2]));
          // print g[1], " case 1: possibly weight 1. Join",g[1]^(g[2]);
        else
          // print g[1], " case 1: not weight 1 because exponent 1";
        end if;
      else
        r := invPol (g[1],epsp);
        j := Position(monSeq,r);
        if j eq 0 then
          // print g[1], " case 4: not weight 1.";
          // case 4, so discard
        else
          // case 2 or 3, no difference
          // we don't want to get the same roots twice
          Append(~notL, j);

          F := SplittingField(g[1]);
          prF<X> := PolynomialRing(F);
          gF := Factorisation(prF!(g[1]));

          if utc then
            // up to conjugation, take one root
            a := Evaluate(gF[1][1],0);
            pol := (X+a)^(mf[i][2]) * (X+epsp*(a^(-1)))^(mf[j][2]);
            d := Lcm({Degree(MinimalPolynomial(Eltseq(pol)[ii])) : ii in [1..Degree(pol)+1]});
            prF := PolynomialRing(GF(Characteristic(F),d));
            Append(~wt1polList, prF!pol);

            // print g[1], " case 2 or 3: possibly weight 1.",i,j," Join ",prF!pol;
          else
            setF := [ Evaluate(u[1],0) : u in gF ];
            while #setF ne 0 do
              a := setF[1];
              setF := Exclude(setF,a);
              setF := Exclude(setF,epsp*(a^(-1)));
              pol := (X+a)^(mf[i][2]) * (X+epsp*(a^(-1)))^(mf[j][2]);
              d := Lcm({Degree(MinimalPolynomial(Eltseq(pol)[ii])) : ii in [1..Degree(pol)+1]});
              prF := PolynomialRing(GF(Characteristic(F),d));
              Append(~wt1polList, prF!pol);
            end while;
          end if;
        end if;
      end if;
    end if;
  end for;

  return wt1polList;
end function;


// *********** Systems of eigenvalues ****************************

intrinsic SystemsOfEigenvalues ( 
   ~WL :: Rec          :
   char := 0,
   UpToConjugation := true,
   Wt1APriori := false
)
{Computes the systems of eigenvalues belonging to eigenforms of 
the given "whole level" WL.}
  N := WL`N;
  k := WL`k;
  require (char eq 0) or (char eq k) : 
    "The characteristic of the modular symbols must be 0 or equal to the weight.";
  if (assigned WL`Character) then
    // The character uniquely determines the characteristic of the modular
    // symbols, thus evtl. overruling the user
    // Problem is with lattices in maximal order of cyclotomic fields.
    char := Characteristic(Parent(Evaluate(WL`Character,1)));
    epsp := Evaluate(WL`Character,k);
  else
    epsp := 1;
  end if;

  // one needs the bigger (than usual) bound to make sure that
  // the subalgebra (outside) p is fully generated.
  B := HeckeBound(N,k+2);

  if char eq 0 then
    vprint Weight1: "Using integral modular symbols.";
    if assigned WL`Character then
      ms := CuspidalSubspace(ModularSymbols(WL`Character,k,+1));
    else
      ms := CuspidalSubspace(ModularSymbols(N,k,+1));
    end if;
    vprint Weight1: "Computing intregral Hecke operators.";
    hl := [HeckeOperator(ms,i) : i in [1..B]];

    vprint Weight1: "Computing the lattice.";
    T := ReduceDimensionAndGetLattice(hl : dim := Dimension(ms));

    vprint Weight1: "Computing reduced Hecke list.";
    algF := MatrixAlgebra(GF(k),Nrows(T[1]));
    WL`HeckeList := [<i,algF!(BaseChange(hl[i],T))> : i in [1..B] | IsPrime(i)];
    delete hl;
    WL`tp := map<Integers() -> algF 
          | i :-> algF!(BaseChange(HeckeOperator(ms,i),T))>;

  else
    vprint Weight1: "Using characteristic", char, "modular symbols."; 
    if assigned WL`Character then
      ms := CuspidalSubspace(ModularSymbols(WL`Character,k));
    else
      ms := CuspidalSubspace(ModularSymbols(N,k,GF(k)));
    end if;
    WL`tp := map<Integers() -> Parent(HeckeOperator(ms,1)) 
          | i :-> HeckeOperator(ms,i)>;
    WL`HeckeList := [<i,WL`tp(i)> : i in [1..B] | IsPrime(i)];
  end if;

  vprint Weight1: "Computing Hecke algebra generators.";
  L := [ HeckeOperator(WL,n) : n in [2..B] | IsPrime(n) and ((n mod k) ne 0) ];

  if Wt1APriori then
    vprint Weight1: "Performing a priori exclusion.";
    Tp := HeckeOperator(WL,k);
    wt1polList := doF2 (Tp, epsp : utc := UpToConjugation);

    Lpos := <>;
    for pol in wt1polList do
      T := BaseChangeMatrices(Kernel(Evaluate(pol, ChangeRing(Tp, CoefficientRing(pol)))));
      Append(~Lpos, T);
    end for;
  else
    Lpos := <<Matrix(Parent(HeckeOperator(WL,1))!1), Matrix(Parent(HeckeOperator(WL,1))!1)>>;
  end if;

  vprint Weight1: "Computing systems of eigenvalues.";
  if UpToConjugation then
    dec := <>;
    for T in Lpos do
      dd := DecompositionUpToConjugation([BaseChange(l,T) : l in L]);
      for da in dd do
        Append(~dec, BaseChange(T,da));
      end for;
    end for;
  else
    dec := <>;
    for T in Lpos do
      dd := DecompositionOverResidueField([BaseChange(l,T) : l in L]);
      for da in dd do
        Append(~dec, BaseChange(T,da));
      end for;
    end for;
  end if;

  output := [];
  for d in dec do
    mf := rec< ModF | Level := N, Characteristic := k>;
    deg := Degree(CoefficientRing(d[1]));
    dim := Nrows(d[1]);
    mf`FieldDegree := deg;
    mf`Dimension := dim;
    mf`Basis := d;
    if assigned WL`Character then
      mf`Character := WL`Character;
    end if;
    Append (~output, mf);
  end for;

  WL`SoEVs := output;
end intrinsic;

