/*
********************************************
* AlgebraFactors.mg                        *
*                                          *
* Computing the direct factors of a        *
* p-adic algebra.                          *
*                                          *
* Gabor Wiese, version of 24/01/2014       *
********************************************
*/

/* create the regular representation, used for restricting algebras to factors */

intrinsic pAdicRegularRepresentation ( AF :: Rec ) -> AlgMat, Any
{Input: A p-adic algebra in pAdicAlgebraFormat.
Output: A matrix algebra corresponding to the regular representation.}
  if not assigned(AF`adic_coords) then _compute_adic_coords (~AF); end if;

  M := MatrixAlgebra(AF`R,AF`dim);

  conv := function(b)
    l := M!0;
    for i := 1 to AF`dim do
      l[i] := AF`adic_coords(b*AF`basis[i]);
    end for;
    return l;
  end function;

  L := [];
  for b in AF`basis do
    Append (~L, conv(b));
  end for;

  return pAdicAlgebra(L), conv;
end intrinsic;


/* functions for cutting the algebra into factors */

intrinsic pAdicAlgebraFactor (AF :: Rec, e :: Any) -> Rec, Any
{Input: pAdicAlgebra AF, idempotent e.
Output: The local factor corresponding to e in pAdicAlgebraFormat and function restricting from AF to the factor.}
    Q := pAdicAlgebra([e*(AF`Aadic!w) : w in AF`basis]);
    T, conv := pAdicRegularRepresentation(Q);
    phi := function(a)
      return conv(e*a);
    end function;
  return T, phi;
end intrinsic;


intrinsic pAdicAlgebraFactors (AF :: Rec) -> SeqEnum
{Input: pAdicAlgebra AF.
Output: Tuples of form <T, phi> with T a local factor in pAdicAlgebraFormat and phi a function restricting from AF to T.}
  _compute_adic_idempotents(~AF);
  out := <>;
  for e in AF`ipadic do
    T, phi := pAdicAlgebraFactor(AF,e);
    Append(~out, <T,phi>);
  end for;

  return out;
end intrinsic;



/* functions for computing idempotents */

intrinsic _compute_adic_idempotents (~AF)
{}
  _compute_mod_idempotents(~AF);
  if not assigned(AF`ipadic) then
    AF`ipadic := [LiftIdempotent(AF,a) : a in AF`ipmod];
  end if;
end intrinsic;

intrinsic _compute_mod_idempotents (~AF)
{}
  if not assigned(AF`ipmod) then
    AF`ipmod := CompleteOrthogonalIdempotents([AF`Amod!a : a in AF`basis]);
  end if;
end intrinsic;



/* functions for computing coordinate functions */

intrinsic _compute_adic_coords (~AF)
{}
  if not assigned(AF`adic_coords) then
    MV := RSpace(AF`R, (AF`deg)^2);
    MR := RMatrixSpace(AF`R,AF`dim,(AF`deg)^2);

    // coordinate function
    basis_matrix := MR!0;
    for i := 1 to AF`dim do
      basis_matrix[i] := MV!(Vector(AF`basis[i]));
    end for;
    _,B := EchelonForm(Transpose(basis_matrix));
    B := Transpose(B);
    AF`adic_coords := function (b)
      return Vector(Eltseq((MV!(Vector(b))*B)) [1..AF`dim]);
    end function;
  end if;
end intrinsic;

intrinsic _compute_mod_coords (~AF)
{}
  if not assigned(AF`mod_coords) then
    MV := RSpace(GF(AF`p), (AF`deg)^2);
    MR := RMatrixSpace(GF(AF`p),AF`dim,(AF`deg)^2);

    // coordinate function
    basis_matrix := MR!0;
    for i := 1 to AF`dim do
      basis_matrix[i] := MV!(Vector(AF`basis[i]));
    end for;
    _,B := EchelonForm(Transpose(basis_matrix));
    B := Transpose(B);
    AF`mod_coords := function (b)
      return Vector(Eltseq((MV!(Vector(b))*B)) [1..AF`dim]);
    end function;
  end if;
end intrinsic;

