!!! MODULE: groups
!!! Description: routines for initializing the group we work with
!!!              including finding coset-representatives and cusps etc.

!!!  Contains:
!!!  subroutine init_g(N,init_skel)    
      module groups
      use commonvariables
      use matrix
      use numtheory
      
      implicit none
      integer::index
      integer,save :: Level  !! Determines a Hecke congruence subgroup Gamma_0(Level)
      integer,dimension(1:2,1:2),save :: mapS,mapT,mapTi,mapId
      integer, dimension(:,:),allocatable :: cusps
      real(DP),dimension(:),allocatable::cusp_width
      integer, dimension(:,:),allocatable :: vertices
      INTEGER,save :: num_reps
      INTEGER,save :: num_cusps
      INTEGER,save :: num_vertex
      integer,save :: group_silent !! =-1 for real silent, 0 for normal and 1 debugging purposes
      logical,save::grp_inited,from_generators
      logical,save::from_perm
      logical,save::gamma_0N,hecke_triangle
      !! from_perm = .true. if the group is inited from permutations
      real(kind=DP),dimension(:,:,:),allocatable :: W
      ! sigma_j is a set of maps that takes each CUSP to infinity     
      ! Well, not exactly, by some misfortune
      ! sigma_j here is N_j^-1 !!!
      real(kind=DP),dimension(:,:,:),allocatable :: sigma_j
      integer(kind=8),dimension(:),allocatable::sigma_j_in_normalizer

!!! Taken from file _permutations.f and included here instead
      integer,dimension(:,:),allocatable::all_cusps_info
      
      integer,dimension(:),allocatable::e2,e3
      integer,dimension(:),allocatable::genus


      ! This is the number of the combinations 
      ! of       integer, dimension(:,:),allocatable :: verticessymmetryeigenvalues of normalized cusps 
      ! e.g. number of combinations of +-1 attached to each normalized cusp
      ! except for the first (that is always =1 to normalize things)
      integer :: num_norm_cusp_combs
      ! and a vector of the said combinations
      integer,dimension(:,:),allocatable :: norm_cusp_combs
      ! vertex maps that maps each vertex to the corresponding
      ! cusp this belongs to the group so the coefficients are integers
      integer,dimension(:,:,:),allocatable :: Uw
      real(DP),dimension(:,:,:),allocatable :: cusp_fixer
      integer,dimension(:,:,:),allocatable::reps
      ! the list of coset representatives

      !! generators for a group (must be given!)
      real(DP),dimension(:,:,:),allocatable::generators
      integer,save ::num_gens
      integer,save::order
!!! This is the order of the elliptic element if we have a two-generator
      !!! group generated by z->z+1 and z->(cos(Pi/2n)z+sin^2(PI72n))/
      !!!   / (-z+cos(PI/2n))

      ! this is actually the same as the reps, but ordered as the cusps
      integer,dimension(:,:,:),allocatable :: Vj
      integer*8,dimension(100000)::reps_code   ! a number describing whhich sequence of S,T,T^-1 reps have
  !    real(DP),dimension(:),allocatable::widths  !! widths for all cusps
      real(DP),save::Width
      real(DP),save::Y0_min  ! actually a value below the minimal point of fund.dom
      integer ::real_num_cosets
      integer ::real_num_cusps
      logical,save::cycloidal
      integer::this_group
      integer,save::hecke_q
      logical::read_one_group
      contains      
      

!! Initializing a Hecke triangle group
      subroutine init_hecke_triangle(q)
      integer::q
      hecke_q=q
      width=2.0_DP*cos(PI/real(Q,DP))
      hecke_triangle=.true.
      call init_g(1)
      end subroutine

      ! Initialize  everything for the group
      ! Gamma_0(N)
     
      subroutine init_g(N,init_skel) !,fr_perm,group_nr)
      use commonvariables
!      use permutations
!     use characters
      implicit none
      integer,intent(in) :: N
! Init only a skeleton, so we can get data from file
      logical,optional,intent(in) :: init_skel
!      logical,optional,intent(in) :: fr_perm 
!      integer,optional::group_nr  !! used in combination with permutation
      logical :: in_skel
      integer::g_nr
!      width=1.0_DP
!      group_silent=2
      if(.not.(hecke_triangle)) then 
         gamma_0N=.true.
      endif
      if(group_silent.gt.0) then
         write(*,*) 'In init_g!'
      endif
#ifdef debug
      write(*,*) 'Debugging mode!'
#endif
      if(PRESENT(init_skel)) then
         in_skel=init_skel
      else
         in_skel=.false.
      endif
!      if(from_perm) then
!      else
!         if(.not.(PRESENT(fr_perm))) then
!            from_perm=.false.
!         else
!            from_perm=fr_perm
!         endif
!      endif
!c$$$      if(from_perm) then
!c$$$         if(this_group.ge.1) then
!c$$$            g_nr=this_group
!c$$$         else
!c$$$            if(.not.(PRESENT(group_nr))) then
!c$$$               g_nr=1
!c$$$               if(group_silent.ge.0) then
!c$$$                  write(*,*) 'Using group number 1!'
!c$$$               endif
!c$$$            else
!c$$$               g_nr=group_nr
!c$$$               if(group_silent.ge.0) then
!c$$$                  write(*,*) 'Using group number ',g_nr,'!'
!c$$$               endif
!c$$$            endif
!c$$$         endif
!c$$$         call init_g_from_perm(N,g_nr)
!c$$$         return
!c$$$      endif

      ! If we are already inited      
      if(((grp_inited).and.(Level.eq.N)).and.(.not.(cycloidal))) then
         if(group_silent.gt.0) then
            write(*,*) 'not initing group again!'
         endif
         return
      elseif(grp_inited) then
         if(group_silent.ge.0) then
            write(*,*) 'old Level(Index)=',Level
         endif
      endif

      Level=N
     !! standard settings
      num_reps=1
      num_cusps=1
      num_vertex=1
      num_norm_cusp_combs=1      
      real_num_cosets=1
      real_num_cusps=1
      call find_number_of_cosets(N)
      call find_number_of_cusps(N)
      num_reps=real_num_cosets
      
!c$$$      if(cycloidal.and.(from_perm)) then
!c$$$         width=real(Level,DP)
!c$$$         write(*,*) 'Width(cycl)=',width
!c$$$      endif
      if(width.eq.0) then
         width=1.0_DP
      endif
!      write(*,*) 'Width=',width
!      aelse
!         Width=1.0_DP           !real(Level,DP)
!      endif

       !! This is to be computed!
!      num_cusps=real_num_cusps

      if(real_num_cosets.eq.0) then
         write(*,*) 'ERROR: No cosets for N=',N,'!!'
         stop
      endif
      ! I overshoot the allocation of all these things
      ! in case my algorithm screws up and result too many 
      ! cusps or something, I don't want to mess everything up

      if(allocated(cusps)) deallocate(cusps)
      if(allocated(cusp_width)) deallocate(cusp_width)
      if(allocated(vertices)) deallocate(vertices)
      if(allocated(W)) deallocate(W)
      if(allocated(sigma_j)) deallocate(sigma_j)
      if(allocated(sigma_j_in_normalizer))                              &
     &        deallocate(sigma_j_in_normalizer)
      if(allocated(reps)) deallocate(reps)
      if(allocated(Vj)) deallocate(Vj)
      if(allocated(Uw)) deallocate(Uw)
      if(allocated(cusp_fixer)) deallocate(cusp_fixer)
      if(allocated(norm_cusp_combs)) deallocate(norm_cusp_combs)
      if(allocated(all_cusps_info)) deallocate(all_cusps_info)

      allocate(cusps(1:real_num_cosets,1:2))
      allocate(cusp_width(1:real_num_cusps))
      ! the last subscript=0  indicates which cusp 
      ! the vertex is associated to 
      allocate(vertices(1:real_num_cosets,0:2))
      allocate(W(1:real_num_cosets,2,2))
      allocate(sigma_j(1:real_num_cusps,2,2))
      allocate(cusp_fixer(1:real_num_cusps,2,2))
      allocate(sigma_j_in_normalizer(1:real_num_cusps))
      allocate(reps(1:real_num_cosets,2,2))
      allocate(Vj(1:real_num_cosets,2,2))
      allocate(Uw(1:real_num_cosets,2,2))
      allocate(all_cusps_info(1,Level))



!Q      if(allocated(mapS)) deallocate(mapS,mapT,mapTi,mapId)
!Q      allocate(mapS(1:2,1:2),mapT(1:2,1:2),mapTi(1:2,1:2),               & 
!Q     &     mapId(1:2,1:2))
!      deallocate(mapT)
!      allocate(mapT(-1:1,-1:1))
!      write(*,*) 'size(mapT)=',size(mapT,1),':',size(mapT,2)
!      write(*,*) 'mapT1=[',lbound(mapT,1),':',ubound(mapT,1),']'
!      write(*,*) 'mapT2=[',lbound(mapT,2),':',ubound(mapT,2),']'
!      write(*,*) 'size(mapS)=',size(mapS,1),':',size(mapS,2)
      mapS(1,:)=(/0,-1/)
      mapS(2,:)=(/1,0/)      
      mapT(1,:)=(/1,1/)
      mapT(2,:)=(/0,1/)
      mapTi(1,:)=(/1,-1/)
      mapTi(2,:)=(/0,1/)
      mapId(1,:)=(/1,0/)
      mapId(2,:)=(/0,1/)

      if(cycloidal) then
         num_cusps=1
         num_vertex=1
      else
         if(Level.eq.1) then
            num_cusps=1
            num_vertex=1
         else
            num_cusps=2         ! the above mentioned
            num_vertex=2
         endif
      endif
      cusps(1,:) = (/1,0/)      !=oo = 1/0
      vertices(1,:)=(/1,1,0/)
      W(1,1,:)=(/1.0_DP,0.0_DP/)
      W(1,2,:)=(/0.0_DP,1.0_DP/)
      Uw(1,:,:)=mapId
      sigma_j(1,:,:)=W(1,:,:)
      sigma_j_in_normalizer(1)=1 ! these two are normalizers
      Vj(1,:,:)=mapId

      cusp_width(1)=width
 
!      write(*,*) 'Width=',width
!      if(cusp_width(1).ne.1.0_DP) then
!         sigma_j(1,1,1:2)=(/sqrt(cusp_width(1)),0.0_DP /)
!         sigma_j(1,2,1:2)=(/0.0_DP,1.0_DP/sqrt(cusp_width(1)) /)
!      endif
!      width=1.0_DP  !!!  The width is handled by the scaling matrices
      if(num_cusps.gt.1) then
         cusps(2,:) = (/0,1/)   !=0  = 0/1
         vertices(2,:)=(/2,0,1/)! we associate a cusp to each vertex
         W(2,1,:)=(/0.0_DP,-1.0_DP/)
         W(2,2,:)=(/real(Level,DP),0.0_DP/)
         Uw(2,:,:)=mapId(:,:)
         sigma_j(2,:,:)=W(2,:,:) 
         sigma_j_in_normalizer(2)=1
         Vj(2,:,:)=mapS(:,:)
      endif
      if(Width.eq.1.0) then
         Y0_min=sqrt(3.0_DP)/real(2*Level,DP)
      else
         Y0_min=sqrt(3.0_DP)/real(2*Level,DP)/Width
      endif
!      write(*,*) 'Y0_min=',Y0_min
      if(cycloidal) then
         cusp_width(1)=width
!         cusp_width(1)=1.0_DP
      endif
      grp_inited=.true.
      if(.not.(in_skel)) then 
         call Get_cosets(N)
#ifdef debug
         call show_cosets()
#endif
         call get_vertices()
#ifdef debug
         call show_cusps()
#endif
         call find_cusp_fixators()
#ifdef debug
         call show_cusps_maps()
#endif
      endif
!      call Get_cusp_combinations()
      if(cycloidal) then
         call get_generators()
      endif
      call get_e2()
      call get_e3()
      call get_genus_congruence()
      call get_cusp_widths()        
!! If we have small level it not necessary to store this temporary data
      if(N.gt.100) then 
         call store_stuff()
      endif
      end subroutine init_g




 
      !! Dellocating arrays
      subroutine clean_group()
      if(allocated(W)) deallocate(W)
      if(allocated(sigma_j)) deallocate(sigma_j)
      if(allocated(sigma_j_in_normalizer))                              &
     &     deallocate(sigma_j_in_normalizer)
      if(allocated(norm_cusp_combs)) deallocate(norm_cusp_combs)
      if(allocated(Uw)) deallocate(Uw)
      if(allocated(cusp_fixer)) deallocate(cusp_fixer)
      if(allocated(reps)) deallocate(reps)
      if(allocated(generators)) deallocate(generators)
      if(allocated(cusps)) deallocate(cusps)
      if(allocated(cusp_width)) deallocate(cusp_width)
      if(allocated(vertices)) deallocate(vertices)
      if(allocated(Vj)) deallocate(Vj)
      if(allocated(genus)) deallocate(genus)
      if(allocated(e2)) deallocate(e2)
      if(allocated(e3)) deallocate(e3)
      if(allocated(all_cusps_info)) deallocate(all_cusps_info)
      grp_inited=.false.
      end subroutine clean_group

    

      ! Use the fact that we need only to construct a sequence
      ! of matrices in PSL(2,Z) inequivalent over Gamma_0(N)
      ! and the number of matrices is finite and known
      ! 
      subroutine Get_cosets(N)
      implicit none
      integer :: i,j,k,l,N,new_code
!      integer,dimension(1000,2,2):: reps
!      integer :: num_reps
      integer,dimension(3,2,2):: PSL2Zgens
      ! :: PSL2Zgens_str
      integer,dimension(2,2)::Mi,Mj,Mj_i,test0,test1,New,New1
      logical :: is_new
!      character(10)::str
!      character(30)::GENSTR,tmpstr
      if(.not.(grp_inited)) then
         call init_g(N)
      endif
      !! For prime numbers all coset-reps are known explicitly
      !! and we choose them to make a symmetric fundamental domain
      
      reps(1,:,:)=mapId
      num_reps=1
      if(N.eq.1) then
         return
      endif
      if(cycloidal) then
         if(MOD(real_num_cosets,2).ne.0) then
            do k=-(real_num_cosets-1)/2,(real_num_cosets-1)/2
!               reps(k+(real_num_cosets-1)/2+2,1,:)=(/1,k/)
!               reps(k+(real_num_cosets-1)/2+2,2,:)=(/0,1/)
               reps(k+(real_num_cosets-1)/2+1,1,:)=(/1,k/)
               reps(k+(real_num_cosets-1)/2+1,2,:)=(/0,1/)

!               reps(k,2,:)=(/0,1/)
!               reps(real_num_cosets+2-k,1,:)=(/1,1-k/)
                                !               reps(real_num_cosets+2-k,2,:)=(/0,1/)
            enddo
         else
            do k=-(real_num_cosets-2)/2,(real_num_cosets)/2
               reps(k+(real_num_cosets-2)/2+1,1,:)=(/1,k/)
               reps(k+(real_num_cosets-2)/2+1,2,:)=(/0,1/)
        !       reps(real_num_cosets+2-k,1,:)=(/1,1-k/)
        !       reps(real_num_cosets+2-k,2,:)=(/0,1/)
            enddo
         endif
         return
      endif
!!  Note  that in any case ST^a  is equivalent to ST^b iff a-b==0 mod N
      if(is_prime(N)) then
         do k=2,real_num_cosets
            reps(k,1,:)=(/0,-1/)
            reps(k,2,:)=(/1,0/)
         enddo
         if(N.gt.2) then !! for odd primes
            do k=-(Level-1)/2,(Level-1)/2
                  j=k+(Level-1)/2+2
                  !!! This is for right coset-reps
                  !!! otherwise it would have been left cosets
                  reps(j,2,2)=k
                                !  I don't remember what this code was good for anyway....
                                !            reps_code(j)=reps_code(i)*10+new_code
            enddo
         else
            reps(3,2,2)=1
         endif
         num_reps=Level+1
         return
      elseif(mod(level,2).eq.0) then 
         do k=-Level/2,Level/2-1
            j=k+Level/2+2
            reps(j,1,:)=(/0,-1/)
            reps(j,2,:)=(/1,k/)
         enddo
         num_reps=Level+1
      else
         do k=-(Level-1)/2,(Level-1)/2
            j=k+(Level-1)/2+2
            reps(j,1,:)=(/0,-1/)
            reps(j,2,:)=(/1,k/)
         enddo            
         num_reps=Level+1
      endif
         
      PSL2Zgens(1,:,:)=mapS(:,:)
      PSL2Zgens(2,:,:)=mapT(:,:)
      PSL2Zgens(3,:,:)=mapTi(:,:)
!      write(*,*) 'PSL2Zgens=',PSL2Zgens(1,:,:)
!      write(*,*) 'PSL2Zgens=',PSL2Zgens(2,:,:)
!      write(*,*) 'PSL2Zgens=',PSL2Zgens(3,:,:)
!      reps_code(1)=0

      do j=1,100000 ! maximum number of reps I should need
!         GENSTR=''
         New=mapId
        ! write(*,*) 'num_reps=',num_reps
         do i=1,num_reps            
            is_new=.false.
            Mi=reps(i,:,:)   
          !  write(*,*)'Mi=', Mi
            do k=1,3
               if(.not.(is_new)) then
                                ! construct a new candidate 
                  is_new=.true. ! assume it is ok
                  call multiply_i(Mi,PSL2Zgens(k,:,:),test0)
                                ! check to see if it is equivalent to an earlier one
                  do l=1,num_reps
                     Mj=reps(l,:,:)
                     call invert_sl2z(Mj,Mj_i)
                     call multiply_i(test0,Mj_i,test1)
                     if(mod(test1(2,1),N).eq.0) then
                        ! is equivalent to Mj under Gamma_0(N)
                        is_new=.false.
                     else
                        New=PSL2Zgens(k,:,:)
                        new_code=k
                     endif
                  enddo
               endif !not (is_new)
            enddo !k

            if(is_new) then
           !    write(*,*) 'is_new'
               exit
            else 
           !    write(*,*) 'nej',is_new
            endif
         enddo
         if(is_new) then
            call multiply_i(Mi,New,New1)
            !! remember the best cusps are of the form 1/N
            !! hence, if possible I want to find those
            !! representatives
           ! if(ii.eq.1) then 
           !    if(abs(New1(1,1).ne.1)) then
           !       cycle
           !    endif
           ! else
           !    if(abs(New1(1,1).eq.1)) then
           !       cycle
           !    endif
           ! endif
            reps(num_reps+1,:,:)=New1
!            reps_code(num_reps+1)=reps_code(i)*10+new_code
            num_reps=num_reps+1

!            write(*,*) reps_code(num_reps)

         else 
            exit  !if we can't find any more reps
         endif
      enddo    
      end subroutine Get_cosets

      subroutine show_cosets()
      implicit none
      integer j
      
      write(*,*) 'Coset Representatives:'
      write(*,*) 'num_reps=',num_reps
      do j=1,num_reps
         write(*,*) '-----------'
         write(*,170)j, reps(j,1,:)
         write(*,171) reps(j,2,:)
      enddo
 170  FORMAT(I3,2X,'[',I4,',',I4,']')
 171  FORMAT(5X,'[',I4,',',I4,']')


      end subroutine show_cosets

      ! Assuming we have all coset reps now we turn to vertices and cusps!
      ! NOTE: If we just wanted to know vertices and cusps there is quicker algs.
      !       but now we want identifying maps also
      subroutine get_vertices()
      implicit none
      integer::a,b,c,d
      real(DP)::ra,rc,rb,rd,vertex,tmp
      real(kind=DP),dimension(2,2)::Amap
      integer ::i,j,k,aa,bb,cc,sgd,m0,n0,t1,t2,c1,c2,m1,n1               
      integer,dimension(2,2)::Vjtmp,Tk,Utmp,Atest,Vi_inv
      integer :: t,tt,ii
      logical::is_new_vertex,is_new_cusp,test,got_norm,is_in_norm
      logical::try_best_cusp
      try_best_cusp=.true.  !!! We first try to get cusps of the form u/v with v|N, (u,v)=1

      do ii=1,2
      cusp_loop: do j=1,size(reps,1)   !num_reps
         a=reps(j,1,1)
         b=reps(j,1,2)
         c=reps(j,2,1)
         d=reps(j,2,2)
         ra=real(a,kind=DP)
         rb=real(b,kind=DP)
         rc=real(c,kind=DP)
         rd=real(d,kind=DP)
         !! first we look through all vertices of the form 1/N so we are able
         !! to chose cusp representatives from that set

         is_new_vertex=.false.
         !!! cycloidal subgroups should be no problem
         if(a*c.ne.0) then
         !   write(*,*) 'A=',a
            t=reps(j,1,1)*Level/reps(j,2,1)
#ifdef debug
            write(*,*) 't=',reps(j,1,1),'/',reps(j,2,1),'*',Level
            write(*,*) 't=',t
#endif
         !  
         !   if((ii.eq.1).and.(mod(Index,t).ne.0)) then
            if(ii.eq.1) then 
               if(abs(a).ne.1) then
          !        write(*,*) 'disregard ',a,'/',c
                  cycle cusp_loop
               else
          !        write(*,*) 'cusp is ok! ',a,'/',c
               endif
            else
               if(abs(a).eq.1) then
          !        write(*,*) 'disregard2 ',a,'/',c
                  cycle cusp_loop
               else
          !        write(*,*) 'cusp is ok2! ',a,'/',c
               endif
            endif
            vertex = ra/rc ! this is the location of the vertex, e.g. where
                                ! infinity is mapped by the coset rep            
#ifdef debug
            write(*,*) 'vertex=',a,'/',c
            write(*,*) 'vertex=',vertex
#endif
            ! Want to have -a/-c = a/c, e.g. neglect double -
            if((a.lt.0).and.(vertex.gt.0)) then
               a=-a
               c=-c
            endif
            ! I also want to have the -sign if any at a
            ! instead of c, e.g. a/-c = -a/c
            if((c.lt.0).and.(a.gt.0)) then
               c=-c
               a=-a
            endif            
            is_new_vertex=.true.
                         ! The first check is that it is not
                         ! the same as another existing cusp or vertex
            do k=3,num_vertex           
                                !               write(*,*) 'cusp=',cusp,' old cusp=',                     &
!     &           real(vertices(k,1),DP)/real(vertices(k,2),DP) 
               if(abs(vertex-real(vertices(k,1),DP)/                       &
     &                   real(vertices(k,2),DP)).lt.EPS0)                   &
     &              then
                  is_new_cusp=.false.
                  is_new_vertex=.false.
               endif
            enddo
                                ! then we have to check if it is equivalent 
                                ! to an existing cusp over Gamma0(N)
                                ! which is checked using Theorem 2 in "Ford polygons..."
            
                                ! note that for square free N it is enough that
                                ! |c1|=|c2|            
           !!! We always prefer to take representatives of the form a/c with (a,c)=1 and c|N
           !!! But if we don't get the correct number at the end we repeat without
           !!!  this condition
            if(try_best_cusp) then 
               call gcd(a,c,sgd,m1,n1)
               if((MOD(Level,c).ne.0).or.(sgd.ne.1)) then
                  cycle cusp_loop
               endif
            endif
            if(is_new_vertex) then               
               is_new_cusp=.true.  ! assume that the new vertex  (not in list) is a new cusp
                                ! we have a new vertex (not in the list at least)
               num_vertex=num_vertex+1
               vertices(num_vertex,1)=a
               vertices(num_vertex,2)=c
               ! Vj associates the vertices with the coset representative
               Vj(num_vertex,1,:)=(/a,b/)   ! this actually is the coset rep for the current vertex
               Vj(num_vertex,2,:)=(/c,d/) 

               vertices(num_vertex,0)=num_vertex ! We simply assume it is equivalent to itself
               ! we will now see if it is equivalent to a previous cusp
               ! otherwise it is a new cusp
               ! first check the two original cusps


               if(a.eq.0) then ! is the cusp at 0
                  Uw(num_vertex,:,:)=mapId
                  vertices(num_vertex,0)=1
                  is_new_cusp=.false.                 
               elseif(c.eq.0) then ! or at oo
                  Uw(num_vertex,:,:)=mapId
                  vertices(num_vertex,0)=0
                  is_new_cusp=.false.                 
               else               
                  call check_equiv_with_0(a,c,Level,Atest,test)
                  if(test) then
!                     write(*,*) 'cusp ',a,'/',c,'is equiv to 0 by'
!                     write(*,*) Atest(1,:)
!                     write(*,*) Atest(2,:)
                     is_new_cusp=.false.
                     vertices(num_vertex,0)=1
                     Uw(num_vertex,:,:)=Atest(:,:)
                  else
                     do i=3,num_cusps
                        ! We have to make sure that the we have a nice vertex
                        ! test if N*a=q*c, q in Z
                        ! If it is then we have the simple cusps, e.g. tj/N with (tj,N)>1
                        if((mod(index*a,c).eq.0).and.                     &
     &                     (mod(Level*cusps(i,1),cusps(i,2)).eq.0)) then
                           t1=a*Level/c
                           t2=cusps(i,1)*Level/cusps(i,2)
                           call get_map_between_vertex(t1,t2,Level,           &
     &                          Atest)
                           if((Atest(1,1)*Atest(2,2)-Atest(2,1)*              &
     &                          Atest(1,2)).ne.0) then
                              is_new_cusp=.false.
                              vertices(num_vertex,0)=i                 
                              Uw(num_vertex,:,:)=Atest(:,:)
                           endif
                        else
!                           call gcd(a,c,sgd1,m1,n1)
!                           call gcd(cusps(i,1),cusps(i,2),sgd2,m1,n1)
!                           if((sgd1.eq.1).and.(sgd2.eq.1)
                           ! want to solve a/c=tj/(cj*N)
                           call gcd(a*Level,c,sgd,m1,n1)
                           t1=a*Level/sgd
                           c1=c/sgd
                           call gcd(cusps(i,1)*Level,cusps(i,2),sgd,m1,  &
     &                          n1)
                           t2=cusps(i,1)*Level/sgd
                           c2=cusps(i,2)/sgd
                           call test_eq2(t1,c1,t2,c2,Level,Atest)
                           if((Atest(1,1)*Atest(2,2)-Atest(2,1)*              &
     &                          Atest(1,2)).ne.0) then
                              is_new_cusp=.false. 
                              vertices(num_vertex,0)=i                                 
                              Uw(num_vertex,:,:)=Atest(:,:)
                           endif                           
                        endif
                        if(is_new_cusp) then
                           t1=a
                           c1=c
                           t2=cusps(i,1)
                           c2=cusps(i,2)
                           ! We only do Brute force for the 
                           ! pairs of cusps that we can't
                           ! get definitive answer for by theory
                           if((mod(Level,c1).ne.0)) then
!                              write(*,*)'bf_test(',t1,c1,t2,c2,Level,'('
                              call bf_test(t1,c1,t2,c2,Level,Atest,          &
     &                             1000)
                              if((Atest(1,1)*Atest(2,2)-Atest(2,1)*              &
     &                             Atest(1,2)).ne.0) then
                                 is_new_cusp=.false. 
                                 vertices(num_vertex,0)=i                                 
                                 Uw(num_vertex,:,:)=Atest(:,:)
                              endif                   
                           endif
                        endif
!                        if(is_new_cusp.eq.(.false.)) then
                        if(.not.(is_new_cusp)) then
                           vertices(num_vertex,0)=i ! It was equivalent to cusp number i
                           exit
                        endif
                     enddo
                  endif
               endif
            endif
            
            if(is_new_cusp) then     
        !!! we prefer to have a cusp of the form t/N with t|N
        !!! so if we can, we take the cusp in that form
        !!! i.e. in this case we have u/v with u=1 and v|N.
!               if((abs(a).ne.1.0).and.(ii.eq.1)) then 
!                  cycle cusp_loop
!               endif
#ifdef debug
               write(*,*) 'New cusp ',a,'/',c
#endif
               num_cusps=num_cusps+1
               cusps(num_cusps,1)=a
               cusps(num_cusps,2)=c
               !! If the cusp is of the form u/v, v|N, (u,v)=1 then I know the width
               !! and it is N/sgd(N,v^2) 
               call gcd(a,c,sgd,m1,n1)
               if((MOD(Level,c).eq.0).and.(sgd.eq.1)) then
                  call gcd(Level,c*c,sgd,m1,n1)
                  cusp_width(num_cusps)=real(Level,DP)/real(sgd,DP)
               else
                  write(*,*) 'ERROR:cusp is not written in correct form'   &
     &                 ,' we have u/v=',a,'/',c
                  !! Here we have no formula for the cusp width but we still know that 
                  !! the width is the number of vertices equivalent to this one
                  !! and that calculaiton can be made later
                  cusp_width(num_cusps)=0.0D0
                  stop
               endif
               
               ! and this vertex was a cusp, e.g. cusp no. num_cusps
               vertices(num_vertex,0)=num_cusps
               Uw(num_vertex,:,:)=mapId

             ! Use the Fricke involutions as normalizers
             ! H_t(N)=[[t*a,b],[N,t]] 
             ! with t^2*a-N*b=t  for t | N
             ! we loop through all divisors of N to see if we find someone useful
             ! for this cusp
             ! (since a general formula might be problematic for N containing squares
               is_in_norm=.false.
               call find_cusp_norm(a,c,Level,Amap,is_in_norm)
               call gl_inverse(Amap,sigma_j(num_cusps,:,:))
!               write(*,*) 'A',j,'='
!               call print_mat(Amap,2)
!               write(*,*) 'Normalizer of u/v=',a,'/',c
!               call print_mat(sigma_j(num_cusps,:,:))
!                  write(*,531) sigma_j(num_cusps,1,:),                  &
!     &                 sigma_j(num_cusps,2,:)


               !! We better make sure it really is a cusp normalizer
               !! sigma_j(00)=a/c
               if(abs(sigma_j(num_cusps,1,1)/sigma_j(num_cusps,2,1)-      &
     &              ra/rc).gt.1.0E-15_DP) then
                  call print_mat(sigma_j(num_cusps,:,:))
!                  write(*,*) 'sigma(1,1)/sigma(2,1)=',                   &
!     &                 sigma_j(num_cusps,1,1)/sigma_j(num_cusps,2,1)
!                  write(*,*) 'ra/rc=',ra/rc
!                  write(*,531) sigma_j(num_cusps,1,:),                  &
!     &                 sigma_j(num_cusps,2,:)
                  write(*,*) 'does not normalize u/v=',a,'/',c
                  stop
               endif
               if(is_in_norm) then
                  sigma_j_in_normalizer(num_cusps)=1
               else
                  sigma_j_in_normalizer(num_cusps)=0
               endif
               got_norm=.true.
            endif
         endif
      enddo cusp_loop
      if(num_cusps.eq.real_num_cusps) then
         exit
      else
         write(*,*) 'Could not get desired form for all cusps!'
         write(*,*) 'Try to get any form of representatives!'
         try_best_cusp=.false.
      endif
      enddo
      !!! Set all cusp widths too
      do j=1,num_cusps
         a=cusps(j,1)
         c=cusps(j,2)
         call gcd(Level,c*c,sgd,m1,n1)
         all_cusps_info(1,j)=Level/sgd
      enddo

 531  FORMAT(10X,' sigma_j  ',23X,'[[',F7.3,F7.3,']','[',F7.3,                      &
     &     F7.3,']]')         
      end subroutine get_vertices

      subroutine show_cusps()
      implicit none
      integer :: i,j
      character*20::TMPSTR
      write(*,*) 'Number of vertices:',num_vertex
      write(*,*) 'Number of inequivalent cusps:',num_cusps
      do i=1,num_cusps
         write(*,296) cusps(i,1),cusps(i,2)
         do j=1,num_vertex
            if(vertices(j,0).eq.i) then
               write(*,297) vertices(j,1),vertices(j,2)
            endif
         enddo
      enddo
 296  FORMAT(2X,'[',I4,'/',I4,']')     
 297  FORMAT(10X,'[',I4,'/',I4,']')     
      end subroutine show_cusps


      subroutine show_cusps_maps()
      implicit none
      integer :: i,j
      character*20::TMPSTR      
      write(*,*) 'Number of vertices:',num_vertex
      write(*,*) 'Number of inequivalent cusps:',num_cusps
      write(*,*) 'Numbers of sigma_j involutions=',                             &
     &     count(sigma_j_in_normalizer(1:real_num_cusps)==1)
      do i=1,num_cusps
         write(*,296) cusps(i,1),cusps(i,2)
         do j=1,num_vertex
            if(vertices(j,0).eq.i) then
          !     write(*,*) 'j,i=',j,i
               write(*,297) vertices(j,1),vertices(j,2),                   &
     &              Uw(j,1,1),Uw(j,1,2),Uw(j,2,1),Uw(j,2,2)
!,               &
!     &              Uw(j,1,1)*Uw(j,2,2)-Uw(j,2,1)*Uw(j,1,2)               
            endif
         enddo
         if(sigma_j_in_normalizer(i).eq.1) then
            write(*,299) sigma_j(i,1,1),sigma_j(i,1,2),sigma_j(i,2,1),     &
     &           sigma_j(i,2,2)
         else
            write(*,298) sigma_j(i,1,1),sigma_j(i,1,2),sigma_j(i,2,1),     &
     &           sigma_j(i,2,2)
         endif
         write(*,300) cusp_fixer(i,1,1),cusp_fixer(i,1,2),              &
     &        cusp_fixer(i,2,1),cusp_fixer(i,2,2)
      enddo
 296  FORMAT(2X,'[',I4,'/',I4,']')     
 297  FORMAT(10X,'[',I4,'/',I4,']',10X,'[[',I6,I6,']','[',I6,I6,']]')     
 298  FORMAT(10X,'sigma_j  ',23X,'[[',F8.3,F8.3,']','[',F8.3,                      &
     &     F8.3,']]')     
 299  FORMAT(10X,' sigma_j (involution)',10X,'[[',F8.3,F8.3,']','[',          &
     &     F8.3,F8.3,']]')     
! 300  FORMAT(10X,'S_j  ',23X,'[[',I4,I4,']','[',I4,I4,']]')     
 300  FORMAT(10X,'S_j  ',23X,'[[',F8.3,F8.3,']','[',F8.3,F8.3,']]')     

! 298  FORMAT(10X,'[',I4,'/',I4,']')     
      end subroutine show_cusps_maps

      subroutine show_cusps_norm()
      implicit none
      integer :: i,j
      character*20::TMPSTR      
!      write(*,*) 'Number of vertices:',num_vertex
      write(*,*) 'Number of cusps:',num_cusps
      write(*,*) 'Numbers of sigma_j who are involutions=',                             &
     &     count(sigma_j_in_normalizer(1:real_num_cusps)==1)-1
      do i=1,num_cusps
         write(*,296) cusps(i,1),cusps(i,2)
         if(sigma_j_in_normalizer(i).eq.1.and.i.gt.1) then
            write(*,299) sigma_j(i,1,1),sigma_j(i,1,2),sigma_j(i,2,1),     &
     &           sigma_j(i,2,2)
         else
            write(*,298) sigma_j(i,1,1),sigma_j(i,1,2),sigma_j(i,2,1),     &
     &           sigma_j(i,2,2)
         endif
!         write(*,300) cusp_fixer(i,1,1),cusp_fixer(i,1,2),              &
!     &        cusp_fixer(i,2,1),cusp_fixer(i,2,2)
      enddo
 296  FORMAT(2X,'[',I4,' /',I4,']')     
 297  FORMAT(10X,'[',I4,'/',I4,']',10X,'[[',I6,I6,']','[',I6,I6,']]')     
 298  FORMAT(10X,'sigma_j  ',23X,'[[',F8.3,F8.3,']','[',F8.3,                      &
     &     F8.3,']]')     
 299  FORMAT(10X,' sigma_j (involution)',11X,'[[',F8.3,F8.3,']','[',          &
     &     F8.3,F8.3,']]')     
! 300  FORMAT(10X,'S_j  ',23X,'[[',I4,I4,']','[',I4,I4,']]')     
 300  FORMAT(10X,'S_j  ',23X,'[[',F10.4,F10.4,']','[',F10.4,F10.4,']]')     

! 298  FORMAT(10X,'[',I4,'/',I4,']')     
      end subroutine show_cusps_norm



      subroutine show_gens()
      implicit none
      integer :: i,j
      character*20::TMPSTR
      write(*,*) 'Number of generators:',num_gens
      do i=1,num_gens
         write(*,685)  i,generators(i,1,1),generators(i,1,2),                    &
     &        generators(i,2,1),generators(i,2,2)
      enddo

! 685  FORMAT(5X,'E(',I2,'):',5X,'[[',I6,I6,']','[',I6,I6,']]')     
 685  FORMAT(5X,'E(',I2,'):',5X,'[[',F8.3,F8.3,']','[',F8.3,F8.3,']]')     


      end subroutine show_gens


      !!!  Check if the cusp a/c is equivalent with 0
      subroutine check_equiv_with_0(a,c,N,U,test)
      implicit none
      integer:: a,c,N
      integer::delta,gamma,sgd
      integer,dimension(2,2)::U
      logical::test
      call gcd(N*a,c,sgd,gamma,delta)
      if(sgd.eq.1) then
         U(1,:)=(/c,-a/)
         U(2,:)=(/N*gamma,delta/)
         test=.true.
      else
         U=mapId
         test=.false.
      endif
      end subroutine check_equiv_with_0

        ! what we want to find is a map Tz=(az+b)/(Ncz+d)
      ! with a*d-N*b*c=1 and T(t1/N)=t2/N
      ! t1/N and t2/N are cusps

      ! For a description of what I am doing, see
      ! p.5-6 in A. Lascurain Orive "Ford polygons for..."

      subroutine get_map_between_vertex(t1,t2,N,AA)
      implicit none
      integer:: t1,t2,N
      integer,dimension(2,2)::AA
      integer:: a,b,c,ai
      integer::d,d1,d2,m0,n0,q,q1,q2,qN
      integer :: i,test,u,m1,n1,k,w
      AA(:,:)=0
      call gcd(t1,N,d1,m0,n0)
      call gcd(t2,N,d2,m0,n0)
!      write(*,*) 'd1,d2=',d1,d2
      if(d1.eq.d2) then
         q1=t1/d1
         q2=t2/d1
         qN=N/d1
!         write(*,*) 'q1,q2,qn=',q1,q2,qN
         call gcd(d1,qN,d,m0,n0)
!         write(*,*) 'mod(',q1,'-',q2,',',d,')=',mod(q1-q2,d)
         if(mod(q1-q2,d).eq.0) then              
            call gcd(q1*q2*d1,qN,w,m1,n1)            
            u=(q2-q1)/w
            if(m1.gt.0) then
               k=u*m1
               AA(1,:)=(/k*t2+1,u*n1 /)
               AA(2,:)=(/k*N,-(k*t1-1) /)
            else
               k=-u*m1
               AA(1,:)=(/k*t2-1,-u*n1 /)
               AA(2,:)=(/k*N,-(k*t1+1) /)
            endif
 !           write(*,*) 't1,t2=',t1,t2,' A='
!            write(*,*) AA(1,:)
!            write(*,*) AA(2,:)
         endif
      endif
      end subroutine get_map_between_vertex

      !!! Test is t1/c1  are equivalent with t2/c2 modulo Gamma_0(N)
      !!!  if they are equivalent then AA gives the equivalence map 
      subroutine test_eq2(t1,c1,t2,c2,N,AA)
      use numtheory
      implicit none
      integer::t1,t2,c1,c2,N
      integer::sgd1,sgd2,m1,n1,q1,q2,qN,w,wp
      integer::i,l,k,q,a,b,c,d,a0,d0,bc,test
      integer :: num,den
      integer,dimension(2,2)::AA
      

      AA(1,:)=(/0,0/)
      AA(2,:)=(/0,0/)
      call gcd(N,t1,sgd1,m1,n1)
      call gcd(N,t2,sgd2,m1,n1)
      if(sgd1.ne.sgd2) then
!         write(*,*) 'Not equivalent!'
         return
      else
!         write(*,*) 'sgd=',sgd1,sgd2
      endif
      q1=t1/sgd1
      q2=t2/sgd1
      qN=N/sgd1
!      write(*,*) 'q1=',q1,' q2=',q2,' qN=',qN
      call gcd(sgd1*q1*q2,c1*c2*qN,w,m1,n1)
      call gcd(q1*c2,q2*c1,wp,a0,d0)
!      write(*,*) 'w=',w,' wp=',wp
!      write(*,*) 'a0=',a0,' d0=',d0
      if( mod(w,wp).ne.0) then
 !        write(*,*) 'Not equivalent, w != l*wp'
         return
      endif
      l=w/wp
!      write(*,*) 'l=',l
      do k=-1000,1000
!         do q=-500,500
         do q=-sgd1,sgd1
            a=k*l*(a0)+q*((q2*c1)/wp)
            d=k*l*(d0)-q*((q1*c2)/wp)
!            write(*,*) 'a,d=',a,d,' q=',q,'kl=',k*l
            if((mod(a*d,N).eq.1).or.(modulo(a*d,N).eq.1)) then
         !      write(*,*) 'a,d=',a,d,' N=',N
!              write(*,*) '(ad-1)=',a*d-1
               bc=(a*d-1)/N               
          !     write(*,*) ' bc=',bc
               do i=-abs(bc),abs(bc)
                  b=i
                  if((b.ne.0).and.(mod(bc,b).eq.0)) then
                     c=bc/b
      !               write(*,*) 'b,c=',b,c
                     num=(a*t1+b*c1*N)
                     den=(c*t1+d*c1)
                     test=num*c2-den*t2
                     
                     !test=c*sgd1*q1*q2-b*c1*c2*qN
                     if(test.eq.0) then
!                        write(*,*) 'Found equivalence!'
!                        write(*,*) 'k=',k,' q=',q
                        AA(1,:)=(/a,b/)
                        AA(2,:)=(/N*c,d/)
                        return
                     endif
                  endif
               enddo
            endif
         enddo
      enddo
      end subroutine test_eq2



      ! A routine that stores everything about the group 
      ! in a file: reps/cusps/vertices/maps
      subroutine store_stuff()
      implicit none
      character*20::FILENAME
!      if(from_perm) then
!         return
!      endif
      if(real_num_cosets.ne.num_reps) then
         write(*,*) 'This DATA is not complete!'
         write(*,*) 'NOTHING IS STORED!'
         write(*,*) 'real_num_cosets=',real_num_cosets
         write(*,*) 'num_reps=',num_reps
         return
      endif
      if(real_num_cusps.ne.num_cusps) then
         write(*,*) 'This DATA is not complete!'
         write(*,*) 'NOTHING IS STORED!'
         write(*,*) 'real_num_cusps=',real_num_cusps
         write(*,*) 'num_cusps=',num_cusps
         return
      endif

      FILENAME = '.dataN'//trim(integer_to_char(Level))//'.dat'
      open(unit=12,file=FILENAME)
      write(12,*) Level
      write(12,*) real_num_cosets
      write(12,*) real_num_cusps

      write(12,*) reps(:,:,:)
!      write(*,*) reps(:,:,:)
      write(12,*) num_vertex
      write(12,*) vertices(1:real_num_cosets,:)

      ! Uw is the maps from vertex to cusp
      write(12,*) Uw(1:real_num_cosets,:,:)
      write(12,*) num_cusps
      write(12,*) cusps(1:real_num_cusps,:)
      
      write(12,*) sigma_j(1:real_num_cusps,:,:)
      write(12,*) sigma_j_in_normalizer(:)
!      write(*,*) sigma_j_is_normalizer(:)
      write(12,*) Vj(1:real_num_cosets,:,:)
      close(12)
      write(*,*) 'Data for the Group is stored in ',FILENAME,'!'
      
      end subroutine store_stuff



      subroutine get_stuff(N)
      implicit none
      character*20::FILENAME
      integer :: test_real_num_cosets,test_real_num_cusps
      integer :: N
      integer,dimension(:,:),allocatable:: t_vertices
      integer,dimension(:,:),allocatable:: t_cusps
      real(kind=DP),dimension(:,:,:),allocatable:: t_sigma_j
      integer,dimension(:),allocatable:: t_sigma_j_in_normalizer
      integer,dimension(:,:,:),allocatable:: t_reps
      integer,dimension(:,:,:),allocatable :: t_Vj
      integer,dimension(:,:,:),allocatable :: t_Uw
      logical:: have_file
      FILENAME = 'dataN'//trim(integer_to_char(Level))//'.dat'
      inquire(FILE=FILENAME,EXIST=have_file)
      if(.not.(have_file)) then
         ! WE have to construct the group in the usual way
         write(*,*) 'No file for N=',N,' exists!'
         write(*,*) 'Group properties are now computed!'         
         Level=N
         call init_g(N)
         write(*,*) 'And is stored for future references!'
         call store_stuff()
      else
         open(unit=12,file=FILENAME)
         read(12,*) N
         write(*,*) 'RETRIEVED index=',N
         read(12,*) test_real_num_cosets
         write(*,*) 'num_cosets=',test_real_num_cosets
         read(12,*) test_real_num_cusps 
         write(*,*) 'num_cusps=',test_real_num_cusps 
         if(N.ne.Level) then
            Level=N
            call init_g(N)
         endif
         call init_g(N,.true.)
                                !      call find_number_of_cusps(Level)
         if(real_num_cusps.ne.test_real_num_cusps) then
            write(*,*) 'This DATA is corrupt!'
            write(*,*) ' It has not same number of cusps !'
            write(*,*) 'NOTHING IS RETRIEVED!'
            write(*,*) 'real_num_cusps=',real_num_cusps
            write(*,*) 'test_real_num_cusps=',test_real_num_cusps
            return
         endif
         num_reps=real_num_cosets
         
         read(12,*) reps(1:real_num_cosets,:,:)
         read(12,*) num_vertex
         read(12,*) vertices(1:real_num_cosets,:)
         read(12,*) Uw(1:real_num_cosets,:,:)
         read(12,*) num_cusps
         read(12,*) cusps(1:real_num_cusps,:)
                                ! sigma_j are cusp-normalizing maps
      ! If possible they should be normalisers of
                                ! the group also but for N with squares this doesn't work
!         read(12,*) sigma_j(0:real_num_cosets,:,:)
         read(12,*) sigma_j(1:real_num_cusps,:,:)
         read(12,*) sigma_j_in_normalizer(:)

         read(12,*) Vj(1:real_num_cosets,:,:)
         close(12)
         write(*,*) 'Stuff is gotten from  ',FILENAME,'!'
      endif
      end subroutine get_stuff

      !! Find a cuspnormalizer
      !! and if possible a normalizing cusp normalizer
      subroutine find_cusp_norm(u,v,N,Nmap,is_invol)
      use commonvariables
      implicit none
      integer :: t,aa,bb,u,v,N,u0,v0,sgd,sgd1,m0,n0
      real(kind=DP),dimension(2,2)::Nmap,Ni,C
      real(DP)::alpha,beta,gamma,delta,scaling(2,2)
      logical:: is_invol,got_norm,need_A
      real(kind=DP)::w,mm
      integer,dimension(2,2)::A
      A(1,:)=(/1,0/)
      A(2,:)=(/0,1/)
      is_invol=.false.
      need_A=.false.
!      if(from_perm) then 
!         write(*,*) ' find_cusp_norm is not to be called with perm.!'
!         stop
!      endif
!      write(*,*) 'Find norm of u/v=',u,'/',v
      ! If not v|N we have to find an equivalent
      ! cusp first of the correct form
      if(mod(N,v).ne.0) then
         ! THen we find a cusp of the appropriate form
!         write(*,*) 'u/v=',u,'/',v,' is not of correct form'
!         write(*,*) 'thus we compute a better cusp!'
         need_A=.true.
         call find_cusp_candidates(u,v,u0,v0,N,A)         
         u=u0
         v=v0
!         write(*,*)'u/v=',u,'/',v,' should be good'
      endif

      if(mod(N,v).ne.0) then
         write(*,*) 'something is wrong! '
         write(*,*) 'We could not get a good cusp!'
         write(*,*)'u/v=',u,'/',v,' should be good?'
         stop
      else
         ! First try to get a groupinvolution as normalizer
         ! E.g. the Fricke involution
         !! rewrite the map as u/v=t/N
         t=u*N/v   !!  t|N
                                ! is t a square factor in N?
                                ! else solve t^2a-N*b=t
                                ! If a*t^2-Nb=t
!         write(*,*) 'u*t^2=',u*t*t,' t=',t
         call gcd(t,N,sgd,m0,n0)
         if(sgd.eq.1) then
            write(*,*) 'Something is wrong!'
            write(*,*) 'The cusp: [',u,',',v,']'
            write(*,*) 'Can not be writen as t/N, t=',t
            stop
         endif
         !!! What we want is a map of the form
         !!!  (x*(t,N) y  // -N  t )
         !!! s.t. x*(t,N)*t+N*y=(N,t) 
         !!! and here we must have (t,N/(N,t))=1
!         call gcd(t*t/sgd,N/sgd,sgd1,m0,n0)
         call gcd(t,N/t,sgd1,m0,n0)
!     write(*,*) 'm0*t+n0*N=',m0,'*t+',n0,'*N/t=',m0*t+              &
!     &        n0*N/t
!         if((modulo(t*t,N).eq.t).or.(mod(t*t,N).eq.t)) then
         if(sgd1.eq.1) then
            if(ceiling(real(bb,DP)).ne.bb) then
               write(*,*) 'ERROR: bb is no integer!' 
               write(*,*) 'Maybe t^2|N =',t*t,'|',N 
               stop
            endif
            aa=real(-n0,DP)
            bb=real(m0*t,DP)
!            bb=real(m0,DP)/real(t,DP)
            Nmap(1,:)=(/real(bb,DP),real(-aa,DP)/)
            Nmap(2,:)=(/real(-N,DP),real(t,DP)/)
            !! remember we want sigma_j(oo)=u/v
!            write(*,*) 'Nmap='
!            call print_mat(Nmap,2)
!            call gcd(N,N/sgd,sgd1,m0,n0)            
!            mm=real(sgd1,DP)
!            scaling(1,:)=(/sqrt(mm),0.0_DP /)
!            scaling(2,:)=(/0.0_DP, 1/sqrt(mm)/)
!            Nmap=matmul(Nmap,scaling)
            got_norm=.true.
            is_invol=.true.
         elseif((modulo(-u*t*t,N).eq.t).or.(mod(-u*t*t,N).eq.t)) then
            bb=(-u*t*t-t)/N
            ! We instead change the sign of t
            if(ceiling(real(bb,DP)).ne.bb) then
               write(*,*) 'ERROR: bb is no integer!' 
               write(*,*) 'Maybe t^2|N =',t*t,'|',N 
               stop
            endif
            aa=u*t

            Nmap(1,:)=(/real(aa,DP),real(bb,DP)/)
            Nmap(2,:)=(/real(N,DP),real(-t,DP)/)           
            !! remember we want sigma_j(oo)=u/v

!            Nmap(1,:)=(/real(-t,DP),real(-bb,DP)/)
!            Nmap(2,:)=(/real(-N,DP),real(aa,DP)/)           
 !           write(*,*) 'got involution'
 !           write(*,1055) Nmap(1,1),Nmap(1,2),Nmap(2,1),Nmap(2,2)
            got_norm=.true.
            is_invol=.true.
         else                   ! This will work 
!            w=real(N,DP)/real(v,DP)
!            Nmap(1,:)=(/sqrt(w)*real(u,DP),0/)
!            Nmap(2,:)=(/real(v,DP)*sqrt(w),1.0_DP/(real(u,DP)*sqrt(w))/)

!            w=real(N,DP)*real(v,DP)
!            Nmap(1,:)=(/sqrt(w)*real(u,DP)/real(v,DP),0/)
!            Nmap(2,:)=(/sqrt(w),real(v,DP)/(real(u,DP)*sqrt(w))/)
!            write(*,*) 'u=',u
!            write(*,*) 'v=',v
            
            !!! In order to be a true cusp normalizing map
            !!! e.g. NmapSNmap^(-1)=S_j
            !!! where N_j = (alpha, beta ; gamma, delta)
            !!! and alpha*delta-beta*gamma=1
            !!! we must have the following
            !!! Level| gamma^2
            !!! v*v  | gamma^2
            !!! gamma^2 and delta^2 are integers, and 
            !!! and -adelta/gamma=u/v
            
!            Nmap(1,:)=(/1.0_DP/real(u,DP),0/)
!            Nmap(2,:)=(/-real(v,DP),real(u,DP)/)

            call gcd(v*v,Level,sgd,m0,n0)
            gamma=sqrt(real(v*v*Level/sgd,DP))
!            delta=-1.0_DP*gamma*real(u,DP)/real(v,DP)
            delta=-1.0_DP*gamma*real(u,DP)/real(v,DP)
            alpha=1.0_DP/delta
            beta=0.0_DP
!(alpha*delta-1.0_DP)/gamma
!-1.0_DP/gamma-gamma*real(u,DP)/real(v,DP)
            Nmap(1,:)=(/alpha,beta/)
            Nmap(2,:)=(/gamma,delta/)

            

            
            !!! remember N_j^-1(oo)=u/v  is what we want later
            !!! but what we find here is the inverse

 !           write(*,*) 'no involution'
 !           write(*,1055) Nmap(1,1),Nmap(1,2),Nmap(2,1),Nmap(2,2)
            got_norm=.true.     ! it is a cusp normalizer
            is_invol=.false.    ! but is not an involution

            !!! Or rather, it might be in a few special cases:
            !!! note that the lower-left corner of 
            !!!  Nmap A Nmap^-1 == -gamma^2+gamma*delta*(a-d)  mod N
            !! if A=[[a,b],[c,d]] in Gamma_0(N)          
            !!! and then if we also have that a==d==1 mod N/gamma
            !!! the map Nmap is clearly in the normalizer of Gamma_0(N)
            !!! if in addition all of its elements are integers

            !!! There are only two explicit cases considered here
            !!! spec 2|N and m=4 (e.g. 8)  or 3|N and m=3 (e.g. 9)

            !! to check this it is sufficient to see if gamma is an integer first of all
            !! then we check if N|gamma and if either N/gamma=2 or 3 we are ok. 

            !! Actually in the present state only N=2 are OK, since for N=3 we have 
            !! complex eigenvalues, e.g. Nmap^3=Id so Nmap is not an involution
            if((floor(gamma).eq.ceiling(gamma)).and.                                    &
     &           (floor(alpha).eq.ceiling(alpha)).and.                                  &
     &           (floor(beta).eq.ceiling(beta)).and.                                  &
     &           (floor(delta).eq.ceiling(delta))) then
               if(MOD(nint(gamma)*nint(gamma),Level).                         &
     &              eq.0) then
                  if((abs(Level/anint(gamma)).eq.2)) then
!     .or.                     &
!     &                 abs((Levelanint(gamma)).eq.3)) then
                     is_invol=.true.
                  endif
               endif
            endif
         endif
      endif
      ! The map that takes our cusp to oo is now a 
      ! combination of first A then Nmap^-1 
      !! The inversion is done manually earlier
      !      call invert_map_r(Nmap,Ni) 
      if(need_A) then
         call multiply_r_i(Ni,A,C)
         Nmap=C
      endif

 1055 FORMAT(10X,'[[',F9.3,F9.3,']','[',F9.3,F9.3,']]')     
      end subroutine find_cusp_norm




      ! Here we try to locate cusps of the form u/v
      ! equivalent to a/c
      ! where (u,v)=1, v|N and -1/2<=u/v<=1/2

!!!  the preferred form is 1/v = t/N with v|N

      ! Note do not cover equivalence to 0 or oo
      ! but that are other functions doing
      subroutine find_cusp_candidates(a,c,u0,v0,N,AA)
      implicit none
      integer:: m0,n0,c,N,tol,sgd,det,t_mult
      integer :: a,b,u,v,test,v_low,v_high,u0,v0
      integer,dimension(2,2)::AA
      AA(1,:)=(/0,0/)
      AA(2,:)=(/0,0/)
      
      if(mod(N,c).eq.0) then ! Then we already have a cusp of desired form
         AA(1,:)=(/1,0/)
         AA(2,:)=(/0,1/)
         u0=a
         v0=c
         return
      endif

      do t_mult=1,10
         tol=100*t_mult
         do v=1,N
            if(mod(N,v).eq.0) then !If v|N
               v_low=floor(real(v)/2.0)
               v_high=ceiling(real(v)/2.0)
               do u=-v_low,v_high
                  if((u.ne.0).and.(abs(2*u).le.v)) then
!                     write(*,*) 'testing u,v=',u,v 
                     call gcd(u,v,sgd,m0,n0)
                     if(sgd.eq.1) then ! we only want rel. prime u and v
                        call bf_test(a,c,u,v,N,AA,tol)
                        det = det_i(AA)
                        if(det.ne.0) then
                 !          write(*,*) 'done !'
                           u0=u
                           v0=v
                           if(abs(u0).eq.1) then
                              return
                           endif
                        endif
                     endif
                  endif
               enddo
            endif
         enddo
      enddo
      end subroutine find_cusp_candidates

      !!! Find the maps in the group which fixates the respectively cusps
      subroutine find_cusp_fixators
      implicit none
      real(DP),dimension(2,2)::sigma_ji,tmp_map,tmp_map2
      real(DP)::det
      complex(DP)::mul_tmp
      integer::a,b,c,d,sgd,j,m,n
                                !      cusp_fixer(1,:,:)=T(:,:)
      cusp_fixer(1,1,:)=(/1.0_DP,cusp_width(1)/)
      cusp_fixer(1,2,:)=(/0.0_DP,1.0_DP/)
      do j=2,num_cusps
         call gl_inverse(sigma_j(j,:,:),sigma_ji(:,:))
!         write(*,*) 'Sigma',j,'='
!         call print_mat(sigma_j(j,:,:),2)
!         write(*,*) 'Sigma',j,'^-1='
!         call print_mat(sigma_ji(:,:),2)
![[',sigma_j(j,1,:),'],[',sigma_j(j,2,:),']]'
!       write(*,*) 'Si',j,'^-1=[[',sigma_ji(1,:),'],[',sigma_ji(2,:),']]'
!         call multiply_i_r(T,sigma_j(j,:,:),tmp_map)
         tmp_map=MATMUL(cusp_fixer(1,:,:),sigma_ji(:,:))
         tmp_map2=MATMUL(sigma_j(j,:,:),tmp_map)
       !  write(*,*)'sigma_j*Ti*sigma_ji=[[',tmp_map2(1,:),'],[',tmp_map2(2,:),']]'
!!! Now, first make the map to have determinant 1 in case we have
!!! not already 
         det=(tmp_map2(1,1)*tmp_map2(2,2)-tmp_map2(1,2)*                  &
     &        tmp_map2(2,1))
         a=anint(tmp_map2(1,1)/sqrt(det))
         b=anint(tmp_map2(1,2)/sqrt(det))
         c=anint(tmp_map2(2,1)/sqrt(det))
         d=anint(tmp_map2(2,2)/sqrt(det))
         !! Have to make sure we are in the group
         !! i.e. divide out by common factors
!         write(*,*) 'map=[[',a,',',b,'],[,',c,',',d,']]'
!!DEB
         call gcd(c,d,sgd,m,n)
         if(sgd.ne.1) then
            if((mod(a,sgd).ne.0).or.(mod(b,sgd).ne.0)) then
               write(*,*) 'Fixator of Cusp(',j-1,')=',cusps(j,1),'/',       &
     &              cusps(j,2),' not in group!'
               write(*,*) '(a,b;c,d)=(',a,b,';',c,d,')'
               stop
            else  !! everything divisible by sgd
               a=a/sgd
               b=b/sgd
               c=c/sgd
               d=d/sgd
            endif
         endif
!         if(((mod(c,Level).ne.0).or.(a*d-b*c.ne.1)).and.                 &
!     &        (.not.(from_perm))) then
         if((mod(c,Level).ne.0).or.(a*d-b*c.ne.1)) then
            write(*,*) 'Fixator of Cusp(',j-1,')=',cusps(j,1),'/',       &
     &           cusps(j,2),' not in group!'
            write(*,*) '(a,b;c,d)=(',a,b,';',c,d,')'
            stop
         endif     
         !! If we have come so far we should be done
         cusp_fixer(j,1,1)=a
         cusp_fixer(j,1,2)=b
         cusp_fixer(j,2,1)=c
         cusp_fixer(j,2,2)=d
         !!! compute the shift at this cusp
       enddo
      end subroutine
      
      
! Brute Force to test if a cusp a1/c1 is equivalent to a2/c2
      subroutine bf_test(a1,c1,a2,c2,N,AA,tol)
      implicit none
      integer:: i,a1,c1,a2,c2,N,tol
      integer :: a,b,c,d,bc,test,a0,d0
      integer,dimension(2,2)::AA
      AA(1,:)=(/0,0/)
      AA(2,:)=(/0,0/)
      
      do i=1,4
!      do a=-tol,tol
!         do d=-tol,tol
         ! I try to achieve better figures in the matrices
         ! by starting from 0 and go upwards
         ! And to get from -tol to +tol I multiply with -1
         ! sometimes
         do a0=0,tol            
            do d0=0,tol
               if(i.eq.1) then
                  a=a0; d=d0
               elseif(i.eq.2) then
                  a=a0; d=-d0
               elseif(i.eq.3) then
                  a=-a0; d=d0
               elseif(i.eq.4) then
                  a=-a0; d=-d0
               endif

               if((mod(a*d,N).eq.1).or.(modulo(a*d,N).eq.1)) then
                  bc=(a*d-1)/N
                  do b=-abs(bc),abs(bc)
                     if(b.eq.0) then 
                        cycle
                     endif
                     if(mod(bc,b).eq.0) then
                        c=bc/b
                        test=a*a1*c2+b*c1*c2-c*a2*N*a1-d*a2*c1
                        if(test.eq.0) then
                           write(*,*) 'Equivalence!'
                           write(*,*) 'between ',a1,'/',c1,' and'
                           write(*,*) a2,'/',c2
                           AA(1,:)=(/a,b/)
                           AA(2,:)=(/N*c,d/)
                           write(*,*) 'map='
                           write(*,*) AA(1,:)
                           write(*,*) AA(2,:)
                           return
                        endif
                     endif
                  enddo
               endif
            enddo
         enddo      
      enddo
      
      end subroutine bf_test
  ! Computes the theoretical formula for the number of cosets
      ! of Gamma_0(1) mod Gamma_0(N)
      !
      subroutine find_number_of_cosets(N)
      implicit none
      integer :: N,tmp1,tmp2,p
      integer :: j
!c$$$      if(from_perm) then
!c$$$         if(index.gt.0) then
!c$$$            real_num_cosets=index
!c$$$         else
!c$$$            write(*,*) 'ERROR: find_number_of_cosets!'
!c$$$            stop
!c$$$         endif
!      elseif(cycloidal.eq.(.false.)) then
      if(.not.(cycloidal)) then
         if(is_prime(N)) then
            real_num_cosets=N+1
         else
            tmp1=1
            tmp2=1
            do p=2,N
               if(is_prime(p)) then
                  if(mod(N,p).eq.0) then
                                !     write(*,*) 'prime divisor=',p
                     tmp1=tmp1*(p+1)
                     tmp2=tmp2*p
                  endif
               endif
            enddo
            real_num_cosets=N*tmp1/tmp2
         endif
      else
!         if(is_prime(N)) then
         !! This is from the definition of the cycloidal subgroups
         !! We define them from their index = number of cosets
            real_num_cosets=N
!         else
!            write(*,*) 'So far only prime index for cycloidal!'
!            stop
!         endif
      endif
      end subroutine find_number_of_cosets
      
      !! Number of cusps for Hecke congruence subgroup
      subroutine get_number_of_cusps(N,nc)
      implicit none
      integer::N,nc
      if(real_num_cusps.eq.0) then 
         call find_number_of_cusps(N)
      endif
      nc=real_num_cusps
      end subroutine

      subroutine find_number_of_cusps(N)
      implicit none
      integer :: N,tmp
      integer :: d,sgd,a,b
      
!      if(from_perm) then
!         write(*,*) 'find_num_cusps not to be calld when perm!'
!         stop
!      endif
      if(cycloidal) then
         real_num_cusps=1
      else
         tmp=0
         do d=1,N
            if(mod(N,d).eq.0) then
               call gcd(d,N/d,sgd,a,b)
               tmp=tmp +EulerPhi(sgd)
            endif
         enddo
         real_num_cusps=tmp
      endif
      end subroutine find_number_of_cusps


      ! given a map and a number of coset representatives
      ! (assigned in a module variable)
      ! return the index of the map wrt to these reps

      integer function get_coset(mapping) 
      implicit none
      integer :: i,m_ci,m_di,r_ci,r_ai
      integer, dimension(2,2) :: mapping
      integer, dimension(2,2) :: inverted
      integer :: test
      
      ! there is always the identity rep that is not needed 
      ! to enter explicitly
      


      ! want to see if mapping^-1*coset_rep^-1 is in Gamma0(N)
      ! e.g. -c*alpha+a*delta = N*k

  !!! REMEMBER THAT THE COSET WE CHECK FOR IS FOR MAPPING^-1
!
!  Observe A\in[V_k] <=> A*V_k^(-1) \in Gamma_0(N)
!
      
!      if(from_perm) then
!         write(*,*) 'Not supposed to be used from a perm.group!'
!         stop
!      endif
      m_ci=-mapping(2,1) 
      m_di=mapping(1,1) 
      
      get_coset=-1
      if(MOD(m_ci,Level)==0) then
         get_coset=0
      else
         do i=1,num_reps
            !!! remember to look at the inverse coset-rep
            r_ai=reps(i,2,2)
            r_ci=-reps(i,2,1)
            test=m_ci*r_ai+m_di*r_ci
            if(MOD(test,Level)==0) then
               get_coset=i
            endif
         enddo
      endif
      if(get_coset.lt.0) then
         write(*,*) 'not complete reps?'
         write(*,*) 'num_reps=',num_reps
         write(*,*) 'real_num_cosets=',real_num_cosets
         write(*,*) mapping
         do i=1,num_reps
            r_ai=reps(i,1,1)
            r_ci=-reps(i,1,2)
            test=m_ci*r_ai+m_di*r_ci
            write(*,*) 'test=',m_ci,'*',r_ai,'+',m_di,'*',r_ci,'=',test
         enddo
         stop
      endif
      end function get_coset


      
  !! Vertex handling routines 
      
      ! seek_vertex: get the index of the vertex closest to x+iy
      !
      subroutine seek_vertex(x,y,ind)      
      use commonvariables
      implicit none
      real(kind=DP) :: x,y,x1,y1,ytmp
      integer :: ind,j,cusp
      real(kind=DP),dimension(2,2)::B,C
      if(num_cusps==0) then
         write(*,*) 'ERROR(check region): There are no cusps!'
         stop
      elseif(num_cusps.eq.1) then
         ! there is only one cusp
         ind=1 
         return
      endif
         
      ytmp=y
      ind=1   ! corresponds to the cusp at infinity
      ! first look through which vertex it corresponds to
      do j=2,num_vertex
         cusp=vertices(j,0)     ! This is the corresponding cusp
         call gl_inverse(sigma_j(cusp,:,:),B)
         call multiply_r_i(B,Uw(j,:,:),C)
         call apply_map_re(x,y,x1,y1,C)
         if(y1.ge.ytmp) then 
            ytmp=y1
            ind=j
         endif
      enddo
 141  FORMAT('[[',F8.3,F8.3,'],[',F8.3,F8.3,']]')
 142  FORMAT('[[',I6,I6,'],[',I6,I6,']]')
     
!     then check if that vertex is equivalent to another (cusp)
!      if(vertices(ind,0).ne.ind) then
!         ind=vertices(ind,0)
!      endif  
      end subroutine seek_vertex

      ! get_cusp_from_vertex:   get index of the cusp that is equivalent 
      ! to a given vertex
      subroutine get_cusp_from_vertex(vertex,cusp)
      use commonvariables
      implicit none
      integer,intent(in) :: vertex
      integer,intent(out) :: cusp
      if(vertices(vertex,0).ne.vertex) then
         cusp=vertices(vertex,0)
      else
         cusp=vertex
      endif
      end subroutine get_cusp_from_vertex


      function V_map(a,b)
      implicit none
      integer::a,b
      real(DP),dimension(2,2)::V_map
      V_map(1,:)=(/real(a,DP),-1.0_DP-real(a*b,DP) /)
      V_map(2,:)=(/1.0_DP,-real(b,DP)/)
      end function
      

      subroutine get_generators()
      use commonvariables
      implicit none
      real(DP)::arg
      integer::j
      !!!! This is not really a cycloidal group since it is not a subgroup of 
      !!!! PSL(2,Z) but at least it has only one cusp
      !!!! 
      if(Level.eq.1) then
         if(order.eq.0.0) then
            order=1.0_DP
            !! Order=1 gives the modular group!
         endif
         arg=PI*0.5_DP/real(order,DP)
    !     write(*,*) 'arg=',arg
         allocate(generators(-2:2,2,2))
         generators(0,:,:)=real(mapId,DP)
         generators(1,1,:)=(/1.0_DP,width/)
         generators(1,2,:)=(/0.0_DP,1.0_DP/)         
         generators(-1,1,:)=(/1.0_DP,-width/)
         generators(-1,2,:)=(/0.0_DP,1.0_DP/)         
         generators(2,1,:)=(/cos(arg),sin(arg)**2/)
         generators(2,2,:)=(/-1.0_DP,cos(arg) /)
         generators(-2,1,:)=(/cos(arg),-sin(arg)**2/)
         generators(-2,2,:)=(/1.0_DP,cos(arg) /)
         num_gens=2
      elseif(Level.eq.3) then
!! might as well allocate for the inverses too
         allocate(generators(-4:4,2,2))
         generators(0,:,:)=real(mapId,DP)
         generators(1,1,:)=(/1.0_DP,width/)
         generators(1,2,:)=(/0.0_DP,1.0_DP/)         
         generators(2,:,:)=V_map(0,0)
         generators(3,:,:)=V_map(1,1)
         generators(4,:,:)=V_map(-1,-1)         
         generators(-1,1,:)=(/1.0_DP,-width/)
         !! In this case they are the same
         generators(-1,2,:)=(/0.0_DP,1.0_DP/)         
         generators(-2,:,:)=V_map(0,0)
         generators(-3,:,:)=V_map(1,1)
         generators(-4,:,:)=V_map(-1,-1)
         num_gens=4
      elseif(Level.eq.4) then
         allocate(generators(-4:4,2,2))
         generators(0,:,:)=real(mapId,DP)
         generators(1,1,:)=(/1.0_DP,width/)
         generators(1,2,:)=(/0.0_DP,1.0_DP/)         
         generators(2,:,:)=V_map(1,0)
         generators(3,:,:)=V_map(2,2)
         generators(4,:,:)=V_map(-1,-1)         
         generators(-1,1,:)=(/1.0_DP,-width/)
         generators(-1,2,:)=(/0.0_DP,1.0_DP/)         
         generators(-2,:,:)=V_map(0,1)
         generators(-3,:,:)=V_map(2,2)
         generators(-4,:,:)=V_map(-1,-1)
         num_gens=4


      else
!         write(*,*) 'Only N=3 or 4 implemented now!!!'
!         stop
         write(*,*) 'number of generators?'
         read(*,*) num_gens
         allocate(generators(-num_gens:num_gens,2,2))
         do j=1,num_gens
            write(*,*) 'generator ',j
            read(*,*) generators(j,1,1:2),generators(j,1,1:2)
            call gl_inverse(generators(j,:,:),generators(-j,:,:))
         enddo
      endif

      end subroutine


!     ! Get number of elliptic points for Gamma_0(N)
      
!     order 2
      subroutine get_e2()
      implicit none
      integer::p,e,tmp
      if(allocated(e2)) then 
         deallocate(e2)
      endif
      allocate(e2(1))
      if(mod(Level,4).eq.0) then 
         e=0
      else
         e=1
         do p=1,Level
            if(is_prime(p).and.(mod(Level,p).eq.0)) then 
!     ! tmp=1+(-1/p)               
               if(mod(p,4).eq.1) then 
                  tmp=2
               elseif(mod(p,4).eq.3) then 
                  tmp=0
               elseif(p.eq.2) then 
                  tmp=1
               endif
               e=e*tmp
            endif
         enddo
      endif
      e2(1)=e
!      write(*,*) 'e2=',e
      end subroutine




      subroutine get_e3()
      implicit none
      integer::p,e,tmp
      if(allocated(e3)) then 
         deallocate(e3)
      endif
      allocate(e3(1))
      if(mod(Level,9).eq.0) then 
         e=0
      else
         e=1
         do p=1,Level
            if(is_prime(p).and.(mod(Level,p).eq.0)) then 
!     !tmp=1+(-3/p)               
               if(mod(p,3).eq.1) then 
                  tmp=2
               elseif(mod(p,3).eq.2) then 
                  tmp=0
               elseif(p.eq.3) then 
                  tmp=1
               endif
               e=e*tmp
            endif
         enddo
      endif
      e3(1)=e
!      write(*,*) 'e3=',e
      end subroutine

      subroutine get_genus_congruence()
      implicit none
      integer::g,i1,i2,i3
      if((.not.(ALLOCATED(e2))).or.(e2(1).eq.0)) then
         call get_e2()
         call get_e3()
      endif
      if(allocated(genus)) deallocate(genus)
      allocate(genus(1))
      if(num_cusps.eq.0) then 
         write(*,*) 'ERROR: No cusps!'
         stop
      endif
      i1=num_reps-3*e2(1)-4*e3(1)-6*num_cusps
      if(mod(i1,12).ne.0) then 
         write(*,*) 'i1=',i1
         write(*,*) num_reps,'-3*',e2(1),'-4*',e3(1),'-6*',num_cusps
         stop
      endif
      g=1+i1/12
      genus(1)=g
      if(g.lt.0) then 
         write(*,*) 'genus=',g
         stop
      endif
      end subroutine
      
      
      subroutine get_cusp_widths()
      integer::j,u,v,n,m,sgd
      
      if(width.ge.1) then 
         cusp_width(1)=width
      endif
      !! note that 0 is not of the form u/v with v|N and (u,v)=1
      do j=2,num_cusps
         u=cusps(j,1)
         v=cusps(j,2)
!         write(*,*) 'u/v(',j,')=',u,v
         if(u.eq.0) then 
            cusp_width(j)=Level
         else
            call gcd(Level,v*v,sgd,m,n)
            cusp_width(j)=real(Level/sgd,DP)
         endif
      enddo
       !! test the correctness
       u=0
       do j=1,num_cusps
          u=u+anint(cusp_width(j))
       enddo
       if(u.ne.num_reps) then 
        write(*,*) 'ERROR: cusp widths are not ok!'
        write(*,*) 'widths=',cusp_width(:)
        stop
       endif 
      end subroutine

      
   
      end module

