!!!  MODULE: interpolation
!!!  Description:  contains necessary functions to interpolate stuff
!!!       both individual K_iR and the V_nl~ 
      module interpolation
      use commonvariables



      implicit none
      
      real(DP)::X_min,X_max,x_start,x_stop,x,h0,h1,low_lim,high_lim
      integer::Y_set
      real(DP),save::the_alpha,The_R,the_x_max,the_x_min  ! The interpolated R
      real(DP),SAVE::the_A
      real(DP),dimension(:,:),allocatable::kbes_vec
      complex(DP),dimension(:,:),allocatable::kbes_cvec
      real(DP),dimension(:,:,:),allocatable::whittw_vec
      integer,save,private::NDEG,num_intervals
      private::X_min,X_max,x_start,x_stop,x,h0,h1,low_lim,high_lim,      &
     &     The_R,the_x_max,Y_set,the_x_min,the_alpha
      logical,save::do_interp
      logical::num_intervals_set,NDEG_set,x_max_set
      logical,save::is_setup
      logical,save::is_setup_w
      integer,save::silent_interpolation

!      interface setup_interp_kbes
!      module procedure setup_interp_kbes
!      module procedure setup_interp_kbes2
!      end interface

      contains
      
      subroutine set_num_intervals(new_num)
      integer::new_num
      if(new_num.gt.0) then
         num_intervals=new_num      
         num_intervals_set=.true.
      else
         num_intervals_set=.false.
      endif
      end subroutine

      subroutine set_X_max(new_x_max)
      real(DP)::new_x_max
      if(x_max.gt.0.0_DP) then
         X_max=new_x_max
         X_max_set=.true.
      else
         X_max_set=.false.
      endif
      end subroutine

      subroutine set_degree(new_deg)
      integer::new_deg
      if(new_deg.gt.0) then
         NDEG=new_deg
         NDEG_set=.true.
      else
         NDEG_set=.false.
      endif
      end subroutine

      function get_size()
      integer::get_size
      get_size=NDEG*(num_intervals+1)

      end function get_size

      !! This sets up an interpolations grid
      !! to interpolate K_iR(x)  over x
      !! Y0 signifiec the smallest value 2PiY0 that we will have 
      !! to interpolate over
      subroutine setup_interp_kbes(R,Y0_in)
      use kbessel
      use commonvariables
      use groups
      implicit none
      real(DP)::R,Rtmp,Y0
      real(DP),optional::Y0_in
      integer::j,k,num_tmp,ND_tmp


      num_tmp=num_intervals
      ND_tmp=NDEG
      if(.not.(NDEG_set)) then
         NDEG=max(20,ND_tmp)
      endif
      if(NDEG.gt.30) then
         if(silent_interpolation.gt.0) then
            write(*,*) 'in setup_interp_kbes: R=',R
            write(*,*) 'x_max=',x_max,'  x_min=',x_min
            write(*,*) 'num_int=',num_intervals
            write(*,*) 'NDEG=',NDEG
         endif
         NDEG=22
         if(silent_interpolation.gt.0) then
            write(*,*) 'NDEG=',NDEG
         endif
      endif
      if(silent_interpolation.gt.0) then
         write(*,*) 'in setup_interp_kbes: R=',R
         write(*,*) 'Number=',(num_intervals+1)*NDEG
         write(*,*) 'x_max=',x_max,'  x_min=',x_min
      endif
      !! We don't have to do the same thing over again
      if((R.eq.the_R).and.(allocated(kbes_vec)).and.                     &
     &     (size(kbes_vec,2).eq.(NDEG+1)).and.                             &
     &     (size(kbes_vec,1).eq.(num_intervals+1)).and.                 &
     &     (X_max.eq.the_x_max)) then
         !!! we can allow some small margins
         if((PRESENT(Y0_in).and.(Y0_in.ge.0.5_DP*Y_set)).or.(                          &
     &        .not.(PRESENT(Y0_in)))) then
!            write(*,*) 'skip this turn...'
            return
         endif

         return
      endif
      !!!   Set up a vector with the points
      !!!   to evaluate K_iR(X) at  (R fixed)
      !!! I know that all pulledbacked points are above Y_min=sqrt(3)/(2*N)
      !!!  So I can start interpolating from there
      if((.not.(PRESENT(Y0_in))).or.(Y0_in.eq.0.0_DP)) then
         if(Level.eq.0) then
            write(*,*) 'ERROR: You must declare the group!!'
            stop
         else
            Y0=sqrt(3.0_DP)/(real(2*Level,DP))
         endif
      else
         Y0=Y0_in
      endif
      X_min=PI2*0.1_DP*Y0
      Y_set=y0
      if(.not.(X_max_set)) then
         if(R.lt.10) then
            x_max=50.0_DP
         elseif(R.lt.35) then
            X_max=75.0_DP
         else   !! this is how far I have to go to be able to say
            !! that exp(Pi*R/2)*K_iR(x_max)=0
            X_max=max(1.2_DP*R,100.0_DP)
         endif
      endif
      !!! Note that x_max=50 is ok since K_ir(50)<10E-16 for "all" R
      !!! remember that K_iR(x) here is evaluated as xexp(Pi/2*R)
      !! but that the exponential will factor through at the end
!      X_max=max(R+15.0_DP*R**(0.3333333333333),50.0_DP)
      !! Above this it is certainly almost 0
                     !! We can interpolate values up to 40 with relative error<=10E-10
                     !! e.g. 10 significant digits!
      !! Split the interval logarithmically
      low_lim=log(X_min)
      high_lim=log(X_max)

      !!! We now have to see that an appropriate number of intervals is used !!!

      if((.not.(num_intervals_set)).or.(num_intervals.lt.16)) then
!         num_intervals=max(ceiling(0.8_DP*R),16)         
         num_intervals=max(ceiling((x_max-x_min)/0.07_DP),16)
      endif
      h0=abs(high_lim-low_lim)/real(num_intervals,DP)
      if(allocated(kbes_vec).and.((The_R.ne.R).or.(size(kbes_vec,1)     &
     &     .ne.(num_intervals+1)).or.(size(kbes_vec,2).ne.(NDEG+1))))     &
     &     then
         deallocate(kbes_vec)
      endif
      if(.not.(allocated(kbes_vec))) then
         allocate(kbes_vec(0:num_intervals,0:NDEG))
      endif
!      write(*,*) 'after allocating'
      h1=h0/real(NDEG,DP)
!      write(*,*) 'evaluating K at ',num_intervals*(NDEG+1),' pts'
      j=0
      x_start=low_lim+real(j,DP)*h0
      x_stop=low_lim+real(j+1,DP)*h0
!         h1=abs(x_stop-x_start)/real(NDEG,DP)
      do k=0,NDEG
         x=exp(x_start+real(k,DP)*h1)            
         kbes_vec(j,k)=RKBessel(R,x)
      enddo

      do j=1,num_intervals
         x_start=low_lim+real(j,DP)*h0
         x_stop=low_lim+real(j+1,DP)*h0
!         h1=abs(x_stop-x_start)/real(NDEG,DP)
         kbes_vec(j,0)=kbes_vec(j-1,NDEG)
         do k=1,NDEG
            x=exp(x_start+real(k,DP)*h1)            
            kbes_vec(j,k)=RKBessel(R,x)
         enddo
      enddo
      The_R=R
      the_x_max=x_max
      is_setup=.true.
      end subroutine


    !! This sets up an interpolations grid
      !! to interpolate K_iR(x)  over x
      !! Y0 signifiec the smallest value 2PiY0 that we will have 
      !! to interpolate over
      !! Complex K-Bessel
!c$$$      subroutine setup_interp_kbesc(A,R,Y0_in)
!c$$$      use kbessel_cplx
!c$$$      use commonvariables
!c$$$      use groups
!c$$$      implicit none
!c$$$      real(DP)::A,R,Rtmp,Y0,xx,yy
!c$$$      real(DP),optional::Y0_in
!c$$$      integer::j,k,num_tmp,ND_tmp
!c$$$
!c$$$
!c$$$      num_tmp=num_intervals
!c$$$      ND_tmp=NDEG
!c$$$      if(.not.(NDEG_set)) then
!c$$$         NDEG=max(20,ND_tmp)
!c$$$      endif
!c$$$      if(NDEG.gt.30) then
!c$$$         if(silent_interpolation.ge.0) then
!c$$$            write(*,*) 'in setup_interp_kbes: R=',R,'A =',A
!c$$$            write(*,*) 'x_max=',x_max,'  x_min=',x_min
!c$$$            write(*,*) 'num_int=',num_intervals
!c$$$            write(*,*) 'NDEG=',NDEG
!c$$$         endif
!c$$$         NDEG=22
!c$$$         if(silent_interpolation.ge.0) then
!c$$$            write(*,*) 'NDEG=',NDEG
!c$$$         endif
!c$$$      endif
!c$$$      if(silent_interpolation.ge.0) then
!c$$$         write(*,*) 'in setup_interp_kbes: R=',R
!c$$$         write(*,*) 'Number=',(num_intervals+1)*NDEG
!c$$$         write(*,*) 'x_max=',x_max,'  x_min=',x_min
!c$$$      endif
!c$$$      !! We don't have to do the same thing over again
!c$$$      if((R.eq.the_R).and.(A.eq.the_A).and.(allocated(kbes_cvec)).and.                     &
!c$$$     &     (size(kbes_cvec,2).eq.(NDEG+1)).and.                             &
!c$$$     &     (size(kbes_cvec,1).eq.(num_intervals+1)).and.                 &
!c$$$     &     (X_max.eq.the_x_max)) then
!c$$$         !!! we can allow some small margins
!c$$$         if((PRESENT(Y0_in).and.(Y0_in.ge.0.5_DP*Y_set)).or.(                          &
!c$$$     &        .not.(PRESENT(Y0_in)))) then
!c$$$            return
!c$$$         endif
!c$$$
!c$$$         return
!c$$$      endif
!c$$$      if(silent_interpolation.ge.0) then
!c$$$         write(*,*) 'Do the setup!'
!c$$$      endif
!c$$$      the_R=R
!c$$$      the_A=A
!c$$$      !!!   Set up a vector with the points
!c$$$      !!!   to evaluate K_iR(X) at  (R fixed)
!c$$$      !!! I know that all pulledbacked points are above Y_min=sqrt(3)/(2*N)
!c$$$      !!!  So I can start interpolating from there
!c$$$      if((.not.(PRESENT(Y0_in))).or.(Y0_in.eq.0.0_DP)) then
!c$$$         if(Level.eq.0) then
!c$$$            write(*,*) 'ERROR: You must declare the group!!'
!c$$$            stop
!c$$$         else
!c$$$            Y0=sqrt(3.0_DP)/(real(2*Level,DP))
!c$$$         endif
!c$$$      else
!c$$$         Y0=Y0_in
!c$$$      endif
!c$$$      X_min=PI2*0.1_DP*Y0
!c$$$      Y_set=y0
!c$$$      if(.not.(X_max_set)) then
!c$$$         if(R.lt.10) then
!c$$$            x_max=50.0_DP
!c$$$         elseif(R.lt.35) then
!c$$$            X_max=75.0_DP
!c$$$         else   !! this is how far I have to go to be able to say
!c$$$            !! that exp(Pi*R/2)*K_iR(x_max)=0
!c$$$            X_max=max(1.2_DP*R,100.0_DP)
!c$$$         endif
!c$$$      endif
!c$$$      !!! Note that x_max=50 is ok since K_ir(50)<10E-16 for "all" R
!c$$$      !!! remember that K_iR(x) here is evaluated as xexp(Pi/2*R)
!c$$$      !! but that the exponential will factor through at the end
!c$$$!      X_max=max(R+15.0_DP*R**(0.3333333333333),50.0_DP)
!c$$$      !! Above this it is certainly almost 0
!c$$$                     !! We can interpolate values up to 40 with relative error<=10E-10
!c$$$                     !! e.g. 10 significant digits!
!c$$$      !! Split the interval logarithmically
!c$$$      low_lim=log(X_min)
!c$$$      high_lim=log(X_max)
!c$$$
!c$$$      !!! We now have to see that an appropriate number of intervals is used !!!
!c$$$
!c$$$      if((.not.(num_intervals_set)).or.(num_intervals.lt.16)) then
!c$$$!         num_intervals=max(ceiling(0.8_DP*R),16)         
!c$$$         num_intervals=max(ceiling((x_max-x_min)/0.07_DP),16)
!c$$$      endif
!c$$$      h0=abs(high_lim-low_lim)/real(num_intervals,DP)
!c$$$      if(allocated(kbes_cvec).and.((The_R.ne.R).or.(size(kbes_cvec,1)     &
!c$$$     &     .ne.(num_intervals+1)).or.(size(kbes_cvec,2).ne.(NDEG+1))))     &
!c$$$     &     then
!c$$$         deallocate(kbes_cvec)
!c$$$      endif
!c$$$      if(.not.(allocated(kbes_cvec))) then
!c$$$         allocate(kbes_cvec(0:num_intervals,0:NDEG))
!c$$$      endif
!c$$$!      write(*,*) 'after allocating'
!c$$$      h1=h0/real(NDEG,DP)
!c$$$!      write(*,*) 'evaluating K at ',num_intervals*(NDEG+1),' pts'
!c$$$      j=0
!c$$$      x_start=low_lim+real(j,DP)*h0
!c$$$      x_stop=low_lim+real(j+1,DP)*h0
!c$$$!         h1=abs(x_stop-x_start)/real(NDEG,DP)
!c$$$      do k=0,NDEG
!c$$$         x=exp(x_start+real(k,DP)*h1)            
!c$$$         call CKBESS(x,A,R,xx,yy)
!c$$$         kbes_cvec(j,k)=dcmplx(xx,yy)
!c$$$      enddo
!c$$$
!c$$$      do j=1,num_intervals
!c$$$         x_start=low_lim+real(j,DP)*h0
!c$$$         x_stop=low_lim+real(j+1,DP)*h0
!c$$$!         h1=abs(x_stop-x_start)/real(NDEG,DP)
!c$$$         kbes_cvec(j,0)=kbes_cvec(j-1,NDEG)
!c$$$         do k=1,NDEG
!c$$$            x=exp(x_start+real(k,DP)*h1)            
!c$$$            call CKBESS(x,A,R,xx,yy)
!c$$$            kbes_cvec(j,k)=dcmplx(xx,yy)
!c$$$!            kbes_cvec(j,k)=RKBessel(R,x)
!c$$$         enddo
!c$$$      enddo
!c$$$      The_R=R
!c$$$      the_x_max=x_max
!c$$$      is_setup=.true.
!c$$$      end subroutine


      subroutine  get_ndeg_and_intervals(num_evals,R,Y0)
      use commonvariables
      use groups
      implicit none
      real(DP)::R,Rtmp
      real(DP),optional::Y0
      integer::j,k,num_tmp,ND_tmp,num_evals

      num_tmp=num_intervals
      ND_tmp=NDEG
      if(.not.(NDEG_set)) then
         NDEG=max(20,ND_tmp)
      endif
      if(NDEG.gt.30) then
         if(silent_interpolation.gt.0) then
            write(*,*) 'x_max=',x_max,'  x_min=',x_min
            write(*,*) 'num_int=',num_intervals
            write(*,*) 'NDEG=',NDEG
         endif
         NDEG=22
         if(silent_interpolation.gt.0) then
            write(*,*) 'NDEG=',NDEG
         endif
      endif

      !! We don't have to do the same thing over again
                                !!! we can allow some small margins
!c$$$      if(((Y_set.gt.0).and.(PRESENT(Y0).and.(Y0.ge.0.5_DP*Y_set))).or.(                          &
!c$$$     &     .not.(PRESENT(Y0)))) then
!c$$$                                !   write(*,*) 'skip this turn...'
!c$$$         return
!c$$$      endif
      if((.not.(PRESENT(Y0))).or.(Y0.eq.0.0_DP)) then
         if(Level.eq.0) then
            write(*,*) 'ERROR: You must declare the group!!'
            stop
         else
            Y0=sqrt(3.0_DP)/(real(2*Level,DP))
         endif
      endif
      X_min=PI2*0.1_DP*Y0
      Y_set=y0
      if(.not.(X_max_set)) then
         if(R.lt.10) then
            x_max=50.0_DP
         elseif(R.lt.35) then
            X_max=75.0_DP
         else   !! this is how far I have to go to be able to say
            !! that exp(Pi*R/2)*K_iR(x_max)=0
            X_max=max(1.2_DP*R,100.0_DP)
         endif
      endif
      !!! Note that x_max=50 is ok since K_ir(50)<10E-16 for "all" R
      !!! remember that K_iR(x) here is evaluated as xexp(Pi/2*R)
      !! but that the exponential will factor through at the end
!      X_max=max(R+15.0_DP*R**(0.3333333333333),50.0_DP)
      !! Above this it is certainly almost 0
                     !! We can interpolate values up to 40 with relative error<=10E-10
                     !! e.g. 10 significant digits!
      !! Split the interval logarithmically
      low_lim=log(X_min)
      high_lim=log(X_max)
      !!! We now have to see that an appropriate number of intervals is used !!!
      if((.not.(num_intervals_set)).or.(num_intervals.lt.16)) then
!         num_intervals=max(ceiling(0.8_DP*R),16)         
         num_intervals=max(ceiling((x_max-x_min)/0.07_DP),16)
      endif
     
      write(*,*) 'x_max=',x_max,'  x_min=',x_min
      write(*,*) 'num_int=',num_intervals
      write(*,*) 'NDEG=',NDEG
      num_evals=num_intervals*(NDEG+1)    
      end subroutine

   !! This sets up an interpolations grid
      !! to interpolate K_iR(x)  over x
      subroutine setup_interp_kbes2(R,myid)
      use kbessel
      use commonvariables
      use groups
!      use mpi
      implicit none
      real(DP)::R,Rtmp
      integer::j,k,num_tmp,ND_tmp
      integer,optional::myid

      num_tmp=num_intervals
      ND_tmp=NDEG
      if(.not.(NDEG_set)) then
         NDEG=max(20,ND_tmp)
      endif
      if(NDEG.gt.30) then
         NDEG=22
      endif
      
      !! We don't have to do the same thing over again
      if((R.eq.the_R).and.(allocated(kbes_vec)).and.                     &
     &     (size(kbes_vec,2).eq.(NDEG+1)).and.                             &
     &     (size(kbes_vec,1).eq.(num_intervals+1)).and.                 &
     &     (X_max.eq.the_x_max)) then
!         write(*,*) 'skip this turn...'
!         return
      endif
      
    !  write(*,*) 'myid=',myid
    
     !!!   Set up a vector with the points
      !!!   to evaluate K_iR(X) at  (R fixed)
      
      !!! I know that all pulledbacked points are above Y_min=sqrt(3)/(2*N)
      !!!  So I can start interpolating from there
     
      if(Level.eq.0) then
         write(*,*) 'ERROR: You must declare the group!!'
         stop
      endif
      X_min=PI2*0.1_DP*sqrt(3.0_DP)/(real(2*Level,DP))
      if(.not.(X_max_set)) then
         if(R.lt.10) then
            x_max=50.0_DP
         elseif(R.lt.35) then
            X_max=75.0_DP
         else   !! this is how far I have to go to be able to say
            !! that exp(Pi*R/2)*K_iR(x_max)=0
            X_max=max(1.2_DP*R,100.0_DP)
         endif
      endif

      !!! Note that x_max=50 is ok since K_ir(50)<10E-16 for "all" R
      !!! remember that K_iR(x) here is evaluated as xexp(Pi/2*R)
      !! but that the exponential will factor through at the end
!      X_max=max(R+15.0_DP*R**(0.3333333333333),50.0_DP)
      !! Above this it is certainly almost 0
                     !! We can interpolate values up to 40 with relative error<=10E-10
                     !! e.g. 10 significant digits!
      !! Split the interval logarithmically
      low_lim=log(X_min)
      high_lim=log(X_max)

      !!! We now have to see that an appropriate number of intervals is used !!!

      if((.not.(num_intervals_set)).or.(num_intervals.lt.16)) then
!         num_intervals=max(ceiling(0.8_DP*R),16)         
         num_intervals=max(ceiling((x_max-x_min)/0.07_DP),16)
      endif
      h0=abs(high_lim-low_lim)/real(num_intervals,DP)

      if(allocated(kbes_vec).and.((The_R.ne.R).or.(size(kbes_vec,1)     &
     &     .ne.(num_intervals+1)).or.(size(kbes_vec,2).ne.(NDEG+1))))     &
     &     then
         deallocate(kbes_vec)
      endif

      if(.not.(allocated(kbes_vec))) then
         allocate(kbes_vec(0:num_intervals,0:NDEG))
      endif
      h1=h0/real(NDEG,DP)
      !! this is j=0
      x_start=low_lim+real(0,DP)*h0
      x_stop=low_lim+real(1,DP)*h0
                                !     write(*,*) 'num computations=',(NDEG+1)+num_intervals*NDEG
      do k=0,NDEG               !! To avoid computing stuff twice
         x=exp(x_start+real(k,DP)*h1)            
      !   write(*,*) 'X(',0,',',k,')=',x
         kbes_vec(0,k)=RKBessel(R,x)
      enddo
      do j=1,num_intervals
         x_start=low_lim+real(j,DP)*h0
         x_stop=low_lim+real(j+1,DP)*h0
         do k=1,NDEG            !! To avoid computing stuff twice
            x=exp(x_start+real(k,DP)*h1)            
       !     write(*,*) 'X(',j,',',k,')=',x
            kbes_vec(j,k)=RKBessel(R,x)
         enddo
      enddo
      The_R=R
      the_x_max=x_max
      is_setup=.true.
      end subroutine
      
      !! Perform the interpolation of K_iR(x)
      !! where R has been used to set up the grid previously
      function interp_kbes(X)
      use commonvariables
      use kbessel
      implicit none
      real(DP)::X
      integer,parameter::MAXDEG=51
      integer :: M0
      real(kind=DP)::interp_kbes
      ! tihs is the interpolated Vnl(R)
      real(kind=DP), dimension(0:MAXDEG)  :: B,D
      integer :: j,k,l,n,j1      
      real(kind=DP) :: SQQ,k1,varX,tmp1,tmp2,R
      !! For numerical reasons I will set a zero-quantity
      !! to avoid dividing with zero later on
      if(.not.(is_setup)) then
         write(*,*) 'the interpolaiton is not set up!!!'
         stop
      endif
      R=the_R
      !! Have to find out which index corresponds to Y
      !! in the grid above
      varX=log(X)
      j1=abs(varX-low_lim)/h0
      if(j1.lt.0) then
         write(*,*) 'X=',x,' j=',j1
      endif
      if(j1.gt.num_intervals) then
                                !     write(*,*) 'interp called with X=',X
         interp_kbes=zero
         return 
         ! there is also an R-dependent cut-off we can make
      elseif(x.gt.R) then
         tmp1=sqrt((x-R)*(x+R))
         tmp2=R*PI*0.5_DP-tmp1-2.0*R*atan(R/(tmp1+X))
         if(tmp2.lt.-125.0_DP) then
            interp_kbes=zero
            return
         endif
      endif
      !! gives the first index
!      k1=ceiling(abs(log(X)-log(X_min)-real(j1,DP)*h0)/h1)
!      varX=X
      k1=(varX-low_lim-real(j1,DP)*h0)/h1
      ! and the second
!      x=k1
      do j=0,NDEG
         B(J)=1.0_DP
         D(J)=1.0_DP
         do  k=0,NDEG
            if(k.ne.j) then
               B(J)=B(J)*(k1-real(k,DP))
               D(J)=D(J)*real((j-k),DP)
            endif
         end do
      end do
      SQQ=0.0_DP
      do  j=0,NDEG
         SQQ=SQQ+kbes_vec(j1,j)*(B(j)/D(j))
      end do
      interp_kbes=SQQ
      end function


    !! Perform the interpolation of K_iR(x)
      !! where R has been used to set up the grid previously
      function interp_kbesc(X)
      use commonvariables
      use kbessel
      implicit none
      real(DP)::X,A,R,VarX,k1,tmp1,tmp2
      integer,parameter::MAXDEG=51
      integer :: M0
      complex(kind=DP)::interp_kbesc
      ! tihs is the interpolated Vnl(R)
      real(kind=DP), dimension(0:MAXDEG)  :: B,D
      integer :: j,k,l,n,j1      
      complex(kind=DP) :: SQQ
      if(.not.(is_setup)) then
         write(*,*) 'the interpolaiton is not set up!!!'
         stop
      endif
      R=the_R
      A=the_A
      !! Have to find out which index corresponds to Y
      !! in the grid above
      varX=log(X)
      j1=abs(varX-low_lim)/h0
      if(j1.lt.0) then
         write(*,*) 'X=',x,' j=',j1
      endif
      if(j1.gt.num_intervals) then
                                !     write(*,*) 'interp called with X=',X
         interp_kbesc=czero
         return 
         ! there is also an R-dependent cut-off we can make
      elseif(x.gt.R) then
         tmp1=sqrt((x-R)*(x+R))
         tmp2=R*PI*0.5_DP-tmp1-2.0D0*R*atan(R/(tmp1+X))
         if(tmp2.lt.-125.0_DP) then
            interp_kbesc=czero
            return
         endif
      endif
      !! gives the first index
!      k1=ceiling(abs(log(X)-log(X_min)-real(j1,DP)*h0)/h1)
!      varX=X
      k1=(varX-low_lim-real(j1,DP)*h0)/h1
      ! and the second
!      x=k1
      do j=0,NDEG
         B(J)=1.0_DP
         D(J)=1.0_DP
         do  k=0,NDEG
            if(k.ne.j) then
               B(J)=B(J)*(k1-real(k,DP))
               D(J)=D(J)*real((j-k),DP)
            endif
         end do
      end do
      SQQ=0.0_DP
      do  j=0,NDEG
         SQQ=SQQ+kbes_cvec(j1,j)*(B(j)/D(j))
      end do
      interp_kbesc=SQQ
      end function





      !! This sets up an interpolations grid
      !! to interpolate W(sgn(n+alpha)*k/2,R;x)  over x
      
      !! alpha should be the minimum shift (wrt. to the cusps)
      subroutine setup_interp_WhittW(R,Y0,alpha_in)
      use whittakerw
      use commonvariables
      use groups
      implicit none
      real(DP)::R,Rtmp,alpha
      real(DP),optional::alpha_in,Y0
      integer::j,k,num_tmp,ND_tmp
      real(DP)::Y
      Y=sqrt(3.0_DP)/real(2*Level,DP)
      if(PRESENT(Y0)) then
         if(Y0.gt.0.0_DP) Y=Y0
      endif
      if(.not.(PRESENT(alpha_in))) then
         alpha=1.0_DP/12.0_DP*weight
      else
         alpha=alpha_in
      endif
      the_alpha=alpha
      num_tmp=num_intervals
      ND_tmp=NDEG
      if(.not.(NDEG_set)) then
         NDEG=max(20,ND_tmp)
      endif
      
!      if(NDEG.gt.30) then
      if(Y.lt.X_min) then 
         deallocate(whittw_vec)
      endif
         write(*,*) 'in setup_interp_whittW: R=',R,'Y=',Y,' weight=',    &
     &     weight
         write(*,*) 'x_max=',x_max,'  x_min=',x_min
         write(*,*) 'num_int=',num_intervals
         write(*,*) 'NDEG=',NDEG
!         NDEG=22
         write(*,*) 'NDEG=',NDEG
!      endif
      !! We don't have to do the same thing over again
      if((R.eq.the_R).and.(allocated(whittw_vec)).and.                     &
     &     (size(whittw_vec,2).eq.(NDEG+1)).and.                             &
     &     (size(whittw_vec,1).eq.(num_intervals+1)).and.                 &
     &     (X_max.eq.the_x_max)) then
         if((PRESENT(Y0).and.(Y0.ge.0.5_DP*Y_set)).or.(                          &
     &        .not.(PRESENT(Y0)))) then
!            write(*,*) 'skip this turn...'
            return
         endif
      endif

    
      !!! remember that the argument in Whittaker function
      !!! is 2*x where it is x in the K-Bessel

     !!!   Set up a vector with the points
      !!!   to evaluate K_iR(X) at  (R fixed)
      
      !!! I know that all pulledbacked points are above Y_min=sqrt(3)/(2*N)
      !!!  So I can start interpolating from there
     
      if(Level.eq.0) then
         write(*,*) 'ERROR: You must declare the group!!'
         stop
      endif
      if(alpha.eq.0.0_DP) then
         X_min=PI4*0.1_DP*Y
      else
         X_min=PI4*0.1_DP*Y*alpha
      endif
      Y_set=Y
!*sqrt(3.0_DP)/(real(2*Level,DP))
      if(.not.(X_max_set)) then         
        if(R.lt.10) then
            X_max=50.0_DP
         elseif(R.lt.35) then
            X_max=75.0_DP
         else   !! this is how far I have to go to be able to say
            !! that exp(Pi*R/2)*K_iR(x_max)=0
            X_max=1.0_DP*max(1.2_DP*R,70.0_DP)
         endif
      endif

      !!! Note that x_max=50 is ok since K_ir(50)<10E-16 for "all" R
      !!! remember that K_iR(x) here is evaluated as xexp(Pi/2*R)
      !! but that the exponential will factor through at the end
!      X_max=max(R+15.0_DP*R**(0.3333333333333),50.0_DP)
      !! Above this it is certainly almost 0
                     !! We can interpolate values up to 40 with relative error<=10E-10
                     !! e.g. 10 significant digits!
      !! Split the interval logarithmically
      low_lim=log(X_min)
      high_lim=log(X_max)

      !!! We now have to see that an appropriate number of intervals is used !!!

      if((.not.(num_intervals_set)).or.(num_intervals.lt.16)) then
!         num_intervals=max(ceiling(0.8_DP*R),16)         
         num_intervals=max(ceiling((x_max-x_min)/0.07_DP),16)
      endif
      h0=abs(high_lim-low_lim)/real(num_intervals,DP)

      if(allocated(whittw_vec).and.((The_R.ne.R).or.(size(whittw_vec,1)     &
     &     .ne.(num_intervals+1)).or.(size(whittw_vec,2).ne.(NDEG+1))))     &
     &     then
         deallocate(whittw_vec)
      endif

      if(.not.(allocated(whittw_vec))) then
         allocate(whittw_vec(0:num_intervals,0:NDEG,1:2))
      endif
      h1=h0/real(NDEG,DP)
      write(*,*) 'evaluating W at ',2*(num_intervals+1)*(NDEG),' pts'
      write(*,*) 'X_min=',x_min,' x_max=',x_max
      write(*,*) 'alpha=',alpha
      j=0
      x_start=low_lim+real(j,DP)*h0
      x_stop=low_lim+real(j+1,DP)*h0
!         h1=abs(x_stop-x_start)/real(NDEG,DP)
      !!! really ugly solution but I dont care right npow
      do k=0,NDEG
         x=exp(x_start+real(k,DP)*h1)
!         write(*,*) 'x(',j,',',k,')=',x
            !!! really ugly solution but I dont care right npow
            whittw_vec(j,k,1)=WhittW(0.5_DP*Weight,R,x)
            whittw_vec(j,k,2)=WhittW(-0.5_DP*Weight,R,x)
!            kbes_vec(j,k)=RKBessel(R,x)
      enddo
         
      do j=1,num_intervals
         x_start=low_lim+real(j,DP)*h0
         x_stop=low_lim+real(j+1,DP)*h0
!         h1=abs(x_stop-x_start)/real(NDEG,DP)
         whittw_vec(j,0,1)=whittw_vec(j-1,NDEG,1)
         whittw_vec(j,0,2)=whittw_vec(j-1,NDEG,2)
         do k=1,NDEG
            x=exp(x_start+real(k,DP)*h1)
!            write(*,*) 'x(',j,',',k,')=',x
            whittw_vec(j,k,1)=WhittW(0.5_DP*Weight,R,x)
            whittw_vec(j,k,2)=WhittW(-0.5_DP*Weight,R,x)
!            kbes_vec(j,k)=RKBessel(R,x)
         enddo
         
      enddo
      The_R=R
      the_x_min=x_min
      the_x_max=x_max
      is_setup_w=.true.
      end subroutine
      
   !! Perform the interpolation of K_iR(x)
      !! where R has been used to set up the grid previously

      !! sign=1 if 1st argument = +weight and sign=2 if -weight
      function interp_whittw(sign,X)
      use commonvariables
      use kbessel
      implicit none
      real(DP)::X
      integer,parameter::MAXDEG=51
      integer :: M0,sign
      real(kind=DP)::interp_whittw
      ! tihs is the interpolated Vnl(R)
      real(kind=DP), dimension(0:MAXDEG)  :: B,D
      integer :: j,k,l,n,j1      
      real(kind=DP) :: SQQ,k1,varX,tmp1,tmp2,R
      if(.not.(is_setup_w)) then
         write(*,*) 'the interpolaiton is not set up!!!'
         stop
      endif
      R=the_R
      !! if we have set it up but with a too small interval we need to redoit
!      write(14,*) X 
      if(X.lt.the_X_min) then
         write(*,*) 'Have to Redo the setup!'
         if(the_alpha.ne.0.0) then 
            tmp1=X*0.1_DP/PI4/the_alpha
            call setup_interp_WhittW(the_R,the_alpha,tmp1)
         else
            tmp1=X*0.1_DP/PI4
            call setup_interp_WhittW(the_R,the_alpha,tmp1)
           endif
      endif

      !! Have to find out which index corresponds to Y
      !! in the grid above
      varX=log(X)
      j1=abs(varX-low_lim)/h0
      if(j1.lt.0) then
         write(*,*) 'X=',x,' j=',j1
         write(*,*) 'h0=',h0
         write(*,*) 'num_intervals=',num_intervals
         write(*,*) 'x_max=',x_max
         write(*,*) 'x_min=',x_min
      endif
      if(j1.gt.num_intervals) then
                                !     write(*,*) 'interp called with X=',X
         interp_whittw=zero
         return 
         ! there is also an R-dependent cut-off we can make
      elseif(x.gt.R) then
         tmp1=sqrt((x-R)*(x+R))
         tmp2=R*PI*0.5_DP-tmp1-2.0*R*atan(R/(tmp1+X))
         if(tmp2.lt.-125.0_DP) then
            interp_whittw=zero
            return
         endif
      endif
      !! gives the first index
!      k1=ceiling(abs(log(X)-log(X_min)-real(j1,DP)*h0)/h1)
!      varX=X
      k1=(varX-low_lim-real(j1,DP)*h0)/h1
      ! and the second
!      x=k1
      do j=0,NDEG
         B(J)=1.0_DP
         D(J)=1.0_DP
         do  k=0,NDEG
            if(k.ne.j) then
               B(J)=B(J)*(k1-real(k,DP))
               D(J)=D(J)*real((j-k),DP)
            endif
         end do
      end do
      SQQ=0.0_DP
      do  j=0,NDEG
         SQQ=SQQ+whittw_vec(j1,j,sign)*(B(j)/D(j))
      end do
      interp_whittw=SQQ
      end function


  
      !!! 
      subroutine interpolate_total_cplx(x,Fx,M0)
      use commonvariables
      use giant_mod ! we use the data that are stored in giant_system
                ! which is a 2M0*2M0*N0 big matrix
      use groups
      IMPLICIT none
      integer, parameter :: MAXDEG=51
      integer,intent(in) :: M0
      real(kind=DP) :: x
      complex(kind=DP), dimension(num_cusps*(2*M0+1),num_cusps*(2*M0+1),  &
     &     1:2) :: Fx
      ! tihs is the interpolated Vnl(R)
      complex(kind=DP), dimension(0:intpol_deg)  :: F
      complex(DP)::B,D
      integer :: j,k,l,n,NDEG
      complex(kind=DP) :: SQQ
      NDEG = intpol_deg
      if(.not.(allocated(giant_total_system_cplx))) then
         write(*,*) 'ERROR: Giant System not allocated!'
         stop
      endif
      do j=0,NDEG
         B=dcmplx(1.0_DP,0.0_DP)
         D=dcmplx(1.0_DP,0.0_DP)
         do  k=0,NDEG
            if(k.ne.j) then
               B=B*(x-real(k,DP))
               D=D*real((j-k),DP)
            endif
         end do
         F(j)=B/D
      end do
      do  n=lbound(Fx,1),ubound(Fx,1)
         do  l=lbound(Fx,2),ubound(Fx,2)
            Fx(n,l,1)=sum(F(0:NDEG)*giant_total_system_cplx(0:NDEG,n,l,  &
     &           1))
            Fx(n,l,2)=sum(F(0:NDEG)*giant_total_system_cplx(0:NDEG,n,l,  &
     &           2))
         end do
      end do
      end subroutine interpolate_total_cplx


      !! Uses the large giant_total_system_cplx2
      !! matrix instead, which I use to store 
      !! the giant matrix for two different Y's at the same time
      !! without having to recompute them in each step
  
      !! no_y is the number of Y to use, either Y1 or Y2

      !!! use for any number of cusps and complex coefficients
      subroutine interpolate_total_cplx2(x,Fx,M0,yi)            
      use commonvariables
      use giant_mod ! we use the data that are stored in giant_system
                ! which is a 2M0*2M0*N0 big matrix
      use groups
      IMPLICIT none
      integer, parameter :: MAXDEG=51
      integer :: M0
      integer::yi
      real(kind=DP) :: x
      complex(kind=DP), dimension(num_cusps*M0,num_cusps*M0,0:1) :: Fx
      ! tihs is the interpolated Vnl(R)
      complex(kind=DP), dimension(0:MAXDEG)  :: B,D
      integer :: j,k,l,n,NDEG
      complex(kind=DP) :: SQQ
      NDEG = intpol_deg
      if(.not.(allocated(giant_total_system_cplx2))) then
         write(*,*) 'ERROR: Giant System not allocated!'
         stop
      endif
      do j=0,NDEG
         B(J)=dcmplx(1.0_DP,0.0_DP)
         D(J)=dcmplx(1.0_DP,0.0_DP)
         do  k=0,NDEG
            if(k.ne.j) then
               B(J)=B(J)*dcmplx((x-real(k,DP)),0.0_DP)
               D(J)=D(J)*dcmplx(real((j-k),DP),0.0_DP)
            endif
         end do
      end do
      do  n=1,num_cusps*M0
         do  l=1,num_cusps*M0
            SQQ=dcmplx(0.0_DP,0.0_DP)
            do  j=0,NDEG
               SQQ=SQQ+giant_total_system_cplx2(n,l,j,0,yi)*(B(j)/D(j))
            end do
            Fx(n,l,0)=SQQ
         end do
      end do
      do  n=1,num_cusps*M0
         do  l=1,num_cusps*M0
            SQQ=dcmplx(0.0_DP,0.0_DP)
            do  j=0,NDEG
               SQQ=SQQ+giant_total_system_cplx2(n,l,j,1,yi)*(B(j)/D(j))
            end do
            Fx(n,l,1)=SQQ
         end do
      end do
      end subroutine interpolate_total_cplx2


      !! Uses the large giant_total_system_cplx
      !! matrix instead, which I use to store 
      !! the giant matrix for different Y
      !! without having to recompute them in each step
      
      !! no_y is the number of Y to use, either Y1 or Y2
      !! With weights!!
      subroutine interpolate_total_cplx_w(x,Fx,M0,yi)
      use commonvariables
      use giant_mod ! we use the data that are stored in giant_system
                ! which is a 2M0*2M0*N0 big matrix
      use groups
      IMPLICIT none
      integer, parameter :: MAXDEG=51
      integer :: M0
      integer::yi
      real(kind=DP) :: x
      complex(kind=DP), dimension(num_cusps*(2*M0+1),num_cusps*(2*M0+1))    &
     &     :: Fx
      ! tihs is the interpolated Vnl(R)
      complex(kind=DP), dimension(0:MAXDEG)  :: B,D
      integer :: j,k,l,n,NDEG

      complex(kind=DP) :: SQQ
      NDEG = intpol_deg
      if(.not.(allocated(giant_total_system_cplx))) then
         write(*,*) 'ERROR: Giant System not allocated!'
         stop
      endif
      do j=0,NDEG
         B(J)=dcmplx(1.0_DP,0.0_DP)
         D(J)=dcmplx(1.0_DP,0.0_DP)
         do  k=0,NDEG
            if(k.ne.j) then
               B(J)=B(J)*dcmplx((x-real(k,DP)),0.0_DP)
               D(J)=D(J)*dcmplx(real((j-k),DP),0.0_DP)
            endif
         end do
      end do
      do  n=1,size(giant_total_system_cplx,1)
         do  l=1,size(giant_total_system_cplx,2)
            SQQ=dcmplx(0.0_DP,0.0_DP)
            do  j=0,NDEG
               SQQ=SQQ+giant_total_system_cplx(j,n,l,yi)*(B(j)/D(j))
            end do
            Fx(n,l)=SQQ
         end do
      end do
      end subroutine interpolate_total_cplx_w



      !!! This should only be used for prime index with two cusps!!!
      subroutine interpolate_total2(x,Fx,M0,yi)            
      use commonvariables
      use giant_mod ! we use the data that are stored in giant_system
                ! which is a 2M0*2M0*N0 big matrix
      IMPLICIT none
      integer, parameter :: MAXDEG=51
      integer :: M0
      integer::yi
      real(kind=DP) :: x
      !! remember two symmetryclasses
      real(kind=DP), dimension(1:M0-1,1:M0,0:1,0:1) :: Fx
      ! tihs is the interpolated Vnl(R)
      real(kind=DP), dimension(0:MAXDEG)  :: B,D
      integer :: j,k,l,n,NDEG
      real(kind=DP),dimension(0:1,0:1) :: SQQ
      NDEG = intpol_deg
      if(.not.(allocated(giant_total_system2))) then
         write(*,*) 'ERROR: Giant System not allocated!'
         stop
      endif
      do j=0,NDEG
         B(J)=1.0_DP   !dcmplx(1.0_DP,0.0_DP)
         D(J)=1.0_DP   !dcmplx(1.0_DP,0.0_DP)
         do  k=0,NDEG
            if(k.ne.j) then
               B(J)=B(J)*(x-real(k,DP))
               D(J)=D(J)*real((j-k),DP)
            endif
         end do
      end do
      do  n=1,M0-1
         do  l=1,M0
            SQQ(0:1,0:1)=0.0_DP
            do  j=0,NDEG
               SQQ(0,0)=SQQ(0,0)+giant_total_system2(n,l,j,0,yi,0)*         &
     &              (B(j)/D(j))
               SQQ(0,1)=SQQ(0,1)+giant_total_system2(n,l,j,0,yi,1)*         &
     &              (B(j)/D(j))
               SQQ(1,0)=SQQ(1,0)+giant_total_system2(n,l,j,1,yi,0)*         &
     &              (B(j)/D(j))
               SQQ(1,1)=SQQ(1,1)+giant_total_system2(n,l,j,1,yi,1)*         &
     &              (B(j)/D(j))
            end do
            Fx(n,l,0,0:1)=SQQ(0,0:1)
            Fx(n,l,1,0:1)=SQQ(1,0:1)
         end do
      end do
      end subroutine interpolate_total2


      subroutine interpolate_total3(x,Fx,NN,yi)            
      use commonvariables
      use giant_mod ! we use the data that are stored in giant_system
                ! which is a 2M0*2M0*N0 big matrix
      IMPLICIT none
      integer, parameter :: MAXDEG=51
      integer :: NN
      integer::yi
      real(kind=DP) :: x
      !! remember two symmetryclasses
      real(kind=DP), dimension(1:NN,1:NN,0:1):: Fx
      ! tihs is the interpolated Vnl(R)
      real(kind=DP), dimension(0:intpol_deg)  :: F
      real(DP)::B,D
      integer :: j,k,l,n,NDEG
      real(kind=DP),dimension(0:1) :: SQQ
      NDEG = intpol_deg
      if(.not.(allocated(giant_total_system3))) then
         write(*,*) 'ERROR: Giant System not allocated!'
         stop
      endif
      do j=0,NDEG
         B=1.0_DP   !dcmplx(1.0_DP,0.0_DP)
         D=1.0_DP   !dcmplx(1.0_DP,0.0_DP)
         do  k=0,NDEG
            if(k.ne.j) then
               B=B*(x-real(k,DP))
               D=D*real((j-k),DP)
            endif
         end do
         F(j)=B/D
      end do
      do  l=1,NN
         do  n=1,NN
            Fx(n,l,0)=sum(F(0:intpol_deg)*giant_total_system3(             &
     &           0:intpol_deg,n,l,0,yi))
            Fx(n,l,1)=sum(F(0:intpol_deg)*giant_total_system3(             &
     &           0:intpol_deg,n,l,1,yi))
         end do
      end do
      end subroutine interpolate_total3

      subroutine clean_interpolation()
      if(allocated(kbes_vec)) deallocate(kbes_vec)
      if(allocated(kbes_cvec)) deallocate(kbes_cvec)
      if(allocated(whittw_vec)) deallocate(whittw_vec)
      end subroutine



      end module
