!!! MODULE: matrix
!!! Description: Subroutines for multiplying matrices with different type of entries
!!!
!!!  NOTE: for same type entries use MATMUL!
      module matrix
      use commonvariables,only:DP

      interface print_mat
      module procedure print_mat_r
      module procedure print_mat_i
      module procedure print_mat_in
      module procedure print_mat_rn
      end interface

      interface print_mat_mupad
      module procedure print_mat_mupad_c
      module procedure print_mat_mupad_r
      end interface 

      interface print_vec
      module procedure print_vec_r
      module procedure print_vec_c
      end interface

      interface equal_vec
      module procedure equal_vec_r
      module procedure equal_vec_c
      end interface

      
      contains
      ! C=A*B  (A,B,C real)
      subroutine multiply_i(A,B,C)
      implicit none
      integer,dimension(2,2)::A,B,C      
      C(1,1)=A(1,1)*B(1,1)+A(1,2)*B(2,1)
      C(1,2)=A(1,1)*B(1,2)+A(1,2)*B(2,2)
      C(2,1)=A(2,1)*B(1,1)+A(2,2)*B(2,1)
      C(2,2)=A(2,1)*B(1,2)+A(2,2)*B(2,2)      

      end subroutine
      ! C=A*B (A integer, B,C real)
      subroutine multiply_i_r(A,B,C)
      implicit none
      integer,dimension(2,2)::A
      real(kind=DP),dimension(2,2)::B,C
      C(1,1)=real(A(1,1),kind=DP)*B(1,1)+real(A(1,2),kind=DP)*B(2,1)
      C(1,2)=real(A(1,1),kind=DP)*B(1,2)+real(A(1,2),kind=DP)*B(2,2)
      C(2,1)=real(A(2,1),kind=DP)*B(1,1)+real(A(2,2),kind=DP)*B(2,1)
      C(2,2)=real(A(2,1),kind=DP)*B(1,2)+real(A(2,2),kind=DP)*B(2,2)      
      end subroutine

      ! C=A*B (A,C real , B integer)
      subroutine multiply_r_i(A,B,C)
      implicit none
      integer,dimension(2,2)::B
      real(kind=DP),dimension(2,2)::A,C
      C(1,1)=A(1,1)*real(B(1,1),DP)+A(1,2)*real(B(2,1),DP)
      C(1,2)=A(1,1)*real(B(1,2),DP)+A(1,2)*real(B(2,2),DP)
      C(2,1)=A(2,1)*real(B(1,1),DP)+A(2,2)*real(B(2,1),DP)
      C(2,2)=A(2,1)*real(B(1,2),DP)+A(2,2)*real(B(2,2),DP)      
      end subroutine

      ! inverse of a matrix in SL(2,Z)
      subroutine invert_sl2z(A,Ai)
      implicit none
      integer,dimension(2,2)::A,Ai  
      Ai(1,1)=A(2,2)
      Ai(1,2)=-A(1,2)
      Ai(2,1)=-A(2,1)
      Ai(2,2)=A(1,1)
      end subroutine

      subroutine mattranspose(A,AT,k)
      implicit none
      integer::i,j,k
      real(DP)::A(k,k),AT(k,k)
      do i=1,k
         do j=1,k
            AT(i,j)=A(j,i)
         enddo
      enddo
      end subroutine

      ! inverse of a matrix in GL(2,R)
      subroutine gl_inverse(A,Ai)      
      implicit none
      real(kind=DP),dimension(2,2)::A,Ai  
      real(kind=DP)::det
      det=A(2,2)*A(1,1)-A(1,2)*A(2,1)
      if(det.eq.0) then
         write(*,*) 'ERROR: det(',A,')=',det
         stop
      endif
!      write(*,*) 'det(',A,')=',det
!      det=sqrt(abs(det))
      Ai(1,1)=A(2,2)/det
      Ai(1,2)=-A(1,2)/det
      Ai(2,1)=-A(2,1)/det
      Ai(2,2)=A(1,1)/det
!      write(*,*) 'A=[',A(1,:),',',A(2,:),']'
!      write(*,*) 'Ai=[',Ai(1,:),',',Ai(2,:),']'
      end subroutine


      !! inverse of map with real entries
 
      subroutine invert_map_r(A,Ai)
      implicit none
      real(kind=DP),dimension(2,2)::A,Ai  
      Ai(1,1)=A(2,2)
      Ai(1,2)=-A(1,2)
      Ai(2,1)=-A(2,1)
      Ai(2,2)=A(1,1)
      end subroutine

      !! The determinant of the integer matrix A
      function det_i(A)
      implicit none
      integer,dimension(2,2)::A
      integer :: det_i
      det_i=A(1,1)*A(2,2)-A(2,1)*A(1,2)
      end function

      subroutine print_mat_i(A)
      implicit none
      integer,dimension(2,2)::A
      write(*,'(A)',ADVANCE='NO') '['
      write(*,'(I4)',ADVANCE='NO') A(1,1)
      write(*,'(A)',ADVANCE='NO') ','
      write(*,'(I4)',ADVANCE='NO') A(1,2)
      write(*,'(A)',ADVANCE='NO') ']'
      write(*,*)
      write(*,'(A)',ADVANCE='NO') '['
      write(*,'(I3)',ADVANCE='NO') A(2,1)
      write(*,'(A)',ADVANCE='NO') ','
      write(*,'(I3)',ADVANCE='NO') A(2,2)
      write(*,'(A)',ADVANCE='NO') ']'
      write(*,*)
      end subroutine

      subroutine print_mat_r(A)
      use commonvariables
      implicit none
      real(DP),dimension(2,2)::A
      write(*,'(A)',ADVANCE='NO') '['
      write(*,'(F6.3)',ADVANCE='NO') A(1,1)
      write(*,'(A)',ADVANCE='NO') ','
      write(*,'(F6.3)',ADVANCE='NO') A(1,2)
      write(*,'(A)',ADVANCE='NO') ']'
      write(*,*)
      write(*,'(A)',ADVANCE='NO') '['
      write(*,'(F6.3)',ADVANCE='NO') A(2,1)
      write(*,'(A)',ADVANCE='NO') ','
      write(*,'(F6.3)',ADVANCE='NO') A(2,2)
      write(*,'(A)',ADVANCE='NO') ']'
      write(*,*)
      end subroutine

      subroutine print_mat_cn(A,k)
      use commonvariables
      implicit none
      integer::k,i,j
      complex(DP),dimension(k,k)::A
      write(*,'(A)',ADVANCE='NO') ''
      do i=1,k
         if(i.gt.1) then 
            write(*,'(A)',ADVANCE='NO') '  ['
         else
            write(*,'(A)',ADVANCE='NO') ' ['
         endif
         do j=1,k
            write(*,'(F10.5)',ADVANCE='NO') A(i,j)
         enddo
         if(i.eq.k) then
            write(*,'(A)',ADVANCE='NO') ']'
         else
            write(*,*) ']'
         endif

      enddo
      write(*,*) ']'
      end subroutine

! Write in the format of mupad to a file
      subroutine print_mat_mupad_c(A,k,l,FILENAME,VNAME)
      use commonvariables
      implicit none
      integer::k,i,j,l
      character*1::VNAME
      character*100::FILENAME
      complex(DP),dimension(k,l)::A
      open(unit=11,file=filename)
      write(11,'(A)',ADVANCE='NO') VNAME//':=matrix(['
      do i=1,k
         write(11,'(A)',ADVANCE='NO') ' ['
         do j=1,l
            write(11,'(ES26.17)',ADVANCE='NO') real(A(i,j),DP)
            write(11,'(A)',ADVANCE='NO') '+I*'
            write(11,'(ES26.17)',ADVANCE='NO') dimag(A(i,j))
            if(j.ne.l) then
               write(11,'(A)',ADVANCE='NO') ', '
            endif
         enddo
         if(i.eq.k) then
            write(11,'(A)',ADVANCE='NO') ']'
         else
            write(11,*) '],'
         endif
      enddo
            write(11,*) ']):'
      close(11)
      end subroutine

      subroutine print_vec_mupad_c(A,k,FILENAME,VNAME)
      use commonvariables
      implicit none
      integer::k,i,j,l
      character*1::VNAME
      character*100::FILENAME
      complex(DP),dimension(k)::A
      open(unit=11,file=filename)
      write(11,'(A)',ADVANCE='NO') VNAME//':=matrix('
      write(11,'(A)',ADVANCE='NO') ' ['
      do j=1,k
         write(11,'(ES26.17)',ADVANCE='NO') real(A(j),DP)
         write(11,'(A)',ADVANCE='NO') '+I*'
         write(11,'(ES26.17)',ADVANCE='NO') dimag(A(j))
         if(j.ne.k) then
            write(11,'(A)',ADVANCE='NO') ', '
         endif
      enddo
      write(11,*) ']):'
      close(11)
      end subroutine

      subroutine print_mat_mupad_r(A,k,l,FILENAME,VNAME)
      use commonvariables
      implicit none
      integer::k,i,j,l
      character*1::VNAME
      character*100::FILENAME
      real(DP),dimension(k,l)::A
      open(unit=11,file=filename)
      write(11,'(A)',ADVANCE='NO') VNAME//':=matrix(['
      do i=1,k
         write(11,'(A)',ADVANCE='NO') ' ['
         do j=1,l
            write(11,'(ES26.17)',ADVANCE='NO') A(i,j)
            if(j.ne.l) then
               write(11,'(A)',ADVANCE='NO') ', '
            endif
         enddo
         if(i.eq.k) then
            write(11,'(A)',ADVANCE='NO') ']'
         else
            write(11,*) '],'
         endif
      enddo
            write(11,*) ']):'
      close(11)
      end subroutine

      subroutine print_mat_rn(M,k)
      implicit none
      real(kind=DP)::M(k,k)
      integer::i,j,k
      write(*,'(A)',ADVANCE='NO') ''
      do i=1,k
         if(i.gt.1) then 
            write(*,'(A)',ADVANCE='NO') '  ['
         else
            write(*,'(A)',ADVANCE='NO') ' ['
         endif
         do j=1,k
            write(*,'(F10.5)',ADVANCE='NO') M(i,j)
         enddo
         if(i.eq.k) then
            write(*,'(A)',ADVANCE='NO') ']'
         else
            write(*,*) ']'
         endif

      enddo
      write(*,*) ']'
      end subroutine
      subroutine print_mat_in(M,k)
      implicit none
      integer::M(k,k)
      integer::i,j,k
      write(*,'(A)',ADVANCE='NO') '['
      do i=1,k
         if(i.gt.1) then 
            write(*,'(A)',ADVANCE='NO') '  ['
         else
            write(*,'(A)',ADVANCE='NO') ' ['
         endif
         do j=1,k
            write(*,'(I10)',ADVANCE='NO') M(i,j)
         enddo
         if(i.eq.k) then
            write(*,'(A)',ADVANCE='NO') ']'
         else
            write(*,*) ']'
         endif
      enddo
      write(*,*) ']'
      end subroutine

      subroutine print_vec_c(V,k)
      use commonvariables
      implicit none
      integer::k,i
      complex(DP),dimension(k)::V
      write(*,'(A)',ADVANCE='NO') ''
      write(*,'(A)',ADVANCE='NO') ' ['
      do i=1,k         
         write(*,'(F10.5)',ADVANCE='NO') V(i)
         if(i.ne.k) then
            write(*,'(A)',ADVANCE='NO') ','
         endif
      enddo
      write(*,*) ']'
      end subroutine

      subroutine print_vec_r(V,k)
      use commonvariables
      implicit none
      integer::k,i
      real(DP),dimension(k)::V
      write(*,'(A)',ADVANCE='NO') ''
      write(*,'(A)',ADVANCE='NO') ' ['
      do i=1,k         
         write(*,'(F10.5)',ADVANCE='NO') V(i)
         if(i.ne.k) then
            write(*,'(A)',ADVANCE='NO') ','
         endif
      enddo
      write(*,*) ']'
      end subroutine


      ! this is supposed to be called on for 
      ! points x+iy with y>0
      ! so if I divide by 0
      ! I simply return x_out=x_i*a/c and y_out= -1.0
      subroutine apply_map_re(x_in,y_in,x_out,y_out,mapping)
      implicit none
      real(kind=DP),intent(in) :: x_in,y_in
      real(kind=DP),intent(out) :: x_out,y_out
      real(kind=DP), dimension(2,2),intent(in) :: mapping
      real(kind=DP) :: a,b,c,d,denominator,numerator

      a = real(mapping(1,1),DP)
      b = real(mapping(1,2),DP)
      c = real(mapping(2,1),DP)
      d = real(mapping(2,2),DP)
      denominator = (c*x_in+d)**(2.0_DP)+(c*y_in)**(2.0_DP)
      numerator = a*c*(x_in*x_in+y_in*y_in)+x_in*(a*d+b*c)+b*d
      if(denominator == 0.0_DP) then
         write(*,*) 'ERROR: division by zero! |cz+d|=0'
         write(*,*) 'z=',x_in,'+i*',y_in
         write(*,*) 'map=',mapping
         stop
      endif
      x_out = numerator/denominator
      y_out = y_in*(a*d-b*c)/denominator
      end subroutine apply_map_re

      subroutine apply_map_int(x_in,y_in,x_out,y_out,mapping)
      implicit none
      real(kind=DP),intent(in) :: x_in,y_in
      real(kind=DP),intent(out) :: x_out,y_out
      integer, dimension(2,2),intent(in) :: mapping
      real(kind=DP) :: a,b,c,d,denominator,numerator

      a = real(mapping(1,1),DP)
      b = real(mapping(1,2),DP)
      c = real(mapping(2,1),DP)
      d = real(mapping(2,2),DP)
      denominator = (c*x_in+d)**(2.0_DP)+(c*y_in)**(2.0_DP)
      numerator = a*c*(x_in*x_in+y_in*y_in)+x_in*(a*d+b*c)+b*d
      if(denominator == 0.0_DP) then
         write(*,*) 'ERROR: division by zero! |cz+d|=0'
         write(*,*) 'z=',x_in,'+i*',y_in
         write(*,*) 'map=',mapping
      endif
      x_out = numerator/denominator
      y_out = y_in*(a*d-b*c)/denominator
      end subroutine apply_map_int

      ! Apply a complex frac lin. map
      !
      subroutine apply_map_cplx(zin,zut,TT)
      implicit none
      complex(kind=DP),dimension(2,2)::TT 
      complex(kind=DP)::a,b,c,d,zin,zut,den
      a=TT(1,1)
      b=TT(1,2)
      c=TT(2,1)
      d=TT(2,2)
      den=(c*zin+d)
      if(abs(den).eq.0) then
         write(*,*) 'Error: Dividing with 0!'
         stop
      endif
      zut=(a*zin+b)/den
      
      end subroutine apply_map_cplx


      ! This is a more efficient routine for a real number x
      subroutine apply_map_R(X,Y,A)
      implicit none
      real(kind=DP)::X,Y,den
      integer,dimension(2,2)::A
      den=real(A(2,1),DP)*x+real(A(2,2),DP)
      if(abs(den).gt.0) then
         Y=(real(A(1,1),DP)*x+real(A(1,2),DP))/den
      else
         write(*,*) 'ERROR: division by 0 with ',x,A
         stop
      endif
      end subroutine apply_map_R


      !!! Are to be used for hyperbolic maps
      subroutine get_fixpts_r(TT,x)
      implicit none
      complex(DP),dimension(1:2)::x
      real(DP),dimension(2,2)::TT
      real(DP)::a,b,c,d,y
      
      a=TT(1,1)
      b=TT(1,2)
      c=TT(2,1)
      d=TT(2,2)
      if(c.eq.0.0_DP) then
         x(:)=b/(d-a)
      else
         y=0.5_DP*(d-a)/c
         x(1)=-y+sqrt(dcmplx(y**2+b/c,0.0_DP))
         x(2)=-y-sqrt(dcmplx(y**2+b/c,0.0_DP))
      endif
      end subroutine get_fixpts_r


      function equal_vec_c(A,B,k)
      implicit none
      integer::j,k
      complex(DP)::A(k),B(k)
      logical::equal_vec_c
      equal_vec_c=.false.
      do j=1,k
         if(A(j).ne.B(j)) return
      enddo
      equal_vec_c=.true.
      end function
      function equal_vec_r(A,B,k)
      implicit none
      integer::j,k
      real(DP)::A(k),B(k)
      logical::equal_vec_r
      equal_vec_r=.false.
      do j=1,k
         if(A(j).ne.B(j)) return
      enddo
      equal_vec_r=.true.
      end function



!! This will get a map in PSL(2,Z) which maps infinity to u/v
      subroutine get_mat_map_uv_to_infinity(A,u,v)
      use numtheory
      implicit none
      integer::u,v,A(2,2)
      integer::i,j,sgd,m,n,up,vp
      if(v.eq.0) then 
         A(1,1:2)=(/1,0/); A(2,1:2)=(/0,1/)
      else
         call gcd(u,v,sgd,m,n)  !! um+vn=sgd
         if(sgd.ne.1) then 
            up=u/sgd; vp=v/sgd
         else
            up=u; vp=v
         endif
         A(1,1)=up
         A(2,1)=vp
         A(1,2)=-n
         A(2,2)=m
         if(A(1,1)*A(2,2)-A(2,1)*A(1,2).ne.1) then 
            write(*,*) 'Error: A not in SL_2!'
            call print_mat(A)
            stop
         endif
      endif
      end subroutine get_mat_map_uv_to_infinity

!! And this returns a map in PSL(2,Z) mapping u/v to p/q
      subroutine maps_uv_to_pq(u,v,p,q,NL,NR,Av)
      implicit none
      integer,intent(in)::NL,NR
      integer,dimension(NL:NR,2,2),intent(out)::Av
      integer,intent(in)::u,v,p,q
      integer,dimension(2,2)::B1,B2,Bi,T,Btmp
      integer::i
      real(DP)::ur,vr,x,y
      call  get_mat_map_uv_to_infinity(B1,u,v)
      call  get_mat_map_uv_to_infinity(B2,p,q)
      call invert_sl2z(B1,Bi)
      do i=NL,NR
         T(1,:)=(/1,i/)
         T(2,:)=(/0,1/)
         Btmp=MATMUL(T,Bi)
         Av(i,:,:)=MATMUL(B2,Btmp)
      !! Test 
         ur=real(u,DP); vr=real(v,DP)
         if(q.eq.0)then
            if(abs(Av(i,2,1)*ur+Av(i,2,2)*vr).gt.1E-8_DP) then 
               write(*,*) 'Error: A(u/v)<>p/q=1/0'
               call apply_map_int(ur/vr,0.0_DP,x,y,Av(i,:,:))
               write(*,*) 'A='
               call print_mat(Av(i,:,:))
               write(*,*) 'A(u/v)=',x,'+i',y
               stop
            endif
         else
            call apply_map_int(ur/vr,0.0_DP,x,y,Av(i,:,:))
            if(abs(x-real(p,DP)/real(q,DP)).gt.1E-8_DP) then 
               write(*,*) 'Error: A(u/v) <> p/q!, NL,NR=',NL,NR
               write(*,*) 'u,v=',u,v
               write(*,*) 'p,q=',p,q
               write(*,*) 'p/q=',real(p,DP)/real(q,DP)
               write(*,*) 'A(u/v)=',x
               write(*,*) 'B1='
               call print_mat(B1)
               write(*,*) 'B2='
               call print_mat(B2)
               write(*,*) 'T='
               call print_mat(T)
               write(*,*) 'Btmp='
               call print_mat(Btmp)
               write(*,*) 'A='
               call print_mat(Av(i,:,:))
               stop
            endif
         endif
      enddo
      end subroutine maps_uv_to_pq


!! Using bubble sort to sort a list according to index ix      
      subroutine sort(list,ix,n,k)
      implicit none
      real(DP)::list(n,k)
      integer::n,k,i,j,ix,m
      real(DP)::tmpl1(k),tmpl2(k)
      logical::swapped
      
      swapped=.true.
      m=n
      do while(swapped) 
         swapped=.false.
         m=m-1
         do j=1,m
            if(list(j,ix).gt.list(j+1,ix)) then ! swap
               tmpl1=list(j,:)
               tmpl2=list(j+1,:)
               list(j,:)=tmpl2
               list(j+1,:)=tmpl1
!               write(*,*) 'change:',j,'and ',j+1
               swapped=.true.
            endif
         enddo
      enddo
      end subroutine


      end module
