                                !!! module : sys_solve
                                !!! Description: routines for solving the linear systems


      module sys_solve
      use commonvariables
      use groups
      integer,save::num_set
      complex(DP),save,dimension(:),allocatable::c_set,inv_ev
      integer,save,dimension(:,:),allocatable::which_c      
      logical,save::set_normalization,use_lap

                                !! indicates whether we have set the normalization or not
      interface SMAT
      module procedure SMAT_real
      module procedure SMAT_cplx
      end interface

      interface solve_system
      module procedure solve_system_COmplex6
      module procedure solve_system_real_use_sym_lap
      module procedure solve_system_cplx_use_sym_lap
      end interface

      interface set_cusp_norm
      module procedure set_cusp_norm_re
      module procedure set_cusp_norm_cplx
      module procedure set_cusp_norm_int
      end interface

      contains

      !! 0 = Even
      !! 1 = Odd
      subroutine set_even_odd_sym(i)
      implicit none
      integer::i
      use_sym_odd=.false.
      use_sym_even=.false.
      if(i.eq.0) then 
         use_sym_even=.true.
      elseif(i.eq.1) then
         use_sym_odd=.true.
      endif
      end subroutine

     !! Set the normalization
     !! inter=-1,0,1
     !!   -1 : Set them interactively
     !!    0 : Set standardized c(1)=1 and c(0)=0 if alpha_0=0 only
     !!    1 : Supply vectors
     !!   j>1: set c(1)=1 c(j)=0
     !!  j<-1: set c(1)=0 c(j)=1
      subroutine set_norm(inter,num_s,c,e,which)
      use characters
      implicit none
      integer::inter
      integer,optional::num_s
      complex(DP),dimension(:),optional::c,e
      integer,dimension(:),optional::which
      integer::j
      logical::deb
      deb=.true.
      deb=.false.
      call get_alphas()
      if((inter.eq.1).and.(PRESENT(num_s))) then
         num_set=num_s
      elseif(inter.eq.0) then
         if(alphas(1).eq.0.0) then 
            num_set=2
         else
            num_set=1
         endif
      elseif(inter.eq.-1) then
         write(*,*) 'How many coefficients to set?'
         read(*,*) num_set
      else
         num_set=2
      endif
      if(num_cusps.eq.0) then
         write(*,*) 'Number of cusps is zero!'
         stop
      endif
      if(allocated(c_set)) deallocate(c_set)
      allocate(c_set(1:num_set))
      if((allocated(inv_ev).and.(inter.ne.0)).or.                         &
     & (allocated(inv_ev).and.(size(inv_ev,1).ne.num_cusps))) then           
         deallocate(inv_ev)       
      endif
      if(.not.(allocated(inv_ev))) then 
         allocate(inv_ev(1:num_cusps))
      endif
      
      if(allocated(which_c)) deallocate(which_c)
      allocate(which_c(1:num_set,1:2))         
      which_c(1:num_set,1)=1
      if(inter.eq.1) then
         if(PRESENT(c).and.PRESENT(which)) then 
                                !.and.(size(c).eq.num_set).and.   &
                                !     &        (size(c).eq.num_set)) then
            c_set(1:num_set)=c(1:num_set)
            !! set only at the first cusp
            which_c(1:num_set,1)=1
            which_c(1:num_set,2)=which(1:num_set)
         else                   !!
            write(*,*) 'ERROR: num_s present but something is wrong!'
            write(*,*) 'num_set=',num_set
            if(PRESENT(c)) then
               write(*,*) 'c=',c(:)
               write(*,*) 'size(c)=',size(c)
            else
               write(*,*) 'nu c in input!'
            endif
            if(PRESENT(which)) then
               write(*,*) 'which=',which(1:num_set)
               write(*,*) 'size(which)=',size(which)
            else
               write(*,*) 'nu which in input!'
            endif            
            
            stop
         endif
      elseif(inter.eq.0) then
         !! c1=1
         which_c(1,1)=1
         which_c(1,2)=1
         c_set(1)=dcmplx(1.0_DP,0.0_DP)
         if(num_set.eq.2) then 
!            write(*,*) 'Set c0=0, c1=1'
            which_c(2,1)=1
            which_c(2,2)=0
            c_set(2)=dcmplx(0.0_DP,0.0_DP)
         endif
      elseif(inter.eq.-1) then
         write(*,'(A)',ADVANCE='NO') 'set which '
         write(*,'(I3)',ADVANCE='NO') num_set
         write(*,'(A)',ADVANCE='NO') ' coefficient(s)? ' 
         read(*,*) which_c(1:num_set,2)
         write(*,*) 'set to what values?' 
         do j=1,num_set
            write(*,'(A)',ADVANCE='NO') 'c('
            write(*,'(I3)',ADVANCE='NO') which_c(j,2)
            write(*,'(A)',ADVANCE='NO') ')= '
            read(*,*) c_set(j)
            write(*,*) c_set(j)
         enddo
      elseif(inter.lt.-1) then
         which_c(1,2)=1
         which_c(2,2)=-inter
         c_set(1)=dcmplx(0.0_DP,0.0_DP)
         c_set(-inter)=dcmplx(1.0_DP,0.0_DP)
      elseif(inter.gt.1) then
         which_c(1,2)=1
         which_c(2,2)=-inter
         c_set(1)=dcmplx(1.0_DP,0.0_DP)
         c_set(inter)=dcmplx(0.0_DP,0.0_DP)
      endif
      inv_ev(1)=dcmplx(1.0_DP,0.0_DP)
      inv_ev(2:num_cusps)=0.0_DP
      if((inter.eq.1).and.(PRESENT(e))) then
         if(size(e).ne.num_cusps) then
            write(*,*) 'Incorrect size of e!'
            stop
         endif
         inv_ev(1:num_cusps)=e(1:num_cusps)
      elseif((inter.lt.0).and.(num_cusps.gt.1)) then
         write(*,*) 'Set involution eigenvalues for:'
         do j=1,num_cusps-1
!            if(Nj_in_normalizer(j).eq.1) then
               write(*,'(I3)',ADVANCE='NO') j+1
               write(*,'(A)',ADVANCE='NO') ':'
               read(*,*) inv_ev(j+1)
               write(*,*) inv_ev(j+1)
!            endif
         enddo
      endif
      set_normalization=.true.
      if(deb) then 
         write(*,*) 'We have set:'
         write(*,*) 'number of set coefficients:',num_set
         do j=1,num_set
            write(*,*) 'C(',j,')=',c_set(j)
         enddo
         write(*,*) 'Cusp eigenvalues:'
         do j=1,num_cusps-1
            write(*,*) 'e(',j,')=',inv_ev(j+1)
         enddo
      endif
                                !      write(*,*) 'norm_set!'
      end subroutine set_norm            
           

      subroutine set_cusp_norm_re(inv_ev_re)
      implicit none
      real(DP)::inv_ev_re(1:num_cusps)
      if(allocated(inv_ev)) deallocate(inv_ev)
      allocate(inv_ev(1:num_cusps))
      inv_ev(:)=dcmplx(inv_ev_re(:))
      end subroutine
      subroutine set_cusp_norm_int(inv_ev_int)
      implicit none
      integer::inv_ev_int(1:num_cusps)
      if(allocated(inv_ev)) deallocate(inv_ev)
      allocate(inv_ev(1:num_cusps))
      inv_ev(:)=dcmplx(real(inv_ev_int(:),DP))
      end subroutine
      subroutine set_cusp_norm_cplx(inv_ev_cplx)
      implicit none
      complex(DP)::inv_ev_cplx(1:num_cusps)
      if(allocated(inv_ev)) deallocate(inv_ev)
      allocate(inv_ev(1:num_cusps))
      inv_ev(:)=inv_ev_cplx(:)
      end subroutine

                                
      subroutine set_norm_allcusps(inter,num_s,c,e,which)
      implicit none
      integer::inter
      integer,optional::num_s
      complex(DP),dimension(:),optional::c,e
      integer,dimension(:,:),optional::which
      integer::j
      logical::deb
      deb=.true.
      deb=.false.
      if((inter.eq.1).and.(PRESENT(num_s))) then
         num_set=num_s
      elseif(inter.eq.0) then
         num_set=2
      elseif(inter.eq.-1) then
         write(*,*) 'How many coefficients to set?'
         read(*,*) num_set
      else
         num_set=2
      endif
      if(allocated(c_set)) deallocate(c_set)
      allocate(c_set(1:num_set))
      if(allocated(inv_ev)) deallocate(inv_ev)
      if(num_cusps.eq.0) then
         write(*,*) 'Number of cusps is zero!'
         stop
      endif
      allocate(inv_ev(1:num_cusps))
      if(allocated(which_c)) deallocate(which_c)
      allocate(which_c(1:num_set,1:2))         
      ! If we say nothing else we set coefficients at the first cusp by default
      which_c(1:num_set,1)=1         
      if(inter.eq.1) then
         if(PRESENT(c).and.PRESENT(which)) then 
                                !.and.(size(c).eq.num_set).and.   &
                                !     &        (size(c).eq.num_set)) then
            c_set(1:num_set)=c(1:num_set)
            which_c(1:num_set,:)=which(1:num_set,:)
         else                   !!
            write(*,*) 'ERROR: num_s present but something is wrong!'
            write(*,*) 'num_set=',num_set
            if(PRESENT(c)) then
               write(*,*) 'c=',c(:)
               write(*,*) 'size(c)=',size(c)
            else
               write(*,*) 'nu c in input!'
            endif
            if(PRESENT(which)) then
               write(*,*) 'which=',which(1:num_set,1:2)
               write(*,*) 'size(which)=',size(which,1)
            else
               write(*,*) 'nu which in input!'
            endif            
            
            stop
         endif
      elseif(inter.eq.0) then
         c_set(1)=dcmplx(1.0_DP,0.0_DP)
         c_set(2)=dcmplx(0.0_DP,0.0_DP)
         which_c(1,2)=1
         which_c(2,2)=0
      elseif(inter.eq.-1) then
         write(*,'(A)',ADVANCE='NO') 'set which '
         write(*,'(I3)',ADVANCE='NO') num_set
         write(*,'(A)',ADVANCE='NO') ' cusps and coefficient(s)? i,n' 
         read(*,*) which_c(1:num_set,1:2)
         write(*,*) 'set to what values?' 
         do j=1,num_set
            write(*,'(AI3A)',ADVANCE='NO') 'c(',which_c(j,2),','
            write(*,'(I3)',ADVANCE='NO') which_c(j,2)
            write(*,'(A)',ADVANCE='NO') ')= '
            read(*,*) c_set(j)
            write(*,*) c_set(j)
         enddo
      elseif(inter.lt.-1) then
         which_c(1,2)=1
         which_c(2,2)=-inter
         c_set(1)=dcmplx(0.0_DP,0.0_DP)
         c_set(-inter)=dcmplx(1.0_DP,0.0_DP)
      elseif(inter.gt.1) then
         which_c(1,2)=1
         which_c(2,2)=-inter
         c_set(1)=dcmplx(1.0_DP,0.0_DP)
         c_set(inter)=dcmplx(0.0_DP,0.0_DP)
      endif
      inv_ev(1)=dcmplx(1.0_DP,0.0_DP)
      inv_ev(2:num_cusps)=0.0_DP
      if((inter.eq.1).and.(PRESENT(e))) then
         if(size(e).ne.num_cusps) then
            write(*,*) 'Incorrect size of e!'
            stop
         endif
         inv_ev(1:num_cusps)=e(1:num_cusps)
      elseif((inter.lt.0).and.(num_cusps.gt.1)) then
         write(*,*) 'Set involution eigenvalues for:'
         do j=1,num_cusps-1
!            if(Nj_in_normalizer(j).eq.1) then
               write(*,'(I3)',ADVANCE='NO') j+1
               write(*,'(A)',ADVANCE='NO') ':'
               read(*,*) inv_ev(j+1)
               write(*,*) inv_ev(j+1)
!            endif
         enddo
      endif
      set_normalization=.true.
      if(deb) then 
         write(*,*) 'We have set:'
         write(*,*) 'number of set coefficients:',num_set
         do j=1,num_set
            write(*,*) 'C(',j,')=',c_set(j)
         enddo
         write(*,*) 'Cusp eigenvalues:'
         do j=1,num_cusps-1
            write(*,*) 'e(',j,')=',inv_ev(j+1)
         enddo
      endif

                                !      write(*,*) 'norm_set!'
      end subroutine set_norm_allcusps            
 !! This is a subroutine which solves the linear system 
                                !! V*C=0
                                !! We use both positive (CY) and negative (CNY) coefficients
                                !! and the possible normalizations are
                                !!  - set any number of coefficients to fixed values
                                !!  - set coefficients of a given cusp to a (complex) 
                                !!    multiple of the coefficients at infinity (i.e. cusp nr.1) 
                                !! Indata:
                                !! VnlCp - complex ((2*M0+1)*num_cusps)**2 matrix
                                !! M0    - integer
                                !! Optional indata for normalizations:
                                !!  param - integer  : short form for set c(1)=0, c(p)=1 or c(1)=1, c(p)=0
                                !!  e_set - complex (1:num_cusps) vector where
                                !!                       c_j(k)=c_1(k)*e_set(k)
                                !!  num_set - integer = the number of coefficients c_1(k) set
                                !!  c_set   - complex (1:num_set) the values of the set coefficients
                                !!  which_c - integer (1:num_set) the index of the cusp set
                                !!  Default behaviour:
                                !!  If none of the optional arguments are supplied we use only the	     normalization c_1(1)=1
                                !     OBS:We always set c_j(0)=0 if alpha_j=0
 

            ! This routine solve the system while allowing

                                ! for a complete involution-symmetrization
                                ! via the parameter-vector e
                                ! e.g. c_j(k)=e(j)c_0(k)

                                !!! Use for a general system with any number of cusps
                                !!! but make sure e(j)=0 for cusps without involutions


      subroutine solve_system_Complex6(VnlCp,A,B,M0,e) 
      use groups
      use commonvariables
      implicit none
      integer :: j,k,N,M0
      real(kind=DP), dimension(0:num_cusps-1)::e
                                !      real(kind=DP), dimension(1:M0*num_cusps),intent(out)::A,B
      real(kind=DP), dimension(1:M0*num_cusps)::A,B
      complex(kind=DP), dimension(:),allocatable::C

      complex(kind=DP), dimension(M0*num_cusps,M0*num_cusps),               &
     &     intent(in)::VnlCp
      complex(kind=DP), dimension(:,:),allocatable :: Utmp
      complex(kind=DP)::tmp,c1
      integer :: num_spec,step,cusp,NN
                                !      write(*,*) 'in solve_syst size(A)=',size(A)
      N=M0*num_cusps
                                !      write(*,*) 'M0*num_cusps=',M0,'*',num_cusps,'=',N
                                ! the number of independent cusps, e.g. that are not 
                                ! set to  c_i(k)=e(i)*c_0(k)
      num_spec=count(abs(e(:))>0)-1
!      write(*,*) 'num_spec=',num_spec
                                ! we also set c_0(1)=1
      NN=N-num_spec*M0-1        ! the number of equations left
      allocate(C(1:NN))
      allocate(Utmp(NN,NN+1))
      c1=dcmplx(1.0_DP,0.0_DP)
      if(num_cusps.gt.1) then 
         do j=1,NN
            tmp=dcmplx(0.0_DP,0.0_DP)
            do cusp=0,num_cusps-1
               tmp=tmp+dcmplx(e(cusp),0.0D0)*                                   &
     &              VnlCp(j+1+num_spec*M0,1+cusp*M0)
            enddo
            Utmp(j,NN+1)=-c1*tmp
         enddo
         do j=1,NN
                                ! remember we only add the other cusps to the first
            do k=1,M0-1
               tmp=dcmplx(0.0_DP,0.0_DP)
               do cusp=0,num_cusps-1
                  tmp=tmp+dcmplx(e(cusp),0.0D0)*                                     &
     &                 VnlCp(j+1+num_spec*M0,1+k+cusp*M0)
               enddo
               Utmp(j,k)=tmp
            enddo
                                !         do k=1,NN-M0+1
            step=0
            do cusp=0,num_cusps-1
               if(e(cusp).eq.0.0) then
                  Utmp(j,M0*(1+step):M0*(2+step)-1)=                              &
     &                 VnlCp(j+1+num_spec*M0,cusp*M0+1:(cusp+1)*M0)
                  step=step+1
               endif
            enddo
         enddo
      else
         !! We only set c1=1
         do j=1,NN
            do k=1,NN
               Utmp(j,k)=VnlCp(j+1,k+1)
!               write(*,*) 'Utmp(',j,',',k,')=',Utmp(j,k)
            enddo
            Utmp(j,NN+1)=-c1*VnlCp(j+1,1)
         enddo
      endif
!      write(*,*) 'NN=',NN
!      write(*,*) 'M0=',M0
!      do j=1,NN
!         write(*,*) 'Utmp(',j,',',j,')=',Utmp(j,j)
!      enddo
!      do j=1,NN
!         write(*,*) 'Utmp(',j,',NN+1)=',Utmp(j,NN+1)
!      enddo
      call SMAT_cplx(Utmp,NN,C)      
!                                      write(*,*) 'C(1)=',C(1)
!                                      write(*,*) 'C(2)=',C(2)
!                                      write(*,*) 'C(3)=',C(3)
!                                      write(*,*) 'C(4)=',C(4)
!                                      write(*,*) 'C(5)=',C(5)
                                !      write(*,*) 'U=',Utmp
      A(1)=real(c1,DP)
      A(2:M0)=REAL(C(1:M0-1),DP)
      B(1)=dimag(c1)
      B(2:M0)=dimag(C(1:M0-1))
      step=0
      do cusp=1,num_cusps-1
         if(e(cusp).eq.0.0) then
            A(M0*cusp+1:M0*cusp+M0)=REAL(C(M0*(1+step):                     &
     &           M0*(2+step)-1),DP)
            B(M0*cusp+1:M0*cusp+M0)=DIMAG(C(M0*(1+step):M0*(2+step)-1))
            step=step+1
         else
            A(M0*cusp+1)=real(e(cusp),DP)*A(1)
            B(M0*cusp+1)=real(e(cusp),DP)*B(1)
            A(M0*cusp+2:M0*cusp+M0)=real(e(cusp),DP)*REAL(C(1:M0-1),DP)
            B(M0*cusp+2:M0*cusp+M0)=real(e(cusp),DP)*DIMAG(C(1:M0-1))
         endif
      enddo
      deallocate(C)
      deallocate(Utmp)
      end subroutine solve_system_Complex6



!!! Sae as above but uses even odd symmetry so deals only with positive coefficients
!! Use LU-factorization through LAPACK to solve for several symmetries at once
      subroutine solve_system_real_use_sym_lap(Vnl,CY,M0,use_lap_in)
                                !,                   &
                                !     &     param,e_set,num_set,c_set,which_c) 
      use groups
      use commonvariables
      use characters 
      implicit none
      integer :: j,k,M0
      logical,optional::use_lap_in
                                !      integer,optional::param   !!! determines which normalization to use
                                !!! par=0  sets only c(1)=1
                                !!! par=1 sets c(1)=1 and c(N)=0
                                !!! par=2 sets c(1)=0 and c(N)=1
      integer::par
      real(kind=DP),dimension(1:num_cusps,1:M0),intent(out)::CY
      real(kind=DP),dimension(1:num_cusps*M0,1:num_cusps*M0)::V,Vnl
!      real(DP),dimension(:,:),allocatable::RHS
!      real(kind=DP), dimension(:,:),allocatable :: VnlCpI
      real(kind=DP),dimension(1:M0)::c
      real(kind=DP), dimension(:),allocatable::D
      real(kind=DP)::tmp
      integer :: NN
!      integer::offs,offs_row,offs_col,I_set
      logical::deb
      real(DP)::rinv_ev(1:num_cusps)
      integer,dimension(1:M0*num_cusps)::col_count,new_count
      integer::m,r_offs,c_offs,jj,rcusp,ccusp,i
      integer::sym_ev  !!+-1 cos/sin , i.e. c(-n)=+-c(n)
      integer,allocatable,dimension(:)::IPIV
      integer::INFO,NRHS,unset_cusps
      integer::n_tmp,icusp,ia1,ia2,ia3,ia4,ib1,ib2,ib3,ib4
!      V(:,:)=Vnl(:,:)
!c$$$      if(deb) then 
!c$$$         do j=1,size(V,1)
!c$$$            write(*,*) 'V0(',j,')=',Vnl(j,j)
!c$$$         enddo
!c$$$      endif
      deb=.false.
!      deb=.true.
      use_lap=.false.
      if(present(use_lap_in)) then 
         use_lap=use_lap_in
      endif
      sym_ev=0
      if(use_sym_even) then
         sym_ev=1
      elseif(use_sym_odd) then
         sym_ev=-1
      endif
      par=1
                                ! Any normalization should be determined before here
      if(.not.(set_normalization)) then
         call set_norm(0)
      endif
                                !! e_set is the vector of coefficients which are set (and the location is stored
                                !! in which_c
                                !! start by assuming that no coefficients are set
      call get_alphas()
      do j=1,num_cusps*M0
         col_count(j)=1
      enddo
      c(:)=0.0_DP
      do j=1,num_set
         if(which_c(j,2).ne.0) then  
            c(which_c(j,2))=real(c_set(j),DP)
            col_count(which_c(j,2))=0
         endif
      enddo
      do j=1,num_cusps
         if(abs(inv_ev(j))<1E-10_DP) then 
            rinv_ev(j)=0.0D0
         else
            rinv_ev(j)=real(inv_ev(j),DP)
         endif
      enddo
!      V(1:M0,1:M0)=Vnl(1:M0,1:M0)
      !!! WE now want to account for any set cusps
      r_offs=0

      do rcusp=1,num_cusps
         if(num_cusps.eq.1) then
            V=Vnl
            exit
         endif
         if((rcusp.gt.1).and.(abs(inv_ev(rcusp)).gt.1.0E-10)) then
            r_offs=r_offs+1
            cycle
         endif                  !! rows
         V((rcusp-1-r_offs)*M0+1:(rcusp-r_offs)*M0,1:M0)=                    &
     &        Vnl((rcusp-1)*M0+1:rcusp*M0,1:M0)
          do ccusp=2,num_cusps
            if(rinv_ev(ccusp).ne.0.0D0) then
               do k=1,M0
                  m=k+(ccusp-1)*M0            
                  do i=M0*(rcusp-1-r_offs)+1,M0*(rcusp-r_offs)                   
                     if(col_count(i).ne.0) then
                        V(i,k)=V(i,k)+rinv_ev(ccusp)
     &                       *Vnl(i+r_offs*M0,m)
                        col_count(m)=0
                     endif
                  enddo
               enddo
            endif
         enddo
         c_offs=0
         do ccusp=2,num_cusps
            if(rinv_ev(ccusp).ne.0.0D0) then
               c_offs=c_offs+1
               cycle
            endif
            ia1=(rcusp-1-r_offs)*M0+1; ib1=(rcusp-r_offs)*M0
            ia2=(ccusp-1-c_offs)*M0+1; ib2=(ccusp-c_offs)*M0
            ia3=(rcusp-1)*M0+1;        ib3=rcusp*M0
            ia4=(ccusp-1)*M0+1;        ib4=ccusp*M0
            V(ia1:ib1,ia2:ib2)=Vnl(ia3:ib3,ia4:ib4)           
  !          V(r_offs+1:r_offs+M0,(ccusp-1)*M0+1:ccusp+M0)=                             &
  !   &           Vnl((rcusp-1)*M0+1:rcusp*M0,(ccusp-1)*M0+1:ccusp*M0)
         enddo
      enddo
                                !! par <0 => set c(|par|)=1, c1=0
                                !! par >=1 => set c(par)=0, c1=1
      if(abs(which_c(num_set,2)).gt.M0) then
         write(*,*) 'Setting too large coefficient!'
         write(*,*) 'set_coefs=',which_c(:,2)
         stop
      endif
      NN=count(col_count(:)>0)
      unset_cusps=count(abs(rinv_ev(:))<1E-10_DP)+1
      if(deb) then
         write(*,*) 'par=',par, 'NN=',NN
         write(*,*) 'col_cnt>0=',count(col_count(:)>0)

         write(*,*) 'c(1)=',c(1)
         write(*,*) 'c(par)=',c(abs(par))
         write(*,*) 'c=',c(:)
         if(ALLOCATED(which_c)) then
            write(*,*) 'which_c(:)=',which_c(:,2)
            do j=1,num_set
               if(which_c(j,2).gt.0) then 
                  write(*,*) 'c(',which_c(j,2),')=',c(which_c(j,2))
               endif
            enddo
         endif
      endif
      if(deb) then
         write(*,*) 'col_count=',col_count(:)
      endif
      new_count(:)=col_count(:)
!     ! We must update the col_count to the new setting where 
!     ! we contracted the system into V instead of Vnl
      do ccusp=num_cusps-1,2,-1
         if(abs(inv_ev(ccusp)).ne.0.0D0) then 
           ! remove section:   (ccusp-1)*MM0+1:ccusp*MM0
            new_count((ccusp-1)*M0+1:num_cusps*M0-M0)=                           &
     &           new_count(ccusp*M0+1:num_cusps*M0)
            new_count((num_cusps-1)*M0+1:num_cusps*M0)=0
          endif
      enddo
      if(deb) then
         write(*,*) 'new_count=',NEW_count(:)
      endif
!c$$$      if(deb) then 
!c$$$         do j=1,NN
!c$$$            write(*,*) 'V1(',j,')=',V(j,j)
!c$$$         enddo
!c$$$      endif
!      allocate(RHS(1:NN,1:NRHS))
      r_offs=0
      do j=1,num_cusps*M0
         if(new_count(j).eq.0) then
            r_offs=r_offs+1
            cycle
         endif
         tmp=0.0D0
         do k=1,M0
            if(new_count(k).eq.0) then
               tmp=tmp-c(k)*V(j,k)
            endif
         enddo
         c_offs=0
         do k=1,unset_cusps*M0
            if(new_count(k).eq.0) then
               c_offs=c_offs+1
               cycle
            endif
            V(j-r_offs,k-c_offs)=V(j,k)
         enddo
         V(j-r_offs,NN+1)=tmp
      enddo 
!c$$$      if(deb) then 
!c$$$         do j=1,NN
!c$$$            write(*,*) 'V2(',j,')=',V(j,j)
!c$$$         enddo
!c$$$      endif
      allocate(D(1:NN))
!! If we have defined use_lapack
#ifdef use_lapack
      if(use_lap) then 
         if(allocated(IPIV)) deallocate(IPIV)
         allocate(IPIV(1:NN))
         call DGETRF( NN, NN, V(1:NN,1:NN), NN, IPIV(1), INFO )
         call dgetrs( 'N',NN,1, V(1:NN,1:NN), NN, IPIV(1), V(1:NN,NN+1),         &
     &        NN, INFO )
         if(allocated(IPIV)) deallocate(IPIV)
         D(1:NN)=V(1:NN,NN+1)
      else
         call SMAT(V(1:NN,1:NN+1),NN,D(1:NN))     
      endif
#else
! Otherwise we just use the standard thing
      call SMAT(V(1:NN,1:NN+1),NN,D(1:NN))     
#endif   

                                !!! Assign the Coefficient-vectors
      r_offs=0
      CY(:,:)=0.0_DP
      do j=1,num_cusps*M0
         if(col_count(j).eq.0) then
            r_offs=r_offs+1
            cycle
         endif 
         k=floor(real(j-1,DP)/real(M0,DP))
         jj=j-k*M0
         if(j-r_offs.gt.NN) then
            write(*,*) 'ERROR: NN=',NN
            write(*,*) 'j-r_offs=',j,'-',r_offs
            stop
         endif
         CY(k+1,jj)=D(j-r_offs)
      enddo
      if(ALLOCATED(D)) deallocate(D)
                                !! Assign the special values of CY
      do j=1,M0
         if(col_count(j).eq.0) then
!            if(deb) then
!               write(*,*) 'set c(',j,')=',c(j)
!            endif
            CY(1,j)=c(j)
         endif
      enddo
                                !!! Assign any cusps that are set
      do ccusp=2,num_cusps
         if(abs(inv_ev(ccusp)).lt.1.0E-12) cycle
!         if(deb) then
!            write(*,*) 'set cusp ',ccusp
!            write(*,*) 'inv_ev(cusp)=',inv_ev(ccusp)
!         endif
         do j=1,M0
            CY(ccusp,j)=real(inv_ev(ccusp),DP)*CY(1,j)
         enddo
      enddo
      end subroutine solve_system_real_use_sym_lap




      subroutine solve_system_cplx_use_sym_lap(Vnl,CY,M0,use_lap_in)
      use groups
      use commonvariables
      use characters 
      implicit none
      integer :: j,k,M0
      integer::par
      complex(kind=DP),dimension(1:num_cusps,-M0:M0),intent(out)::CY
      complex(kind=DP),dimension(1:num_cusps*(2*M0+1),                   &
     &     1:num_cusps*(2*M0+1))::V,Vnl
      complex(kind=DP),dimension(-M0:M0)::c
      complex(kind=DP), dimension(:),allocatable::D
      real(kind=DP)::tmp
      integer :: NN
      logical::deb
      logical,optional::use_lap_in
      integer,dimension(1:(2*M0+1)*num_cusps)::col_count,new_count
      integer::m,r_offs,c_offs,jj,rcusp,ccusp,i
      integer::sym_ev  !!+-1 cos/sin , i.e. c(-n)=+-c(n)
      integer,allocatable,dimension(:)::IPIV
      integer::INFO,NRHS,unset_cusps,MM0,n_tmp,icusp
      integer::n,ia1,ia2,ia3,ia4,ib1,ib2,ib3,ib4
!      deb=.true.
      deb=.false.
      deb=.true.
!      V=Vnl
      use_lap=.false.!! Do not Use LAPACK by default
      if(present(use_lap_in)) then
         use_lap=use_lap_in
      endif
      sym_ev=0
      MM0=2*M0+1
      if(use_sym_even) then
         sym_ev=1
      elseif(use_sym_odd) then
         sym_ev=-1
      endif
      par=1
      if(.not.(set_normalization)) then
         call set_norm(0)
      endif
      if(deb) then
         write(*,*) 'inv_ev=',inv_ev(:)
         do icusp=1,num_cusps
           do n=0,5
               n_tmp=n+M0+1+(icusp-1)*(2*M0+1)
               write(*,*) 'Vnl',n_tmp,')=',Vnl(n_tmp,N_tmp)
            enddo
         enddo 
      endif
      call get_alphas()
      do j=1,num_cusps*MM0
         col_count(j)=1
      enddo
      c(:)=0.0_DP
      do j=1,num_set
         c(which_c(j,2))=c_set(j)
         col_count(which_c(j,2)+M0+1)=0
         if(which_c(j,2).eq.0) then  
!     ! We are looking for cusp forms
!     ! So we should set zeroth coefficient to zero at other cusps too
!     ! if necessary
            do ccusp=2,num_cusps
               m=M0+1+(ccusp-1)*MM0
               if(alphas(ccusp).eq.0.0D0) then
                  col_count(m)=0
               endif
            enddo

         endif
      enddo
      !!! WE now want to account for any set cusps
      r_offs=0
      do rcusp=1,num_cusps
!         write(*,*) 'inv_ev(',rcusp,')=',inv_ev(rcusp)
         if(num_cusps.eq.1) exit
         if((rcusp.gt.1).and.(abs(inv_ev(rcusp)).gt.1.0E-10)) then
            r_offs=r_offs+1
            cycle
         endif                  !! rows
         V((rcusp-1-r_offs)*MM0+1:(rcusp-r_offs)*MM0,1:MM0)=            &
     &        Vnl((rcusp-1)*MM0+1:rcusp*MM0,1:MM0)
         do ccusp=2,num_cusps
            if(abs(inv_ev(ccusp)).ne.0.0D0) then
               do k=1,MM0
                  m=k+(ccusp-1)*MM0
                  do i=MM0*(rcusp-1-r_offs)+1,MM0*(rcusp-r_offs)                   
                     V(i,k)=V(i,k)+inv_ev(ccusp)*Vnl(i+r_offs*MM0,m)
                     col_count(m)=0
                  enddo
               enddo
            endif
         enddo
         c_offs=0
         do ccusp=2,num_cusps
            if(abs(inv_ev(ccusp)).ne.0.0D0) then
               c_offs=c_offs+1
               cycle
            endif
            ia1=(rcusp-1-r_offs)*MM0+1; ib1=(rcusp-r_offs)*MM0
            ia2=(ccusp-1-c_offs)*MM0+1; ib2=(ccusp-c_offs)*MM0
            ia3=(rcusp-1)*MM0+1;        ib3=rcusp*MM0
            ia4=(ccusp-1)*MM0+1;        ib4=ccusp*MM0
            if(deb) then
               write(*,*) 'rcusp,ccusp=',rcusp,ccusp
               write(*,*) 'R_offs=',r_offs
               write(*,*) 'C_offs=',c_offs
               write(*,*) 'ia1:ib1=',ia1,':',ib1
               write(*,*) 'ia2:ib2=',ia2,':',ib2
               write(*,*) 'ia3:ib3=',ia3,':',ib3
               write(*,*) 'ia4:ib4=',ia4,':',ib4
               write(*,*) 'V(ia1:ib1,ia2:ib2)=Vnl(ia3:ib3,ia4:ib4)'
            endif
            V(ia1:ib1,ia2:ib2)=Vnl(ia3:ib3,ia4:ib4)
         enddo
      enddo
      if(abs(which_c(num_set,2)).gt.(M0+1)) then
         write(*,*) 'Setting too large coefficient!'
         write(*,*) 'set_coefs=',which_c(:,2)
         stop
      endif
      NN=count(col_count(:)>0)
      unset_cusps=count(abs(inv_ev(:))<1E-10_DP)+1
!      write(*,*) 'unset_cusps=',unset_cusps
      if(deb) then
         write(*,*) 'use_lap=',use_lap
         write(*,*) 'par=',par, 'NN=',NN
         write(*,*) 'col_cnt>0=',count(col_count(:)>0)
!         do j=1,num_set
!            write(*,*) 'c(',which_cj,')=',c(j)
!         enddo
         if(ALLOCATED(which_c)) then
            write(*,*) 'which_c(:)=',which_c(:,2)
            do j=1,num_set
               write(*,*) 'c(',which_c(j,2),')=',c(which_c(j,2))
            enddo
         endif
      endif
      if(deb) then
         write(*,*) 'col_count=',col_count(:)
      endif
      new_count(:)=col_count(:)
!     ! We must update the col_count to the new setting where 
!     ! we contracted the system into V instead of Vnl
      do ccusp=num_cusps-1,2,-1
         if(abs(inv_ev(ccusp)).ne.0.0D0) then 
           ! remove section:   (ccusp-1)*MM0+1:ccusp*MM0
            new_count((ccusp-1)*MM0+1:num_cusps*MM0-MM0)=                           &
     &           new_count(ccusp*MM0+1:num_cusps*MM0)
            new_count((num_cusps-1)*MM0+1:num_cusps*MM0)=0
          endif
      enddo
      if(deb) then
         write(*,*) 'new_count=',NEW_count(:)
      endif
      r_offs=0
         do j=1,unset_cusps*MM0
         if(new_count(j).eq.0) then
            r_offs=r_offs+1
            cycle
         endif
         tmp=0.0D0
!! Only coefficients at the first cusp or zeroth coefficients at other cusps are set individually
         do k=1,MM0
            if(new_count(k).eq.0) then
               tmp=tmp-c(k-(M0+1))*V(j,k)
            endif
         enddo
         c_offs=0
         do k=1,unset_cusps*MM0
!         do k=1,num_cusps*MM0
            if(new_count(k).eq.0) then
!               write(*,*) 'Remove colnr. ',k
               c_offs=c_offs+1
               cycle
            endif
            V(j-r_offs,k-c_offs)=V(j,k)
         enddo
         V(j-r_offs,NN+1)=tmp
      enddo     
      if(deb) then
         write(*,*) 'inv_ev=',inv_ev(:)
!         do n_tmp=1,NN
!            write(*,*) 'V',n_tmp,')=',V(n_tmp,N_tmp)
!         enddo 
      endif
      allocate(D(1:NN))
!! If we have defined use_lapack
#ifdef use_lapack
      if(use_lap) then 
         if(allocated(IPIV)) deallocate(IPIV)
         allocate(IPIV(1:NN))
!         D(1:NN)=V(1:NN,NN+1)
         call ZGETRF( NN, NN, V(1:NN,1:NN), NN, IPIV(1), INFO )
         call zgetrs( 'N',NN,1, V(1:NN,1:NN), NN, IPIV(1), V(1:NN,NN+1),         &
     &        NN,INFO)
!         call zgetrs( 'N',NN,1, V(1:NN,1:NN), NN, IPIV(1), D,NN, INFO )
         if(allocated(IPIV)) deallocate(IPIV)
         D(1:NN)=V(1:NN,NN+1)
      else
         call SMAT_cplx(V(1:NN,1:NN+1),NN,D(1:NN))     
      endif
#else
! Otherwise we just use the standard thing
      call SMAT_cplx(V(1:NN,1:NN+1),NN,D(1:NN))     
#endif   
!c$$$      write(*,*) 'inv_ev(:)=',inv_ev(:)
!      do j=-3,3
!         write(*,*) 'D(',j,')=',D(M0+1+j)
!      enddo
!c$$$                                !!! Assign the Coefficient-vectors
      if(abs(D(1)).eq.0) stop
      r_offs=0
      CY(:,:)=0.0_DP
      do j=1,num_cusps*MM0
         if(col_count(j).eq.0) then
            r_offs=r_offs+1
            cycle
         endif 
         k=floor(real(j-1,DP)/real(2*M0+1,DP))
         jj=j-k*MM0
         if(j-r_offs.gt.NN) then
            write(*,*) 'ERROR: NN=',NN
            write(*,*) 'j-r_offs=',j,'-',r_offs
            stop
         endif
         CY(k+1,jj-(M0+1))=D(j-r_offs)
      enddo
      if(ALLOCATED(D)) deallocate(D)
      !! Assign the special values of C in the first component
!      do j=1,2*M0+1
!         if(col_count(j).eq.0) then
!            CY(1,j-(M0+1))=c(j-(M0+1))
!         endif
!      enddo
      do j=1,num_set
        if(which_c(j,2).ne.0) then
           CY(1,which_c(j,2))=c_set(j)
         endif
      enddo
                                !!! Assign any cusps that are set
      do ccusp=2,num_cusps
         if(abs(inv_ev(ccusp)).lt.1.0E-12) cycle
!         if(deb) then
!            write(*,*) 'set cusp ',ccusp
!            write(*,*) 'inv_ev(cusp)=',inv_ev(ccusp)
!         endif
         do j=-M0,M0
            CY(ccusp,j)=real(inv_ev(ccusp),DP)*CY(1,j)
         enddo
      enddo
       end subroutine solve_system_cplx_use_sym_lap


      !! Primitive routines for system solving

                                !! Use gauss elimination with partial pivoting to solve
                                !! AX=B, where
                                !! A=U(1:N,1:N)
                                !! X=C(1:N)
                                !! B=U(1:N,N+1)

      SUBROUTINE SMAT_cplx(U,N,C)
      use commonvariables

      implicit none
      integer,intent(in)::N
                                !      PARAMETER(NMAX=190,NPAX=95,NCAX=95)
                                !	      complex(kind=DP), dimension(N,N+1) ::U
      complex(kind=DP), dimension(1:N,1:N+1) ::U
      complex(kind=DP), dimension(1:N) :: C
      integer :: M,MAXI,J,K
      complex(kind=DP) :: TT,TEMP2,tmpu(1:N,1:N+1)
      real(DP)::temp
      integer,dimension(1:N)::PIV,USED
      if(size(U,1).ne.N) then
         write(*,*) 'NN=',N
         write(*,*) 'size(vnlCp)=',size(U,1),size(U,2)
         stop
      endif
      used(:)=0
                                !	      write(*,*) 'entering smat> N=',N
                                !              write(*,*) 'size(vnlCp)=',size(U,1),size(U,2)
      DO m=1,N
         TEMP=0.0_DP 
         MAXI=1
                                !         write(*,*) 'U(m,m)=',U(m,m)
!c     Locate maximum
         do j=1,N
            if(USED(j).ne.0) cycle
            if((abs(U(j,M)).le.TEMP)) cycle 
            MAXI=j
            TEMP=abs(U(J,M))
         enddo
                                !         write(*,*) 'maxi=',maxi
                                !         write(*,*) 'maxloc=',loc(1)
                                !         loc=maxloc(abs(U(1:N,M)),USED==0)
                                !         maxi=loc(1)
         PIV(m)=maxi
         USED(MAXI)=1
         TEMP2=U(Maxi,M)
                                !         U(Maxi,M)=dcmplx(1.0_DP,0.0_DP)
         if(abs(TEMP2).eq.0.0) then 
            write(*,*) 'ERROR: pivot(',M,') == 0, system bad!!!'
                                !            TEMP2=dcmplx(1.0E-32_DP,0.0_DP)
            stop
         endif
         do j=M+1,N+1
            U(Maxi,j)=U(Maxi,j)/TEMP2
         enddo
                                !         U(Maxi,M+1:N+1)=U(Maxi,M+1:N+1)/TEMP2
                                ! eliminate from all other rows
!         DO J=1,Maxi-1
         DO J=1,maxi-1
            TT=U(J,M)
            do K=M+1,N+1
               U(J,k)=U(j,k)-U(Maxi,K)*TT
            enddo            
         enddo
         DO J=Maxi+1,N
            TT=U(J,M)
            do K=M+1,N+1
               U(J,k)=U(j,k)-U(Maxi,K)*TT
            enddo
         enddo
      enddo
                                !! now remember we have pivot for x_j at PIV(j) not j
      DO M=1,N
         C(M)=U(PIV(M),N+1)
      enddo
                                !		write(*,*) 'leaving smat'
      RETURN      
      END subroutine SMAT_cplx


      SUBROUTINE SMAT_real(U,N,C)
      use commonvariables
      implicit none
      real(kind=DP), dimension(N,N+1) ::U
      real(kind=DP), dimension(1:N) :: C
      integer :: M,MAXI,J,K,N
      real(kind=DP) :: TT,TEMP2
      real(DP)::temp
      integer,dimension(1:N)::PIV,USED
      integer,dimension(1)::loc
      used(:)=0
                                !      write(*,*) 'N=',N
      DO m=1,N
         TEMP=0.0_DP 
         MAXI=0
!c     Locate maximum
         do j=1,N
!            write(*,*) 'U(',j,',',m,')=',U(j,m)
            if(USED(j).ne.0) cycle
            if((abs(U(j,M)).le.TEMP)) cycle 
            MAXI=j
            TEMP=abs(U(J,M))
           
         enddo
                                !         write(*,*) 'maxi=',maxi
                                !         write(*,*) 'maxloc=',loc(1)
         loc=maxloc(abs(U(1:N,M)),USED==0)
         maxi=loc(1)
         PIV(m)=maxi
         USED(MAXI)=1
         TEMP2=U(Maxi,M)
                                !         U(Maxi,M)=dcmplx(1.0_DP,0.0_DP)
         if(abs(TEMP2).eq.0.0) then 
            write(*,*) 'ERROR: pivot(',M,') == 0, system bad!!!'
                                !            TEMP2=dcmplx(1.0E-32_DP,0.0_DP)
            stop
         endif
         U(Maxi,M+1:N+1)=U(Maxi,M+1:N+1)/TEMP2
                                ! eliminate from all other rows
         DO J=1,Maxi-1
            TT=U(J,M)
            do K=M+1,N+1
               U(J,k)=U(j,k)-U(Maxi,K)*TT
            enddo            
         enddo
         DO J=Maxi+1,N
            TT=U(J,M)
            do K=M+1,N+1
               U(J,k)=U(j,k)-U(Maxi,K)*TT
            enddo
         enddo
      enddo
                                !! now remember we have pivot for x_j at PIV(j) not j
                                !      write(*,*) 'PIV=',PIV(1:N)
      DO M=1,N
         C(M)=U(PIV(M),N+1)
      enddo
      RETURN      
      END subroutine SMAT_real

      

      SUBROUTINE SMAT_cplx2(U,N,C)
      use commonvariables
      implicit none
                                !      PARAMETER(NMAX=190,NPAX=95,NCAX=95)
      complex(kind=DP), dimension(N,N+1) ::U
      complex(kind=DP), dimension(1:N) :: C
      integer :: M,MAXI,J,K,N
      complex(kind=DP) :: TT,T,TEMP2
      real(DP)::temp
                                !      integer,dimension(1) :: loc
      DO m=1,N
         TEMP=0.0_DP
         MAXI=1
         DO J=M,N
            if(TEMP.lt.ABS(U(J,M))) then
               TEMP=abs(U(J,M))
               MAXI=J
            endif
         enddo
         if(maxi.ne.m) then
            do J=1,N+1
               T=U(MAXI,J)
               U(MAXI,J)=U(M,J)
               U(M,J)=T
            enddo
         endif
                                ! normalize s.t. max(j=m+1,..,N){|U(m,j)|}=1
                                !  use that element as pivot                  
         TEMP2=U(M,M)
         U(M,M)=dcmplx(1.0_DP,0.0_DP)
         if(abs(TEMP2).eq.0.0) then 
            write(*,*) 'ERROR: pivot(',M,') == 0, system bad!!!'
            TEMP2=dcmplx(1.0E-32_DP,0.0_DP)
            stop
         endif
         DO 1300 J=M+1,N+1
                                !            if(j.ne.M) then
            U(M,J)=U(M,J)/TEMP2
                                !            endif
 1300    CONTINUE
                                ! eliminate from all other rows
         DO J=1,M-1
            TT=U(J,M)
            U(J,M)=dcmplx(0.0_DP,0.0_DP)
            do K=M+1,N+1
               U(J,k)=U(j,k)-(U(M,K)*TT)
            enddo
         enddo
         DO J=M+1,N
            TT=U(J,M)
            U(J,M)=dcmplx(0.0_DP,0.0_DP)
            do K=M+1,N+1
               U(J,k)=U(j,k)-(U(M,K)*TT)
            enddo
         enddo

      enddo

      DO 2001 M=1,N
         C(M)=U(M,N+1)
 2001 CONTINUE
      RETURN      
      END subroutine SMAT_cplx2

      subroutine clean_sys_solve()
      if(allocated(c_set)) deallocate(c_set)
      if(allocated(inv_ev)) deallocate(inv_ev)
      if(allocated(which_c)) deallocate(which_c)
      end subroutine clean_sys_solve


      end module
