!c!!  Contains  algorithms for Confluent hypergeometric functions

      module confluent
      use commonvariables
  !    PRIVATE   !!! No variables should be accessible
      private
      real(DP):: eps=1.0E-13_DP
      real(DP),save::c_rx
      integer,save::k_rx
      integer,parameter::N_max_pow=500
      integer,parameter::N_max_as=1500
      integer,parameter::N_max_recursive=1500
      real(DP),dimension(:,:),allocatable,save::A_uk
      integer,parameter::N_uk=20  ! The max number of terms in u_k
      real(DP),dimension(:,:),allocatable,save::a_k,b_k
      complex(DP),dimension(:),allocatable,save::lambda
      real(DP)::a_set_l,c_set_l
      logical,save::lambda_set
      logical,save::coeff_psi_pow3_is_set
      real(DP),save::a_set
      public::psi_r_c,psi_r,phi_pow_cf_c,psi_pow2_r_c,phi_as_cf_r_c
!      public::bessk_real
      contains
!c     psi_r_c(a,c,x)  for a,c=real and x complex
!c     psi_r(a,c,x)  for a,c,x=real and x>0

!c     ! Confluent hypergeometric U=Psi 
!c     ! for arguments: a,c,x real and x>0
      function psi_r(a,c,x)
      use gamma_function
      implicit none
      integer::N,j
      real(kind=DP)::a,c,sum,psi_r,psi,psi0
      real(kind=DP)::x,term,jj,factor
      integer::j_start
      integer,parameter::j_max=350
!      write(*,*) 'here! A=',A,' C=',c,' x=',x
      if(x.lt.0.0_DP) then
         !In general this becomes a complex number
         write(*,*) 'x<0 not good'
         stop
      endif
      if(a.eq.0.0_DP) then
         psi_r=1.0_DP      
         return
      endif
      if((a.eq.-0.5D0).and.(c.eq.0.5D0)) then
         psi_r=sqrt(x)
         return
      endif
      if(x.eq.0.0_DP) then
         psi_r=gamma_r(1.0_DP-c)/gamma_r(a-c+1.0_DP)
      elseif((x.lt.5.0_DP).and.(abs(abs(c)-real(floor(abs(c)),kind=DP))     &
     &        .gt.0.1_DP)) then
                                ! if c is too close to an integer
         psi_r=psi_pow_cf_r(a,c,x,J)
                                !   write(*,*) 'psi_pow_cf_r=',psi_r
      elseif(x.lt.35.0_DP) then
         if(x.gt.20.0_DP) then 
            if(abs(a).lt.0.1_DP) then
               j_start=15
            else
               j_start=25
            endif
         elseif(x.gt.10.0_DP) then
            if(abs(a).lt.0.01_DP) then
               j_start=20
            elseif(abs(a).lt.0.1_DP) then
               j_start=20
            else
               j_start=40
            endif
         elseif(x.gt.5.0_DP) then
            if(abs(a).lt.0.01_DP) then
               j_start=30
            elseif(abs(a).lt.0.1_DP) then
               j_start=40
            else
               j_start=60
            endif
         elseif(x.gt.4.0_DP) then
            if(abs(a).lt.0.01_DP) then
               j_start=40
            elseif(abs(a).lt.0.1_DP) then
               j_start=50
            else            
               j_start=75
            endif
         elseif(x.gt.3.0_DP) then
            if(abs(a).lt.0.01_DP) then
               j_start=55
            elseif(abs(a).lt.0.1_DP) then
               j_start=75
            else
               j_start=90 
            endif
         elseif(x.gt.2.0_DP) then
            if(abs(a).lt.0.01_DP) then
               j_start=105
            elseif(abs(a).lt.0.1_DP) then
               j_start=110
            else
               j_start=140
            endif
         else
            if(abs(a).lt.0.000001_DP) then
               j_start=105
            elseif(abs(a).lt.0.0001_DP) then
               j_start=140
            elseif(abs(a).lt.0.01_DP) then
               j_start=180
            else
               j_start=300
            endif    
         endif
            do j=j_start,j_max
            psi=psi_recursive_r(a,c,x,j)
            if(abs(psi-psi0).lt.1.0e-12_DP) then
               exit
            endif
            psi0=psi
         enddo
         psi_r=psi           
      else
         psi_r=psi_asymp_pow_r(a,c,x)
      endif
      end function  



!c     ! Confluent hypergeometric U=Psi 
!c     ! for arguments: a,c real and x complex
      function psi_r_c(a,c,x)
      use gamma_function
      implicit none
      integer::N,j,j_start
      real(kind=DP)::a,c
      complex(kind=DP)::psi_c,psi0_c,x
      complex(kind=DP)::psi_r_c
      logical::deb
!      deb=.true.
!         write(*,*) 'a,c=',a,c
!         write(*,*) 'x=',x
!         write(*,*) 'abs(x)=',abs(x)
      !!! have to see if a is integer or half-integer
      if(real(x).lt.0.0_DP) then         
         psi_r_c=psi_r_c_neg(a,c,x)
      elseif((a.lt.0.0_DP).and.(floor(abs(a)).eq.ceiling(abs(a)))) then
         !!a is a negative integer
         if(c.eq.0.5D0) then
            psi_r_c=dcmplx(sqrtPI/gamma_r(a+0.5_DP),0.0_DP)*                &
     &           phi_int_r_c(a,c,x)
         else
            psi_r_c=dcmplx(gamma_r(1.0_DP-c)/gamma_r(a-c+1.0_DP),0.0_DP) &
     &           *phi_int_r_c(a,c,x)
         endif
      elseif((a-c+1.0_DP.lt.0.0_DP).and.(floor(abs(a-c)).eq.               &
     &           ceiling(abs(a-c)))) then
                                !! a-c is a negative half-integer
         if(c.eq.0.5D0) then
            psi_r_c=dcmplx(gamma_r(-0.5_DP)/gamma_r(a),0.0_DP)*                       &
     &           phi_int_r_c(a+0.5_DP,1.5_DP,x)*sqrt(x)
         else
            psi_r_c=dcmplx(gamma_r(c-1.0_DP)/gamma_r(a),0.0_DP)*                       &
     &           phi_int_r_c(a-c+1.0_DP,2.0_DP-c,x)*                       &
     &           (x**dcmplx(1.0_DP-c,0.0_DP))
         endif         
      elseif((a.ne.0.0D0).and.((a.ne.-0.5D0).or.(c.ne.0.5D0))) then
         if(abs(x).gt.35.0_DP) then
            psi_r_c=psi_asymp_pow_r_c(a,c,x)            
         elseif((abs(x).gt.10.0_DP).or.(c.ne.0.5_DP)) then 
            if(abs(x).gt.30.0_DP) then
               j_start=14
            elseif(abs(x).gt.25.0D0) then
               j_start=16
            elseif(abs(x).gt.20.0D0) then
               j_start=18
            elseif(abs(x).gt.15.0D0) then
               j_start=22
            elseif(abs(x).gt.12.0D0) then
               j_start=25
            elseif(abs(x).gt.9.0_DP) then
               j_start=30
            elseif(abs(x).gt.7.0_DP) then
               j_start=36
            elseif(abs(x).gt.6.0_DP) then
               j_start=40
            elseif(abs(x).gt.5.0_DP) then           
               j_start=45
            endif
            psi0_c=dcmplx(1000.0_dp,0.0_DP)
            do j=j_start,N_max_recursive
               psi_c=psi_recursive_r_c(a,c,x,j)
               if(abs(psi_c-psi0_c).lt.1.0e-14_DP) then
                  exit
               endif
               psi0_c=psi_c
            enddo
 !           if(deb) then
 !              write(*,*) 'j=',j
 !              write(*,*) 'psi=',psi_c
 !           endif
            psi_r_c=psi_c
         elseif(abs(x).ne.0.0_DP) then
            psi_r_c=psi_pow3_r_c(a,c,x)         
  !          write(*,*) 'a,c,x=',a,c,x
  !          write(*,*) 'psi=',psi_r_c
         else
            psi_r_c=dcmplx(gamma_r(1.0_DP-c)/gamma_r(a-c+1.0_DP),0.0_DP)
         endif
      elseif((a.eq.-0.5_DP).and.(c.eq.0.5_DP)) then
         psi_r_c=sqrt(x)
      elseif(a.eq.0.0_DP) then
         psi_r_c=dcmplx(1.0_DP,0.0_DP)
      else
         write(*,*) 'something is wrong with the cases in psi_r_c!'
         stop
      endif      
    !  write(*,*) 'psi=',psi_r_c
      end function




   






!c     !!! USES THE FUNCTIONS BELOW


!c     !     a,c=real and x complex with negative real part
!c     ! Uses Kummers transformation to get back in the right half-plane
      function psi_r_c_neg(a,c,x)
      use gamma_function
      implicit none
      integer::N,j,j_start
      real(kind=DP)::a,c
      complex(DP)::psi_c,psi0_c,x
      complex(DP)::psi_r_c_neg
      ! The same alternatives goes as for the other case
!      write(*,*) 'a,c,x,abs(x)=',a,c,x,abs(x)
      if((a.lt.0.0_DP).and.(floor(abs(a)).eq.ceiling(abs(a)))) then
         !!a is negative integer
         if(c.eq.0.5_DP) then
            psi_r_c_neg=dcmplx(sqrtPI/gamma_r(a+0.5_DP),0.0_DP)*                &
     &           phi_pow_cf_r_c(0.5_DP-a,0.5_DP,-x)
         else
            psi_r_c_neg=dcmplx(gamma_r(1.0_DP-c)/gamma_r(a-c+1.0_DP),       &
     &           0.0_DP)*phi_pow_cf_r_c(c-a,c,-x)
         endif
         psi_r_c_neg=psi_r_c_neg*exp(x)
      elseif((a-c+1.0_DP.lt.0.0_DP).and.(floor(abs(a-c)).eq.               &
     &        ceiling(abs(a-c)))) then
         !! a-c+1 is a negative integer
!         write(*,*) 'should be here!'
         if(c.eq.0.5_DP) then
            psi_r_c_neg=dcmplx(gamma_r(-0.5_DP)/gamma_r(a),0.0_DP)*                       &
     &           phi_pow_cf_r_c(1.0_DP-a,1.5_DP,-x)*sqrt(x)
!            write(*,*) 'phi(',1.0_DP-a,1.5,-x,')=',phi_pow_cf_r_c(1.0_DP  &
!     &           -a,1.5_DP,-x)
         else
            psi_r_c_neg=dcmplx(gamma_r(c-1.0_DP)/gamma_r(a),0.0_DP)*                       &
     &           phi_pow_cf_r_c(1.0_DP-a,2.0_DP-c,-x)*                                  &
     &           (x**dcmplx(1.0_DP-c,0.0_DP))
         endif
         psi_r_c_neg=psi_r_c_neg*exp(x)
      elseif((a.ne.0.0_DP).and.((a.ne.-0.5_DP).or.(c.ne.0.5_DP))) then
         if(abs(x).gt.35.0_DP) then 
            write(*,*) 'Asymptot!'
            psi_r_c_neg=psi_asymp_pow_r_c_neg(a,c,x)
            write(*,*) 'psi_r_c(',a,c,x,')=',psi_r_c_neg
         elseif(abs(x).gt.10.0_DP) then 
            if(abs(x).gt.30.0_DP) then
               j_start=14
            elseif(abs(x).gt.25.0_DP) then
               j_start=16
            elseif(abs(x).gt.20.0_DP) then
               j_start=18
            elseif(abs(x).gt.15.0_DP) then
               j_start=22
            elseif(abs(x).gt.12.0_DP) then
               j_start=25
            elseif(abs(x).gt.11.0_DP) then
               j_start=30
            else
               j_start=35
            endif
            psi0_c=dcmplx(1000.0_dp,0.0_DP)

            do j=j_start,200
               psi_c=psi_recursive_r_c(a,c,x,j)
                                !               if(abs(psi_c-psi0_c).lt.1.0e-14_DP) then
               if(abs(psi_c-psi0_c).lt.1.0e-14_DP) then
                  exit
               endif
               psi0_c=psi_c
            enddo
            psi_r_c_neg=psi_c
         else
            ! remember x!=0 since by the above test Im(x)<0
            psi_r_c_neg=psi_pow_cf_r_c_neg(a,c,x)
         endif
      elseif((a.eq.-0.5_DP).and.(c.eq.0.5_DP)) then
         ! Still valid
         psi_r_c_neg=sqrt(x)
      else
         psi_r_c_neg=dcmplx(1.0_DP,0.0)
      endif
      end function


      
!c     !!! USES THE BELOW Algorithms
      

      ! for a,c real and z positive

      function psi_pow_r(a,c,z,N)
      use gamma_function
      implicit none
      integer,optional::N     
      real(kind=DP)::a,c,sum,psi_pow_r
      real(kind=DP)::z,tmp1,tmp2,nn
      tmp1=phi_pow2_r(a,c,z,N)*gamma_r(1.0_DP-c)/gamma_r(a-c+1.0_DP)
      tmp2=phi_pow2_r(a-c+1.0_DP,2.0_DP-c,z,N)*(z**(1.0_DP-c))
      tmp2=tmp2*gamma_r(c-1.0_DP)/gamma_r(a)
      psi_pow_r=tmp1+tmp2
      end function

      function psi_pow_cf_r(a,c,z,N)
      use gamma_function
      implicit none
      integer::N
      real(kind=DP)::a,c,sum,psi_pow_cf_r
      real(kind=DP)::z,tmp1,tmp2,nn
      tmp1=phi_pow_cf_r(a,c,z,N)*gamma_r(1.0_DP-c)/gamma_r(a-c+1.0_DP)
      tmp2=phi_pow_cf_r(a-c+1.0_DP,2.0_DP-c,z,N)*(z**(1.0_DP-c))
      tmp2=tmp2*gamma_r(c-1.0_DP)/gamma_r(a)
      psi_pow_cf_r=tmp1+tmp2
      end function



  ! z complex, Re(z)>0
      function psi_pow_cf_r_c(a,c,z)
      use gamma_function
      implicit none
      real(kind=DP)::a,c
      complex(kind=DP)::z,tmp1,tmp2,psi_pow_cf_r_c
      if(c.ne.0.5_DP) then
         tmp1=phi_pow_cf_r_c(a,c,z)*dcmplx(gamma_r(1.0_DP-c),0.0_DP)/                            &
     &        dcmplx(gamma_r(1.0_DP+a-c),0.0_DP)
         tmp2=phi_pow_cf_r_c(1.0_DP+a-c,2.0_DP-c,z)*                              &
     &        (z**(dcmplx(1.0_DP-c,0.0_DP)))
         tmp2=tmp2*dcmplx(gamma_r(c-1.0_DP)/gamma_r(a),0.0_DP)
         psi_pow_cf_r_c=tmp1+tmp2
      elseif(a.eq.0.25_DP) then
         tmp1=phi_pow_cf_r_c(0.25_DP,0.5_DP,z)*                             &
     &        dcmplx(1.446409084632077142535701_DP,0.0_DP)
         tmp2=phi_pow_cf_r_c(0.75_DP,1.5_DP,z)*sqrt(z)
         tmp2=tmp2*dcmplx(-0.9777410674469237976315355_DP,0.0_DP)
         psi_pow_cf_r_c=tmp1+tmp2
      elseif(a.eq.-0.25_DP) then
         tmp1=phi_pow_cf_r_c(-0.25_DP,0.5_DP,z)*                                       &
     &        dcmplx(0.4888705337234618988157676_DP,0.0_DP)
         tmp2=phi_pow_cf_r_c(0.25_DP,1.5_DP,z)*sqrt(z)
         tmp2=tmp2*dcmplx(0.7232045423160385712678508_DP,0.0_DP)
         psi_pow_cf_r_c=tmp1+tmp2

      else
         tmp1=phi_pow_cf_r_c(a,0.5_DP,z)*dcmplx(gamma_r(0.5_DP)/                       &
     &        gamma_r(0.5_DP+a),0.0_DP)
         tmp2=phi_pow_cf_r_c(0.5_DP+a,1.5_DP,z)*sqrt(z)
         tmp2=tmp2*dcmplx(gamma_r(-0.5_DP)/gamma_r(a),0.0_DP)
         psi_pow_cf_r_c=tmp1+tmp2
      endif
!      write(*,*) 'psi_pow_cf_r_c(a,c',z,')=',psi_pow_cf_r_c
      end function



      ! For Re(z)<0
      ! Uses Kummers transformation
      !!! OBS: With Intel compiler ** is not valid operation for complex quadruple precision
      function psi_pow_cf_r_c_neg(a,c,z)
      use gamma_function
      implicit none
      real(kind=DP)::a,c
      complex(kind=DP)::z,tmp1,tmp2,psi_pow_cf_r_c_neg
      tmp1=phi_pow_cf_r_c(c-a,c,-z)*dcmplx(gamma_r(1.0_DP-c),0.0_DP)/                            &
     &     dcmplx(gamma_r(a-c+1.0_DP),0.0_DP)
      if(c.eq.0.5_DP) then
         tmp2=phi_pow_cf_r_c(1.0_DP-a,1.5_DP,-z)
                                !        write(*,*) 'phi_pow(',1.0_DP-a,',1.5,',-z,'=',tmp2
         tmp2=tmp2*sqrt(z)
         tmp2=tmp2*dcmplx(-3.544907701811032054596335_DP/gamma_r(a),      &
     &        0.0_DP)
      else 
         tmp2=phi_pow_cf_r_c(1.0_DP-a,2.0_DP-c,-z)*                                   &
     &     (z**(dcmplx(1.0_DP-c,0.0_DP)))
         tmp2=tmp2*dcmplx(gamma_r(c-1.0_DP)/gamma_r(a),0.0_DP)
      endif
!c$$$      write(*,*) 'tmp2=',phi_pow_cf_r_c(1-a,2.0_DP-c,-z),'*',                 &
!c$$$     &     (z**(dcmplx(1.0_DP-c,0.0_DP))),'*',dcmplx(gamma_r(c-1.0_DP)/       &
!c$$$     &     gamma_r(a),0.0_DP)
!      write(*,*) 'tmp2=',tmp2
!      write(*,*) 'tmp1=',tmp1
      psi_pow_cf_r_c_neg=exp(z)*(tmp1+tmp2)
      end function

      function psi_pow_cf_c(a,c,z)
      use gamma_function
      implicit none
      complex(kind=DP)::a,c
      complex(kind=DP)::z,tmp1,tmp2,psi_pow_cf_c
      tmp1=phi_pow_cf_c(a,c,z)*gamma_c(cone-c)/gamma_c(a-c+cone)
      if(c.eq.chalf) then
         tmp2=phi_pow_cf_c(a+chalf,dcmplx(1.5_DP),z)*sqrt(z)
      else 
         tmp2=phi_pow_cf_c(a-c+cone,ctwo-c,z)*                              &
     &        (z**(cone-c))
      endif
      tmp2=tmp2*gamma_c(c-cone)/gamma_c(a)
      psi_pow_cf_c=tmp1+tmp2
      end function

!c     ! Asymptotic series for psi:


   !!!   Asymptotic Powerseries in 1/z
      !!!   for real and complex arguments
      !!!   a,c=real and z=complex

      ! All arguments real
      function psi_asymp_pow_r(a,c,z)
      use gamma_function
      implicit none
      integer::N,j
      integer,parameter::N_max=30
      real(kind=DP)::a,c,sum,psi_asymp_pow_r
      real(kind=DP)::z,tmp1,tmp2,Psi1,T1,nn,factor
      
      T1=-1.0_DP*a*(a-c+1.0_DP)/z
      Psi1=1.0_DP+T1
      do j=1,N_max
         nn=real(j,DP)
         factor=(a+nn)*(a-c+1.0_DP+nn)/((nn+1.0_DP)*z)
         ! remember this is an asymptotic expansion      
         ! we must cut of the series when it start to grow
         if(factor.ge.1.0_DP) then
            exit
         endif
         T1=-1.0_DP*T1*factor
         Psi1=Psi1+T1
         if(abs(T1/Psi1).lt.eps) then             
            exit
         endif
      enddo
      psi_asymp_pow_r=z**(-a)*Psi1
      end function

      ! z complex and a,c real
      ! trying to do everything in complex arithmetic
      function psi_asymp_pow_r_c(a,c,z)
      use gamma_function
      implicit none
      integer::N,j
      integer,parameter::N_max=200
      real(kind=DP)::a,c
      complex(DP)::z,tmp1,tmp2,Psi1,T1,nn,factor,a_c_m_c_c,nn_ett_c
      complex(DP)::psi_asymp_pow_r_c,a_c,c_c,ett_c,nn_c
      a_c=dcmplx(a,0.0_DP)
      c_c=dcmplx(c,0.0_DP)
      a_c_m_c_c=dcmplx(a-c,0.0_DP)
      ett_c=dcmplx(1.0_DP,0.0_DP)
      T1=-cone*a_c*(a_c-c_c+cone)/z
      Psi1=ett_c+T1
      loop: do j=1,N_max
         nn_c=dcmplx(j,0.0_DP)
         nn_ett_c=dcmplx(1+j,0.0_DP)
!         factor=(a+nn_c)*(a_c-c_c+ett_c+nn_c)/((nn_c+ett_c)*z)
         factor=(a_c+nn_c)*(a_c_m_c_c+nn_ett_c)/((nn_ett_c)*z)
         ! remember this is an asymptotic expansion
         ! we must cut of the series when it start to grow
!         if(abs(factor).ge.1.0_DP) then
!            write(*,*) 'err=',abs(T1/Psi1),' abs(factor)=',abs(factor)
!            write(*,*) 'exit 1'
!            exit
!         endif
         T1=-ett_c*T1*factor
         Psi1=Psi1+T1
         if((abs(T1/Psi1).lt.eps).or.(abs(factor).ge.1.0_DP)) then             
!            write(*,*) 'exit 2'
            exit loop
         endif
      enddo loop
      psi_asymp_pow_r_c=(z**(-a_c))*Psi1
!      write(*,*) 'psi_asyp=',psi_asymp_pow_r_c
      end function

      
      ! For Re(z)<0
      function psi_asymp_pow_r_c_neg(a,c,z)
      use gamma_function
      implicit none
      integer::N,j
      integer,parameter::N_max=200
      real(kind=DP)::a,c
      complex(kind=DP)::z,tmp1,tmp2,Psi1,T1,nn,factor
      complex(DP)::psi_asymp_pow_r_c_neg,a_c,c_c,ett_c,nn_c
      a_c=dcmplx(a,0.0_DP)
      c_c=dcmplx(c,0.0_DP)
!      ett_c=dcmplx(1.0_DP,0.0_DP)
      T1=-cone*a_c*(a_c-c_c+cone)/z
      Psi1=ett_c+T1
      do j=1,N_max
         nn_c=dcmplx(j,0.0_DP)
         factor=(a_c+nn_c)*(a_c-c_c+cone+nn_c)/((nn_c+cone)*(-z))
         ! remember this is an asymptotic expansion
         ! we must cut of the series when it start to grow
         if(abs(factor).ge.1.0_DP) then
!            write(*,*) 'err=',abs(T1/Psi1),' abs(factor)=',abs(factor)
            exit
         endif
         T1=T1*factor  ! No (-1)^n factor now
         Psi1=Psi1+T1
         if(abs(T1/Psi1).lt.eps) then             
            exit
         endif
      enddo
      psi_asymp_pow_r_c_neg=z**(-a_c)*Psi1
!      write(*,*) 'psi_as_p_r_c_neg=',z**(-a_c),'*',Psi1
      end function
      


!c     !! Recursive algorithms:
      
         function psi_recursive_r(a,c,x,N)
      use gamma_function
      implicit none
      real(DP)::a,c,x,psi_recursive_r
      real(DP)::an,bn,Sn0,Sn1,Rn0,Rn1,tmp,den
      real(DP),dimension(:),allocatable::lambda2
      integer :: N,j
      allocate(lambda2(0:N))
      lambda2(0)=1.0_DP
      !! make sure 1+a+c
      if((1.0_DP+a-c.lt.0.0_DP).and.(floor(a-c).eq.ceiling(a-c))) then
         write(*,*) 'Recursive method only for 1+a-c not neg. int!'
         write(*,*) 'we have 1+a-c=',1.0_DP+a-c
         stop
      endif
      do j=1,N
!         tmp=real((-1)**j,DP)*gamma_r(c-a)/gamma_r(c-a-real(j,DP))
!         lambda(j)=tmp/gamma_r(real(j+1,DP))
         lambda2(j)=-1.0_DP*lambda2(j-1)*(c-a-real(j,DP))/real(j,DP)
      enddo
      Sn0=0.0_DP
      Rn0=0.0_DP
      do j=N,1,-1
         den=(real(1+j,DP)+a-c)
         an=(c-2.0_DP*a-real(2*j,DP)-x)/den
         bn=(a+real(j-1,DP))/den
         !R(n-1)
         Rn1=-1.0_DP*bn/(an+Rn0)
         !Sn(n-1)
         Sn1=Rn1*(lambda2(j)+Sn0)
         Rn0=Rn1
         Sn0=Sn1
      enddo
      deallocate(lambda2)
      psi_recursive_r=(x)**(-a)/(1.0_DP+Sn0)
  !    write(*,*) 'psi_recursive=',psi_recursive_r
      end function



!c     $ The functions below are specialized to c=1/2
!c     $ and are suboptimal algorithms for use in the 
!c     $ Whittaker function

   !! First set up all the lambdas
      !! Observe that when computing the Whittaker function
      !! a=-weight/2 and c=1/2 in all cases
      !! hence the lambdas are the same...
      subroutine get_lambda(a,c)
      use gamma_function
      implicit none
      integer::j
      real(DP)::a,c
      complex::a_c,c_c,jj_c
      if(lambda_set.and.(allocated(lambda))) then
         if((a.eq.a_set_l).and.(c.eq.c_set_l)) then
            return
         endif  
      endif
      a_set_l=a
      c_set_l=c
!      a_c=dcmplx(a,0.0_DP)
!      c_c=dcmplx(c,0.0_DP)
      if(allocated(lambda)) then
         deallocate(lambda)
      endif
      allocate(lambda(0:N_max_recursive))
      lambda(0)=dcmplx(1.0,0.0_DP)
      do j=1,N_max_recursive 
!         tmp=real((-1)**j,DP)*gamma_r(c-a)/gamma_r(c-a-real(j,DP))
!         lambda(j)=dcmplx(tmp/gamma_r(real(j+1,DP)),0.0_DP)
!         jj_c=dcmplx(j,0.0_DP)
         lambda(j)=-lambda(j-1)*                                          &
     &        dcmplx((c-a-real(j,DP))/real(j,DP),0.0_DP)

      enddo
!      write(*,*) 'Lambda is now set!'
!      write(*,*) 'size(l)=',size(lambda)
      lambda_set=.true.
      end subroutine get_lambda


    ! Here x=z= is complex
      ! I use only complex arithmetic
      ! This works much better than mixing...
      function psi_recursive_r_c(a,c,z,N)
      use gamma_function
      implicit none
      real(DP)::a,c
      complex(DP)::Sn0,Sn1,Rn0,Rn1,a_c,c_c,jj_c,ett_c,noll_c
      complex(DP)::an,bn,den
!      complex(DP),dimension(:),allocatable::lambda
      complex(DP)::z,psi_recursive_r_c     
      integer :: N,j
      if(N.gt.N_max_recursive) then
         N=N_max_recursive
      endif
      call get_lambda(a,c)
      ett_c=dcmplx(1.0,0.0_DP)
      noll_c=dcmplx(0.0,0.0)
      a_c=dcmplx(a,0.0_DP)
      c_c=dcmplx(c,0.0_DP)
      Sn0=noll_c
      Rn0=noll_c
      do j=N,1,-1
         jj_c=dcmplx(j,0.0_DP)
         den=(ett_c+jj_c+a_c-c_c)
         an=(c_c-ctwo*a_c-ctwo*jj_c-z)/den

         bn=(a_c+jj_c-ett_c)/den
         !R(n-1)
         Rn1=-bn/(an+Rn0)
         !Sn(n-1)
         Sn1=Rn1*(lambda(j)+Sn0)
         Rn0=Rn1
         Sn0=Sn1
      enddo
!      psi_recursive_r_c=((z)**(-a_c))/(ett_c+Sn0)
      psi_recursive_r_c=ett_c/(z**(a_c)*(ett_c+Sn0))
      end function






!c     !! Next best power series function
!c     !! For c==1/2
!c     !! Uses the subroutine below to optimize coefficients

         ! c==1/2
      ! try to split real and imaginar parts
      !! This is the most efficient algorithm now
      function psi_pow3_r_c(a,c,z,N)
      use gamma_function
      implicit none
      integer::j
      integer,optional::N
      integer,parameter::N_max=1000
      real(kind=DP)::a,c,sum
      real(kind=DP)::term,jj,factor,jj1
      complex(DP)::psi_pow3_r_c,z,TI,MI,ettc,a00,b00
      real(DP)::a0,b0,f0,a1,b1,f1,b2
      complex(DP)::TI2,a11,b11,f11,MI2,f00,jjc
      real(DP)::f0I,f0R,f1I,f1R,MI2R,MI2I,TI2R,TI2I,x,y
      real(DP)::argz,Resqz,Imsqz
      real(DP),dimension(0:N_max)::TIR,TII
      INTEGER::K,ix
      !!! since we usually call this for both +-a
      !!! we set up both sets of coeffs at once
      !!! and simply use the correct one
      if(c.ne.0.5_DP) then
         write(*,*) 'Only for c=1/2!!'
         stop
      endif
      if(a_set.ne.abs(a)) then
         call setup_coeffs_for_psi_pow3(a)
      endif
      if(a.lt.0.0_DP) then
         ix=1
      else
         ix=2
      endif
      x=real(z,DP)
      y=imag(z)
      argz=argument(x,y)
      argz=argz*0.5_DP
      Resqz=sqrt(abs(z))*cos(argz)
      Imsqz=sqrt(abs(z))*sin(argz)
      !! No use to have this special case since
      !! cancellation will be strong
      a0=a_k(ix,0)
      b0=b_k(ix,0)
      b1=2.0_DP*b_k(ix,0)
      f0I=0.0_DP
      f0R=1.0_DP      
      MI2R=(a0-Resqz*b1)
      MI2I=-1.0_DP*Imsqz*b1
      a1=a0
      b1=b0
      do j=1,N_max_pow
         jj=real(j,DP)
!         f1R=real(z**jj,DP) !/gamma_r(jj+0.5_DP)
!         f1I=imag(z**jj)       !/gamma_r(jj+0.5_DP)
         !!! TOO EXPENSIVE TO DO exponentials!!!
         f1R=f0R*x-f0I*y
         f1I=f0I*x+f0R*y
         f0R=f1R
         f0I=f1I
         a1=a_k(ix,j)
         b1=b_k(ix,j)
         TI2R=a1*f1R-Resqz*b1*f1R+Imsqz*b1*f1I
         TI2I=a1*f1I-Imsqz*f1R*b1-Resqz*f1I*b1
         MI2R=TI2R+MI2R
         MI2I=TI2I+MI2I
         if(abs(TI2R/MI2R).lt.eps) then 
            exit
         endif
      enddo
      psi_pow3_r_c=dcmplx(MI2R,MI2I)/dcmplx((a_k(ix,0)*b_k(ix,0)),0.0D0)
!      write(*,*)'psi_p3=',dcmplx(MI2R,MI2I),'/',a_k(ix,0),'*',b_k(ix,0)
      if(j.ge.N_max_pow) then
         write(*,*) 'N_max(psi_pow3_r_c) reached!'
         write(*,*) 'a=',a
         write(*,*) 'c=',c
         write(*,*) 'z=',z
         write(*,*) 'psi_pow=',psi_pow3_r_c
         stop
      endif
!      write(*,*) 'a=',a
!      write(*,*) 'psi_pow=',psi_pow3_r_c
!      write(*,*) 'err=',abs(TI/MI)
      end function


!c     !!! USES THIS SUBROUTINE:
      !! Set up the coefficients needed for computing
      !! Psi(1,1/2,z) by powerseries
      !! Not more than 100 coefficients should be needed
      subroutine setup_coeffs_for_psi_pow3(a)
      use gamma_function
      implicit none
      real(DP)::a
      real(DP)::b,jj,tmp
      integer::j
      logical::b_k1_0,a_k1_0,b_k2_0,a_k2_0
      b_k1_0=.false.
      a_k1_0=.false.
      b_k2_0=.false.
      a_k2_0=.false.
!      if(coeff_psi_pow3_is_set) then
      if(abs(a).eq.a_set) then       !already set coeffs for this a
      else
         if(allocated(a_k)) then
            deallocate(a_k)
            deallocate(b_k)
         endif
         allocate(a_k(1:2,0:N_max_pow))
         allocate(b_k(1:2,0:N_max_pow))
         a_k(:,:)=0.0_DP
         b_k(:,:)=0.0_DP
         if(abs(a).eq.0.25_DP) then
            a_k(1,0)=-4.901666809860710580516393213451562_DP
            b_k(1,0)=3.6256099082219083119306851558676720_DP
            a_k(2,0)=3.6256099082219083119306851558676720_DP
            b_k(2,0)=1.2254167024651776451290983033628905_DP
         elseif(abs(a).eq.0.1_DP) then 
            a_k(1,0)=-10.6862870211931935489730533569_DP
            b_k(1,0)=2.21815954375768822305905402191_DP
            a_k(2,0)=9.51350769866873183629248717727_DP
            b_k(2,0)=1.48919224881281710239433338832_DP
         elseif(abs(a).eq.0.2_DP) then 
            a_k(1,0)=-5.82114856862651686818160469134_DP
            b_k(1,0)=2.99156898768759062831251651590_DP
            a_k(2,0)=4.59084371199880305320475827593_DP
            b_k(2,0)=1.29805533264755778568117117915_DP
         elseif(abs(a).eq.0.3_DP) then 
            a_k(1,0)=-4.32685110882519261893723726384_DP
            b_k(1,0)=4.59084371199880305320475827593_DP
            a_k(2,0)=2.99156898768759062831251651590_DP
            b_k(2,0)=1.16422971372530337363632093827_DP
         elseif(abs(a).eq.0.4_DP) then 
            a_k(1,0)=-3.72298062203204275598583347080_DP
            b_k(1,0)=9.51350769866873183629248717727_DP
            a_k(2,0)=2.21815954375768822305905402191_DP
            b_k(2,0)=1.06862870211931935489730533569_DP
         elseif(abs(a).eq.0.5_DP) then
            a_k(2,0)=1.77245385090551602729816748334_DP
            b_k(2,0)=1.0_DP
            if(a.eq.-0.5_DP) then 
                                !! Shouldn't be called for weight +-1/2
               write(*,*) 'Should not be called for weight -1!'
               stop
            endif
         elseif(abs(a).eq.0.6_DP) then 
            a_k(1,0)=-3.69693257292948037176509003651_DP
            b_k(1,0)=-10.6862870211931935489730533569_DP
            a_k(2,0)=1.48919224881281710239433338832_DP
            b_k(2,0)=0.951350769866873183629248717727_DP
         elseif(abs(a).eq.0.7_DP) then 
            a_k(1,0)=-4.27366998241084375473216645129_DP
            b_k(1,0)=-5.82114856862651686818160469134_DP
            a_k(2,0)=1.29805533264755778568117117915_DP
            b_k(2,0)=.918168742399760610640951655186_DP
         elseif(abs(a).eq.0.8_DP) then 
            a_k(1,0)=-5.73855463999850381650594784491_DP
            b_k(1,0)=-4.32685110882519261893723726384_DP
            a_k(2,0)=1.16422971372530337363632093827_DP
            b_k(2,0)=.897470696306277188493754954771_DP
         elseif(abs(a).eq.0.9_DP) then 
            a_k(1,0)=-10.5705641096319242625472079747_DP
            b_k(1,0)=-3.72298062203204275598583347080_DP
            a_k(2,0)=1.06862870211931935489730533569_DP
            b_k(2,0)=.887263817503075289223621608763_DP
         elseif(abs(a).eq.1.0_DP) then 
            a_k(2,0)=1.0_DP
            b_k(2,0)=.886226925452758013649083741671_DP
         elseif(a.eq.0.0_DP) then 
                 !! Shouldn't be called for weight 0
            write(*,*) 'Should not be called for weight 0!'
         else
            a_k(1,0)=gamma_r(-abs(a))
            b_k(1,0)=gamma_r(-abs(a)+0.5_DP)
            a_k(2,0)=gamma_r(abs(a))
            b_k(2,0)=gamma_r(abs(a)+0.5_DP)
         endif
         a_k(1,0)=a_k(1,0)/sqrtPi             !gamma_r(0.5_DP)
         b_k(1,0)=b_k(1,0)/sqrtPi             !gamma_r(0.5_DP) !sqrt(Pi)
         a_k(2,0)=a_k(2,0)/sqrtPi             !gamma_r(0.5_DP)
         b_k(2,0)=b_k(2,0)/sqrtPi   !gamma_r(0.5_DP) !sqrt(Pi)
!         write(*,*) 'computing a!'
         tmp=sqrtPi
         do j=1,N_max_pow
            jj=real(j,DP)         
            tmp=gamma_r(jj+0.5_DP)*gamma_r(jj+1.0_DP)
!            write(*,*) 'a_k(1,',j,')=',gamma_r(-abs(a)+jj),'/',tmp
!            write(*,*) 'a_k1_0=',a_k1_0
            !!If the pochammer symbol (a)_n or (b)_n becomes 0 then it stays 0
            if((jj-abs(a).eq.0.0_DP).or.(a_k1_0)) then
               a_k(1,j)=0.0_DP
               a_k1_0=.true.
            else
               a_k(1,j)=gamma_r(-abs(a)+jj)/tmp

            endif
            if((-abs(a)+jj+0.5_DP.eq.0.0_DP).or.(b_k1_0)) then
               b_k(1,j)=0.0_DP
               b_k1_0=.true.
            else
               b_k(1,j)=gamma_r(-abs(a)+jj+0.5_DP)/(jj+0.5_DP)/                 &
     &              tmp
            endif
            if((jj+abs(a).eq.0.0_DP).or.(a_k2_0)) then
               a_k(2,j)=0.0_DP
               a_k2_0=.true.
            else
               a_k(2,j)=gamma_r(abs(a)+jj)/tmp
            endif
            if((abs(a)+jj+0.5_DP.eq.0.0_DP).or.(b_k2_0)) then
               b_k(2,j)=0.0_DP
               b_k2_0=.true.
            else
               b_k(2,j)=gamma_r(abs(a)+jj+0.5_DP)/(jj+0.5_DP)/                 &
     &              tmp
            endif
            !!! These will converge to 0 but are actually non-zero
            !!! hence when they get small enough we simply jump out
!            if((abs(a_k(1,j)).eq.0.0).or.(abs(a_k(2,j)).eq.0.0)) then

!            write(*,*) 'a_k(1,',j,')=',a_k(1,j)
!            write(*,*) 'a_k(2,',j,')=',a_k(2,j)
!            write(*,*) 'b_k(1,',j,')=',b_k(1,j)
!            write(*,*) 'b_k(2,',j,')=',b_k(2,j)
!            stop                    !               exit
!            endif
            !! WE cn get out of the loop if all terms are 0
            if(a_k1_0.and.a_k2_0.and.b_k1_0.and.b_k2_0) then
               exit
            endif
       enddo
         a_set=abs(a)
      endif
      end subroutine



!c     !! Another version of the above algorithm

      ! c==1/2
      function psi_pow2_r_c(a,c,z,N)
      use gamma_function
      implicit none
      integer::j
      integer,optional::N
      integer,parameter::N_max=200
      real(kind=DP)::a,c,sum
      real(kind=DP)::term,jj,factor,jj1
      complex(DP)::psi_pow2_r_c,z,TI,MI,ettc,a0,b0,f0,a1,b1,f1,a00,b00
      complex(DP)::TI2,a11,b11,f11,MI2,f00,jjc
      if(c.ne.0.5_DP) then
         write(*,*) 'c <> 1/2!'
         stop
      endif
      if(a.eq.c) then 
         psi_pow2_r_c=exp(z) ! always valid if a=c
         return 
      endif
      ettc=dcmplx(1.0_DP,0.0_DP)
      if(a.eq.-0.9_DP) then 
         a00=dcmplx(-10.5705641096319242625472079747_DP,0.0_DP )
         b00=dcmplx(-3.72298062203204275598583347080_DP,0.0_DP )
      elseif(a.eq.-0.8_DP) then 
         a00=dcmplx(-5.73855463999850381650594784491_DP,0.0_DP )
         b00=dcmplx(-4.32685110882519261893723726384_DP,0.0_DP )
      elseif(a.eq.-0.7_DP) then 
         a00=dcmplx(-4.27366998241084375473216645129_DP,0.0_DP )
         b00=dcmplx(-5.82114856862651686818160469134_DP,0.0_DP )
      elseif(a.eq.-0.6_DP) then 
         a00=dcmplx(-3.69693257292948037176509003651_DP,0.0_DP )
         b00=dcmplx(-10.6862870211931935489730533569_DP,0.0_DP )
      elseif(a.eq.-0.5_DP) then 
         !! Shouldn't be called for weight +-1/2
         write(*,*) 'Should not be called for weight -1/2!'
         stop
!         a00=dcmplx(-3.54490770181103205459633496668_DP,0.0_DP )
!         b00=dcmplx(Float(infinity)+Float(infinity)*I_DP,0.0_DP )
      elseif(a.eq.-0.4_DP) then 
         a00=dcmplx(-3.72298062203204275598583347080_DP,0.0_DP )
         b00=dcmplx(9.51350769866873183629248717727_DP,0.0_DP )
      elseif(a.eq.-0.3_DP) then 
         a00=dcmplx(-4.32685110882519261893723726384_DP,0.0_DP )
         b00=dcmplx(4.59084371199880305320475827593_DP,0.0_DP )
      elseif(a.eq.-0.2_DP) then 
         a00=dcmplx(-5.82114856862651686818160469134_DP,0.0_DP )
         b00=dcmplx(2.99156898768759062831251651590_DP,0.0_DP )
      elseif(a.eq.-0.1_DP) then 
         a00=dcmplx(-10.6862870211931935489730533569_DP,0.0_DP )
         b00=dcmplx(2.21815954375768822305905402191_DP,0.0_DP )
      elseif(a.eq.0.0_DP) then 
         !! Shouldn't be called for weight 0
         write(*,*) 'Should not be called for weight 0!'
         stop
         !a00=dcmplx(Float(infinity)+Float(infinity)*I_DP,0.0_DP )
         !b00=dcmplx(1.77245385090551602729816748334_DP,0.0_DP )
      elseif(a.eq.0.1_DP) then 
         a00=dcmplx(9.51350769866873183629248717727_DP,0.0_DP )
         b00=dcmplx(1.48919224881281710239433338832_DP,0.0_DP )
      elseif(a.eq.0.2_DP) then 
         a00=dcmplx(4.59084371199880305320475827593_DP,0.0_DP )
         b00=dcmplx(1.29805533264755778568117117915_DP,0.0_DP )
      elseif(a.eq.0.3_DP) then 
         a00=dcmplx(2.99156898768759062831251651590_DP,0.0_DP )
         b00=dcmplx(1.16422971372530337363632093827_DP,0.0_DP )
      elseif(a.eq.0.4_DP) then 
         a00=dcmplx(2.21815954375768822305905402191_DP,0.0_DP )
         b00=dcmplx(1.06862870211931935489730533569_DP,0.0_DP )
      elseif(a.eq.0.5_DP) then
         a00=dcmplx(1.77245385090551602729816748334_DP,0.0_DP )
         b00=dcmplx(1._DP,0.0_DP )
      elseif(a.eq.0.6_DP) then 
         a00=dcmplx(1.48919224881281710239433338832_DP,0.0_DP )
         b00=dcmplx(0.951350769866873183629248717727_DP,0.0_DP )
      elseif(a.eq.0.7_DP) then 
         a00=dcmplx(1.29805533264755778568117117915_DP,0.0_DP )
         b00=dcmplx(.918168742399760610640951655186_DP,0.0_DP )
      elseif(a.eq.0.8_DP) then 
         a00=dcmplx(1.16422971372530337363632093827_DP,0.0_DP )
         b00=dcmplx(.897470696306277188493754954771_DP,0.0_DP )
      elseif(a.eq.0.9_DP) then 
         a00=dcmplx(1.06862870211931935489730533569_DP,0.0_DP )
         b00=dcmplx(.887263817503075289223621608763_DP,0.0_DP )
      elseif(a.eq.1.0_DP) then 
         a00=dcmplx(1.0_DP,0.0_DP )
         b00=dcmplx(.886226925452758013649083741671_DP,0.0_DP )
      else
         a00=dcmplx(gamma_r(a),0.0_DP)
         b00=dcmplx(2.0_DP*gamma_r(a+0.5_DP),0.0_DP)
      endif
      f00=dcmplx(0.564189583547756286948079451561_DP,0.0_DP)
      a00=dcmplx(gamma_r(a),0.0_DP)
      b00=dcmplx(gamma_r(a+0.5_DP),0.0_DP)
      TI=(a00-dcmplx(2.0_DP,0.0)*sqrt(z)*b00)/                               &
     &     dcmplx(gamma_r(0.5_DP),0.0_DP)
      MI=TI
      a0=a00
      b0=ctwo*b00
      f0=f00
!      write(*,*) 'MI =',Mi
      f1=dcmplx(1.0_DP/sqrtPi,0.0_DP)
      do j=1,N_max
         jj=real(j,DP)
         jjc=dcmplx(jj,0.0_DP)
         a1=dcmplx(gamma_r(a+jj),0.0_DP)
         b1=dcmplx(gamma_r(a+jj+0.5_DP)/(jj+0.5_DP),0.0_DP)
         f1=f1*(z/(jjc*jjc-chalf*jjc))
         TI=(a1-sqrt(z)*b1)*f1        
         MI=TI+MI
         if(abs(TI/MI).lt.eps) then 
            exit
         endif
      enddo
      psi_pow2_r_c=MI*CPI/(a00*b00)
      if(j.ge.N_max) then
         write(*,*) 'N_max(psi_pow2_r_c) reached!'
      endif
!      write(*,*) 'err=',abs(TI/MI)
      end function



!c     !! Power series using Phi
      
   ! a,c complex but x real
      function phi_pow3(a,c,z,N)
      use gamma_function
      implicit none
      integer::N,j
      complex(kind=DP)::phi_pow3,z,ac,cc
      real(kind=DP)::a,c
      ac=dcmplx(a,0.0)
      cc=dcmplx(c,0.0)
      phi_pow3=dcmplx(1.0_DP)+ac*z/cc
      do j=2,N
         phi_pow3=phi_pow3+pochammer(ac,j)/                                   &
     &        pochammer(cc,j)*(z**dcmplx(j,0.0_DP))/                               &
     &        dcmplx(real(factorial(j),kind=DP),0.0D0)
      enddo
      end function 

      function psi_pow(a,c,x,N)
      use gamma_function
      implicit none
      integer::N
      complex(kind=DP)::a,c,sum,psi_pow
      complex(kind=DP)::z,tmp1,tmp2
      real(kind=DP)::x
      tmp1=phi_pow(a,c,x,N)*gamma_c(dcmplx(1)-c)/gamma_c(a-c+dcmplx(1))
      tmp2=phi_pow(a-c+dcmplx(1.0_DP),dcmplx(2.0_DP)-c,x,N)
      tmp2=tmp2*gamma_c(c-dcmplx(1))/gamma_c(a)    
      psi_pow=tmp1+tmp2
      end function
      
 








!c     !!! Recursive functions
      





  !! Modified, for c=0.5
      function psi_recursive2_r_c(a,c,z,N)
      use gamma_function
      implicit none
      real(DP)::a,c,tmp
      complex(DP)::Sn0,Sn1,Rn0,Rn1,a_c,c_c,jj_c,ett_c,noll_c
      complex(DP)::an,bn,den
!      complex(DP),dimension(:),allocatable::lambda
      complex(DP)::z,psi_recursive2_r_c     
      integer :: N,j
      real(DP)::x,y,anR,anI,jj
      if(N.gt.N_max_recursive) then
         N=N_max_recursive
      endif
      call get_lambda(a,c)
      x=real(z,DP)
      y=dimag(z)
      a_c=dcmplx(a,0.0_DP)
      c_c=dcmplx(0.5_DP,0.0_DP)
      Sn0=czero
      Rn0=czero
      do j=N,1,-1
         jj=real(j,DP)
         jj_c=dcmplx(j,0.0_DP)
         den=(jj_c+a_c+chalf)
         an=(c_c-ctwo*a_c-ctwo*jj_c-z)/den
         anR=c-2.0_DP*a-2.0_DP*jj-x  !/den
!         anI=-y/den
         bn=(a_c+jj_c-cone)/den
         !R(n-1)
         Rn1=-bn/(an+Rn0)
         !Sn(n-1)
         Sn1=Rn1*(lambda(j)+Sn0)
         Rn0=Rn1
         Sn0=Sn1
      enddo
      psi_recursive2_r_c=ett_c/(z**(a_c)*(ett_c+Sn0))
      end function
      


      function psi_recursive_r_c2(a,c,z,N)
      use gamma_function
      implicit none
      real(DP)::a,c,x,psi_recursive_r
      real(DP)::Sn0_r,Sn0_i,Sn1_r,Sn1_i,Rn0_r,Rn0_i,Rn1_r,Rn1_i
      real(DP)::an_r,an_i,bn,temp,den
      real(DP),dimension(:),allocatable::lambda_r
      complex(DP)::z,psi_recursive_r_c2
      integer :: N,j
      allocate(lambda_r(0:N))
      lambda_r(0)=1.0_DP
      do j=1,N
         lambda_r(j)=-1.0_DP*lambda_r(j-1)*(c-a-real(j,DP))/real(j,DP)
      enddo
      Sn0_r=0.0_DP
      Sn0_I=0.0_DP
      Rn0_r=0.0_DP
      Rn0_I=0.0_DP
      do j=N,1,-1
         den=(real(1+j,DP)+a-c)
         an_r=(c-2.0_DP*a-real(2*j,DP)-real(z,DP))/den
         an_i=(-imag(z))/den
!         Iman=(c-2.0_DP*a-real(2*j,DP)-z)/den
         bn=(a+real(j-1,DP))/den
         !R(n-1)
         den=(an_r+Rn0_r)**2+(an_i+Rn0_i)**2
         Rn1_r=-1.0_DP*bn*(an_r+Rn0_r)/den
         Rn1_i=bn*(an_i+Rn0_i)/den
         !Sn(n-1)
         temp=lambda_r(j)+Sn0_r
         Sn1_r=Rn1_r*temp-Rn1_i*Sn0_i
         Sn1_i=Rn1_i*temp+Rn1_i*Sn0_i

         Rn0_r=Rn1_r
         Rn0_i=Rn1_i
         Sn0_r=Sn1_r
         Sn0_i=Sn1_i
      enddo
      deallocate(lambda_r)
      psi_recursive_r_c2=((z)**(dcmplx(-a)))/                             &
     &     (dcmplx(1.0_DP+Sn0_r,Sn0_i))
      end function

      


!c     !!!! Confluent Hypergeometric M or Phi
      





      ! a,c complex but x real
      function phi_pow(a,c,x,N)
      use gamma_function
      implicit none
      integer::N,j
      complex(kind=DP)::a,c,sum,phi_pow,z
      real(kind=DP)::x
      z=dcmplx(x,0.0_DP)
      phi_pow=dcmplx(1.0_DP)+a*z/c
      do j=2,N
         phi_pow=phi_pow+pochammer(a,j)/                                   &
     &        pochammer(c,j)*(z**dcmplx(j,0.0_DP))/                               &
     &        dcmplx(real(factorial(j),DP),0.0_DP)
      enddo
      end function

    

      !! a=neg. integer collapses to a polynomial...
      function phi_int_r_c(a,c,z)
      use gamma_function
      implicit none
      real(DP)::a,c,term
      complex(DP)::z,phi_int_r_c,sum
      integer::m,n,fak
      if(a.gt.0.0_DP) then
         write(*,*) 'Only call this for negative integer a!'
         write(*,*) 'a=',a
         stop
      endif
      m=floor(-a)
      if(m.ne.ceiling(-a)) then
         write(*,*) 'Only call this for negative integer a!'
         write(*,*) 'a=',a
         stop
      endif
      sum=dcmplx(0.0_DP,0.0_DP)
      do n=0,m
         fak=factorial(n)
         term=pochammer_r(a,n)/pochammer_r(c,n)/real(fak,DP)
         sum=sum+dcmplx(term,0.0_DP)*(z**dcmplx(real(n,DP),0.0_DP))      
      enddo
      phi_int_r_c=sum
      end function


      ! The phi function evaluated by N-terms powerseries
      function phi_pow_r(a,c,z,N)
      use gamma_function
      implicit none
      integer::N,j
      real(kind=DP)::a,c,sum,phi_pow_r
      real(kind=DP)::z,term,jj,factor
      if(a.eq.c) then 
         phi_pow_r=exp(z) ! always valid if a=c
         return 
      endif
      phi_pow_r=1.0_DP+a*z/c
      do j=2,N
         jj=real(j,DP)
         factor=gamma_r(a+jj)*gamma_r(c)
         factor=factor/gamma_r(a)/gamma_r(c+jj)/gamma_r(jj+1.0_DP)
!         factor=pochammer_r(a,j)/pochammer_r(c,j)/factorial(j)
!         phi_pow_r=phi_pow_r+pochammer_r(a,j)/pochammer_r(c,j)*         &
         phi_pow_r=phi_pow_r+factor*(z**j)
!     &        (z**real(j,DP))/factorial(j)
      enddo
      end function

    


      ! The phi function evaluated by powerseries
      ! (The N is a void parameter but could be changed to contain number of terms if )
      function phi_pow2_r(a,c,z,N)
      use gamma_function
      implicit none
      integer::N,j
      integer,parameter::N_max=200
      real(kind=DP)::a,c,sum,phi_pow2_r
      real(kind=DP)::z,term,jj,factor,MI,TI
      if(a.eq.c) then 
         phi_pow2_r=exp(z) ! always valid if a=c
         return 
      endif
      TI=a*z/c
      MI=1.0_DP+TI
      do j=1,N_max
         jj=real(j,DP)
         TI=TI*(a+jj)*z/((c+jj)*(jj+1.0_DP))
         MI=MI+TI
         if(abs(TI/MI).lt.eps) then 
            exit
         endif
      enddo
      phi_pow2_r=MI
      if(j.ge.N_max) then
         write(*,*) 'N_max(phu_pow2_r) reached!'
      endif
!      write(*,*) 'err=',abs(TI/MI)
      end function

      ! really pow2
      function phi_pow_r_c(a,c,z,N)
      use gamma_function
      implicit none
      integer::j
      integer,optional::N
      integer,parameter::N_max=200
      real(kind=DP)::a,c,sum
      real(kind=DP)::term,jj,factor
      complex(DP)::phi_pow_r_c,z,TI,MI,ettc
      if(a.eq.c) then 
         phi_pow_r_c=exp(z) ! always valid if a=c
         return 
      endif
      ettc=dcmplx(1.0_DP,0.0_DP)
      TI=z*dcmplx(a/c,0.0_DP)
      MI=ettc+TI
      do j=1,N_max
         jj=real(j,DP)
         TI=TI*z*dcmplx((a+jj)/((c+jj)*(jj+1.0_DP)),0.0_DP)
         MI=TI+MI
         if(abs(TI/MI).lt.eps) then 
            exit
         endif
      enddo
      phi_pow_r_c=MI
!      write(*,*) 'j=',j
      if(j.ge.N_max) then
         write(*,*) 'N_max(phi_pow_r_c) reached!'
      endif
!      write(*,*) 'err=',abs(TI/MI)
      end function

 ! The phi function evaluated by N-terms powerseries
      ! using continued fractions
      function phi_pow_cf_r(a,c,z,N)
      use gamma_function
      implicit none
      integer::N,j
      integer,parameter::N_max=1000
      real(kind=DP)::a,c,sum,phi_pow_cf_r,A0,A1,A2,Ri
      real(kind=DP)::z,term,jj,factor,MI,TI
      if(a.eq.c) then 
         phi_pow_cf_r=exp(z) ! always valid if a=c
         return 
      endif
      A0=1.0_DP

      A1=1.0_DP+z*a/c
!      write(*,*) 'A1=',A1
      do j=2,N_max
         jj=real(j,DP)
         Ri=(a+jj-1.0_DP)/(jj*(c+jj-1.0_DP))
         A2=A1+(A1-A0)*Ri*z
         A0=A1
         A1=A2

         if(abs(A0-A1).lt.eps) then
            exit
         endif
      enddo
      phi_pow_cf_r=A1
      end function


! The phi function evaluated by N-terms powerseries
      ! using continued fractions
      ! z=complex
      function phi_pow_cf_r_c(a,c,z)
      use gamma_function
      implicit none
      integer::j
      integer,parameter::N_max=2500
      complex(kind=DP)::phi_pow_cf_r_c,A0,A1,A2
      real(kind=DP)::a,c,Ri,jj
      complex(kind=DP)::z,term,factor,MI,TI,tmp
      if(a.eq.c) then 
         phi_pow_cf_r_c=exp(z) ! always valid if a=c
         return 
      endif
      A0=dcmplx(1.0_DP,0.0_DP)
      A1=A0+z*dcmplx(a/c,0.0_DP)
      do j=2,N_max
!         write(*,*) 'A1(',j,')=',A1
         jj=real(j-1,DP)
!!!!         Ri=(a+jj-1.0_DP)/(jj*(c+jj-1.0_DP))
         Ri=(a+jj)/((jj+1.0_DP)*(c+jj))
         tmp=(A1-A0)*z
         A2=tmp*dcmplx(Ri,0.0_DP)+A1
         A0=A1
         A1=A2
         if(abs(tmp).lt.eps) then
            exit
         endif
      enddo
!!      write(*,*) 'j=',j
      phi_pow_cf_r_c=A1
      end function


! The phi function evaluated by N-terms powerseries
      ! using continued fractions
      ! z=complex
      function phi_as_cf_r_c(a_r,c_r,z)
      use gamma_function
      implicit none
      integer::i
      integer,parameter::N_max=1000
      complex(kind=DP)::phi_as_cf_r_c,A1,Am1,Am2
      real(kind=DP)::a_r,c_r
      complex(kind=DP)::z,term,ii,Ri,a,c
      complex(DP),parameter::ett=dcmplx(1.0_DP,0.0_DP)
      if(a_r.eq.c_r) then 
         phi_as_cf_r_c=exp(z)  ! always valid if a=c
         return 
      endif
      a=dcmplx(a_r)
      c=dcmplx(c_r)
      Am2=ett
      if(real(z,DP).gt.0.0_DP) then
         Am1=ett+(c-a)*(ett-a)/z
         do i=2,N_max
            ii=dcmplx(i,0.0_DP)
            Ri=(c-a+ii-ett)*(ii-a)/ii
            term=(Am1-Am2)*Ri/z
         
            A1=Am1+term
            Am2=Am1
            Am1=A1

            if(abs(term).lt.eps) then
         !      exit
            endif
         enddo
         A1=A1*exp(z)*gamma_c(c)*(z**(a-c))/gamma_c(a)
      else
         Am2=ett
         Am1=ett+(ett+a-c)*a/z
         do i=2,N_max
            ii=dcmplx(i,0.0_DP)
            Ri=(a-c-ii)*(a-ii-ett)/ii
            term=(Am1-Am2)*Ri/z
            A1=Am1+term
            Am2=Am1
            Am1=A1
            if(abs(term).lt.eps) then
       !        exit
            endif
         enddo
         write(*,*) 'A1=',A1
         A1=A1*gamma_c(c)*(z**(-a))/gamma_c(c-a)
      endif
      write(*,*) 'j=',i
      phi_as_cf_r_c=A1
      end function

      ! Full complex constants
      function phi_pow_cf_c(a,c,z)
      use gamma_function
      implicit none
      integer::j
      integer,parameter::N_max=1500
      complex(kind=DP)::phi_pow_cf_c,A0,A1,A2
      complex(kind=DP)::a,c,Ri
      complex(kind=DP)::z,term,jj,factor,MI,TI,jjm1
      if(a.eq.c) then 
         phi_pow_cf_c=exp(z) ! always valid if a=c
         return 
      endif
      A0=dcmplx(1.0_DP,0.0_DP)
      A1=A0+z*a/c
      do j=2,N_max
         jjm1=dcmplx(real(j,DP)-1.0_DP,0.0_DP)
         jj=dcmplx(j,0.0_DP)
         Ri=(a+jjm1)/(jj*(c+jjm1))
         A2=A1+(A1-A0)*Ri*z
         A0=A1
         A1=A2
         if(abs(A0-A1).lt.eps) then
            exit
         endif

      enddo
!      write(*,*) 'j=',j
      phi_pow_cf_c=A1
      end function


   


      function phi_1_3_2(z)
      use gamma_function
      implicit none
      complex(DP)::z,phi_1_3_2,c0,c1,summa
      real(DP)::jj
      integer::j
      integer,parameter::N_max=300

      c0=dcmplx(2.0_DP/sqrt(Pi),0.0_DP)
      summa=c0
      do j=1,N_max
         jj=real(j,DP)
         c1=z*c0/dcmplx(1.5_DP+jj,0.0_DP)
         c0=c1
         c1=(z**dcmplx(jj,0.0_DP))/dcmplx(gamma_r(1.5_DP+jj),0.0_DP)
         summa=summa+c1
         if(abs(c1/summa).lt.eps) then 
            exit
         endif
      enddo
      phi_1_3_2=summa*dcmplx(sqrt(Pi)/2.0_DP,0.0_DP)
      end function
      

    


      

      function pochammer(a,k)
      implicit none
      integer:: k,j
      complex(kind=DP):: a,pochammer
      pochammer=a
      do j=1,k-1
         pochammer=pochammer*(a+DCMPLX(j))
      enddo
      end function

      function pochammer_r(a,k)
      implicit none
      integer:: k,j
      real(kind=DP):: a,pochammer_r
      pochammer_r=1.0_DP
      do j=0,k-1
         pochammer_r=pochammer_r*(a+real(j,DP))
      enddo
      end function




      end module
      
