!     
!     Algorithms for computing the K-Bessel function K_iR(U)
!     
!     Copyright : Dennis Hejhal
!   
!     usage: 
!      CALL KBMAT(R,U)
!     
!     dependency on modules: 
!         cray_functions
!         bes_globals
!            declares a lot of constants needed, plus the double precision parameter, DP
!         
!
!
!     The explicit accuracy can be found elsewhere (e.g. DAH notes)
!     but for testing for some small R~1000 and X up to ~ R 
!     I got between 14-15 significant digits with this precision
!     (comparing with maple)
!

      module bes_globals
      use commonvariables      
      real(kind=DP),save :: PIH,RTM
      integer, save :: NN1R,NN1L,NN2,NN3
      logical::bes_glob_inited
      integer, PARAMETER::MG=16
      contains 

      subroutine init_bes_globals()
   
      implicit none
      
      integer  :: MM,J,K,N
      integer, parameter :: MG=16
      real (kind=DP), dimension(16) :: A1,W
      integer, dimension(20) ::FAC
!!$
      NN1R=4
      NN1L=16
      NN2=8
      NN3=4
      MM=MG/16
      RTM=(1.0_DP)/real(2*MM,DP)
      bes_glob_inited=.true.
      end subroutine 
      end module








      module kbessel
      use commonvariables
      use bes_globals
      use gauss_quad
      use confluent
      real(DP),dimension(:,:),allocatable,save::A_uk
      integer,parameter::N_uk=20 ! The max number of terms in u_k
      INTEGER FEL
      character(300)::errormsg
      private::fel
      
      contains

   


      ! computes K_iR(X)=K_classical_iR(X)*exp(R*pi/2)
      ! R-K-Bessel to signify the iR
      function RKBessel(R,X)
      use bes_globals
      use commonvariables
      implicit none
      real(kind=DP) :: R,X,C,S,S2,T1,T2,RKBES,RKBessel
      integer :: j,stat
 !     logical::low_prec
      !! Initialize global variables
!      if(do_exceptional) then
!         RKBEssel=bessk_real(r,x)
!         return
!      endif
      stat=1
      rkbes=0.0D0
      if(.not. bes_glob_inited) then
         call init_bes_globals()
      endif
      if(R.le.0.0_DP) then
         if(R.eq.0.0_DP) then
            if(low_prec) then
               rkbessel=bessk0(x)
            else               
               RKBessel = kbes_rec(0.0_DP,x,stat) 
            endif
            return
         endif
         R=-1.0_DP*R            ! K-Bessel is symmetric after all
      endif
!      if(.false.) then
      ! It really is only best for R<<1000, but I don't put a test
      ! anyway since it takes up valuable time

!      low_prec=.true.
!     low_prec=.false.
      if(low_prec) then
         RKBes=kbes_asympt(r,x,stat) 
      elseif(R.lt.200.0_DP) then
         if(x.lt.max(2.0_DP,R*0.3_DP)) then
            call bessk_pow(R,x,rkbes,stat)
         elseif(x.lt.(9.0_DP+R*1.47_DP)) then
            rkbes=kbes_rec(R,x,stat)
         endif
      elseif(R.lt.500.0_DP) then
         if(x.lt.max(1.0_DP,R*0.3_DP)) then
            call bessk_pow2(R,x,rkbes,stat)
         elseif(x.lt.700.0_DP) then
            rkbes=kbes_rec(R,x,stat)
         endif
      elseif(R.lt.1000.0_DP) then
         if(x.lt.100.0_DP) then
            call bessk_pow3(R,x,rkbes,stat)
         elseif((x.gt.700.0_DP).and.(x.lt.1200.0_DP)) then  
            rkbes=kbes_rec(R,x,stat)
         endif
      endif
      if(stat.ne.0) then        !! WE called a K-Bessel routine successfully
!     !      ! Otherwise we go down to the gauss-quad-functions below
!     !      ! (DAH-stlye, it is more optimized right now than mine )
         if((R.lt.50).and.(X.ge.0.1)) then
           ! then we use the recursive algorithm
            rkbes=kbes_rec(R,X,stat)
         else
            rkbes=RKBessel_int(R,X)
         endif
      endif
      rkbessel=rkbes
      end function RKBessel

      function RKBessel_int(R,X)
      use bes_globals
      use commonvariables
      implicit none
      real(kind=DP) :: R,X,C,S,S2,T1,T2,RKBES,RKBessel_int
      integer :: j,stat
      S2=(5.0_DP)/R
      T1=(1.0_DP)/(3.0_DP)
      T2=(2.0_DP)/(3.0_DP)
      S=MIN(T1,S2**T2)
      C=(ABS(R-X))/R
                                !      write(*,*) 'C=',C, 'S=',S
      IF(C.LE.S) THEN
         !write(*,*) 'KBES2'
                                !            If x close to R
         CALL KBES2(X,R,RKBES)
      ELSE IF(X.LT.R) THEN
         !write(*,*) 'KBES1'
                                !            if X<R
         CALL KBES1(X,R,RKBES)
      ELSE
         !write(*,*) 'KBES3'
                                !            if X>R
         CALL KBES3(X,R,RKBES)
      END IF
      RKBessel_int = RKBES
      END function RKBessel_int







      function bessk_real(r,x)
      implicit none
      real(DP)::bessk_real,r,x,psi
      psi=psi_r(0.5_DP+r,2.0_DP*r+1.0_DP,2.0_DP*x)
      bessk_real=sqrtPi*((2.0_DP*x)**r)*exp(-x)*psi      
      end function




!C     
      SUBROUTINE KBES1(YN,R,RKBES)
!      use cray_functions 
      use bes_globals
      use gauss_quad
      use commonvariables
      implicit none
!      integer, PARAMETER :: MG=16
      real(kind=DP) :: YN,R,RKBES
      real(kind=DP), dimension(MG):: U,V,VPRM,T,E
      real(kind=DP), dimension(MG):: E1,E2,A9,B9,Z

      real(kind=DP) :: ZCRIT1,ZCRIT2,ETA,X,T8,C25
      real(kind=DP) :: DD,RDD,U1,C24,C23,S,C6,S6,XXR,XXL
      real(kind=DP) :: UT,S3,C3,C,S1,C74,VT,TT,CC23
      real(kind=DP) :: H1,H2,XJ,ZT
      real(kind=DP) :: AQ0,AQ1,AQ2,AQ3,AQ4,BQ0,BQ1,BQ2,BQ3,BQ4
      real(kind=DP) :: SM,SN2,CXX,X1,X2
      real(kind=DP) :: AA,C3T,SN1,C1,VX, XXH,C8
      real(kind=DP) :: U0,HH
      integer :: I,J1,J2,N1,N2,KK,NH,JH
!      WRITE(*,*) 'hej NN1R=',NN1R
      if(.not. bes_glob_inited) then
         call init_bes_globals()
      endif
      ! Error handling
      if(YN.eq.0.0_DP) then
         write(*,*) 'Execution aborted! ERROR: Y=0'
         STOP
      endif
!      WRITE(*,*) 'hej NN1R=',NN1R,' YN=',YN
      N1=NN1R
      N2=NN1L
!C     ADDITIONAL PARAMETERS:
      ZCRIT1=0.25_DP
      ZCRIT2=0.25_DP
      ETA=0.00_DP  
!C     
!C     
!C     for ETA:   at level exp(-51)
!C     .00   R<98        .33    R>492
!C     .02   R>98        .50    R>1430
!C     .05   R>113       .66    R>5831
!C     .10   R>144       .80    R>31157
!C     .20   R>238
!C     
      X=YN
      T8=R/X
      C25=SQRT((R-X)*(R+X))
      DD=(C25)/R
      RDD=(1.0_DP)/DD
      U1=LOG((R+C25)/X) 
      C24=110.00_DP/C25
      C23=SQRT(C24)  
      S=T8*(U1-DD)
      C6=COS(X*S)
      S6=SIN(X*S)
      XXR=0.0_DP
      XXL=0.0_DP


!      WRITE(*,*) 'hej 0'

     
      IF (R.LT. (40.0_DP)) THEN
         DO I=2,7
            UT=U1+REAL(I,DP)
            S3=SINH(UT)
            C3=COSH(UT)
            C=T8*UT-S
            S1=C/S3
            C74=SQRT((S3-C)*(S3+C))
            C1=C74/S3
            VT=(2.0_DP)*ATAN(S1/(1.0_DP+C1))
            TT=-1.0_DP*X*C3*C1 + R*(PIH-VT)
            IF(TT.LT.(-55.0_DP)) THEN
               CC23=REAL(I,DP)
               GOTO 9382
            ENDIF
 9381     END DO
 9382     H1=MIN(C23,CC23)/REAL(N1,DP)
         H2=(DD-ETA*(DD**3))/REAL(N2,DP)
!*     
!*     
      ELSE

         H1=MIN(C23,2.0_DP)/REAL(N1,DP)
         H2=(DD-ETA*(DD**3))/REAL(N2,DP)

      ENDIF
!*     
!*     
!      WRITE(*,*) 'hej 1'

!C     DIR$  NEXTSCALAR
      DO 300  J1=1,N1
         XJ=REAL(-1+J1,DP)
         AA=XJ*H1
         IF(AA.GE.ZCRIT1) THEN
            GOTO 33
         ELSE
            GOTO 34
         ENDIF
 33      DO 400  KK=1,MG
            U(KK)=U1+AA+(RTM)*H1*AD(KK)
            S3=SINH(U(KK))
            C3=COSH(U(KK))
            C=T8*U(KK)-S
            S1=C/S3
            C74=SQRT((S3-C)*(S3+C))
            C1=C74/S3  
            V(KK)=(2.0_DP)*ATAN(S1/(1.0_DP+C1))
            T(KK)=-1.0_DP*X*C3*C1+R*(PIH-V(KK))
            E(KK)=EXP(MAX(T(KK),-70.0_DP))
            VPRM(KK)=(T8*S3-C*C3)/(C74*S3)
            E1(KK)=E(KK)*WD(KK)  
            E2(KK)=E1(KK)*VPRM(KK)
 400     END DO
         GOTO 35
 34      DO 402  KK=1,MG
            Z(KK)=AA+(RTM)*H1*AD(KK)
            ZT=Z(KK)
            AQ0=(ZT**14)*(RF(16)*(DD+RG(17)*ZT)+                                    &
     &           RF(18)*(DD+RG(19)*ZT)*ZT*ZT)
            AQ1=(ZT**10)*(RF(12)*(DD+RG(13)*ZT)+                                    &
     &           RF(14)*(DD+RG(15)*ZT)*ZT*ZT)+AQ0
            AQ2=(ZT**6)*(RF(8)*(DD+RG(9)*ZT)+                                    &
     &           RF(10)*(DD+RG(11)*ZT)*ZT*ZT)+AQ1
            AQ3=(ZT**2)*(RF(4)*(DD+RG(5)*ZT)+                                    &
     &           RF(6)*(DD+RG(7)*ZT)*ZT*ZT)+AQ2
            AQ4=RF(2)*(DD+RG(3)*ZT)+AQ3
            A9(KK)=AQ4*(RDD)
            BQ0=(ZT**14)*(RF(15)*(DD+RG(16)*ZT)+                                    &
     &           RF(17)*(DD+RG(18)*ZT)*ZT*ZT)
            BQ1=(ZT**10)*(RF(11)*(DD+RG(12)*ZT)+                                    &
     &           RF(13)*(DD+RG(14)*ZT)*ZT*ZT)+BQ0
            BQ2=(ZT**6)*(RF(7)*(DD+RG(8)*ZT)+                                    &
     &           RF(9)*(DD+RG(10)*ZT)*ZT*ZT)+BQ1
            BQ3=(ZT**2)*(RF(3)*(DD+RG(4)*ZT)+                                    &
     &           RF(5)*(DD+RG(6)*ZT)*ZT*ZT)+BQ2
            BQ4=(DD+RG(2)*ZT)+BQ3
            B9(KK)=BQ4*(RDD)
            C3T=1.0_DP+ZT*DD*B9(KK)
            SM=1.0_DP+(RDD*ZT)
            SN1=SM+A9(KK)*ZT*ZT
            SN2=(2.0_DP*SM)+A9(KK)*ZT*ZT
            CXX=SQRT(A9(KK))*SQRT(SN2)
            S1=SM/(SN1)
            C1=ZT*(CXX/SN1)
            V(KK)=(2.0_DP)*ATAN(S1/(1.0_DP+C1))
            VX=-1.0_DP*B9(KK)-(B9(KK)-A9(KK))*RDD*ZT
            VPRM(KK)=VX/(CXX*SN1)
            T(KK)=R*(PIH-V(KK)-C3T*C1)
            E(KK)=EXP(MAX(T(KK),-70.0_DP)) 
            E1(KK)=E(KK)*WD(KK)
            E2(KK)=E1(KK)*VPRM(KK)
 402      END DO
 35       X1=sum( E1(MG:1:-1) )
         X2=sum( E2(MG:1:-1) )
!        X1=ssum(MG,E1,-1)
!         X2=ssum(MG,E2,-1)


         XXR=XXR+(RTM)*H1*(X1*C6-X2*S6)
         IF(T(MG).LT.-55.0_DP) THEN
            GOTO 3
         ENDIF
 300   END DO

!C     DIR$  NEXTSCALAR
 3    DO 600  J2=1,N2
         XJ=REAL(-1+J2,DP)
         AA=-1.0_DP*XJ*H2  
         IF(AA.LE.(-ZCRIT2)) THEN
            GOTO 53
         ELSE
            GOTO 54
         ENDIF
 53      DO 700  KK=1,MG
            U(KK)=U1+AA-(RTM)*H2*AD(KK)
            S3=SINH(U(KK))
            C3=COSH(U(KK))
            C=T8*U(KK)-S
            S1=C/S3
            C74=SQRT((S3-C)*(S3+C))
            C1=-1.0_DP*C74/S3
            V(KK)=PI-(2.0_DP)*ATAN(S1/(1.0_DP-C1))
            T(KK)=-1.0_DP*X*C3*C1+R*(PIH-V(KK))
            E(KK)=EXP(MAX(T(KK),-70.0_DP))
            VPRM(KK)=-1.0_DP*(T8*S3-C*C3)/(C74*S3)
            E1(KK)=E(KK)*WD(KK)  
            E2(KK)=E1(KK)*VPRM(KK)
 700     END DO
         GOTO 55
 54      DO 702  KK=1,MG
            Z(KK)=AA-(RTM)*H2*AD(KK)
            ZT=Z(KK)
            AQ0=(ZT**14)*(RF(16)*(DD+RG(17)*ZT)+                                    &
     &           RF(18)*(DD+RG(19)*ZT)*ZT*ZT)
            AQ1=(ZT**10)*(RF(12)*(DD+RG(13)*ZT)+                                    &
     &           RF(14)*(DD+RG(15)*ZT)*ZT*ZT)+AQ0
            AQ2=(ZT**6)*(RF(8)*(DD+RG(9)*ZT)+                                    &
     &           RF(10)*(DD+RG(11)*ZT)*ZT*ZT)+AQ1
            AQ3=(ZT**2)*(RF(4)*(DD+RG(5)*ZT)+                                    &
     &           RF(6)*(DD+RG(7)*ZT)*ZT*ZT)+AQ2
            AQ4=RF(2)*(DD+RG(3)*ZT)+AQ3
            A9(KK)=AQ4*(RDD)
            BQ0=(ZT**14)*(RF(15)*(DD+RG(16)*ZT)+                                    &
     &           RF(17)*(DD+RG(18)*ZT)*ZT*ZT)
            BQ1=(ZT**10)*(RF(11)*(DD+RG(12)*ZT)+                                    &
     &           RF(13)*(DD+RG(14)*ZT)*ZT*ZT)+BQ0
            BQ2=(ZT**6)*(RF(7)*(DD+RG(8)*ZT)+                                    &
     &           RF(9)*(DD+RG(10)*ZT)*ZT*ZT)+BQ1
            BQ3=(ZT**2)*(RF(3)*(DD+RG(4)*ZT)+                                    &
     &           RF(5)*(DD+RG(6)*ZT)*ZT*ZT)+BQ2
            BQ4=(DD+RG(2)*ZT)+BQ3
            B9(KK)=BQ4*(RDD)
            C3T=1.0_DP+ZT*DD*B9(KK)
            SM=1.0_DP+(RDD*ZT)
            SN1=SM+A9(KK)*ZT*ZT
            SN2=(2.0_DP*SM)+A9(KK)*ZT*ZT
            CXX=SQRT(A9(KK))*SQRT(SN2)
            S1=SM/(SN1)
            C1=ZT*(CXX/SN1)
            V(KK)=PI-(2.0_DP)*ATAN(S1/(1.0_DP-C1))
            VX=-1.0_DP*B9(KK)-(B9(KK)-A9(KK))*RDD*ZT
            VPRM(KK)=VX/(CXX*SN1)
            T(KK)=R*(PIH-V(KK)-C3T*C1)
            E(KK)=EXP(MAX(T(KK),-70.0_DP)) 
            E1(KK)=E(KK)*WD(KK)
            E2(KK)=E1(KK)*VPRM(KK)
 702     END DO 
 55      X1=sum( E1(MG:1:-1) )
         X2=sum( E2(MG:1:-1) )
         !      X1=ssum(MG,E1,-1)
         !         X2=ssum(MG,E2,-1)
         XXL=XXL+(RTM)*H2*(X1*C6-X2*S6)
         IF(T(MG).LT.-55.0_DP) THEN
            GOTO 4
         ENDIF
 600   END DO
 4    RKBES=XXL+XXR
!C     
!C     
      XXH=0.0_DP
      IF(R.LT.(75.0_DP)) THEN
         U0=U1-DD
         NH=16*(1+INT(U0))
         HH=U0/REAL(NH,DP)
         DO 9711 JH=1,NH
            XJ=REAL(-1+JH,DP)
            AA=XJ*HH
!CPAR$  DOALL
            DO 9722 KK=1,MG
               U(KK)=AA+(RTM)*HH*AD(KK)
               C3=COSH(U(KK))
               C8=COS(R*U(KK))
               T(KK)=X*C3-(PIH)*R
               E(KK)=EXP(MAX(T(KK),-70.0_DP))*C8
               E1(KK)=E(KK)*WD(KK)
 9722       END DO

            X1=sum( E1(MG:1:-1) )
!            X1=ssum(MG,E1,-1)
            XXH=XXH+(RTM)*HH*X1
 9711     END DO
      ENDIF
      RKBES=RKBES+XXH
      RETURN
      END SUBROUTINE KBES1
!C     
!C     
      SUBROUTINE KBES2(YN,R,RKBES)
!      use cray_functions
      use bes_globals 
      use commonvariables
      implicit none 
!      integer, PARAMETER :: MG=16
      real(kind=DP) :: YN,R,RKBES
      real(kind=DP), dimension(MG) :: U,V
      real(kind=DP), dimension(MG) :: VPRM,T,E,E4
      real(kind=DP), dimension(MG) :: A9,B9
      real(kind=DP) :: X,H,G1,G2,U12,H1,XX1,XJ,AA
      real(kind=DP) :: S3,C3,S33,S1,C21,C1,C6,S6
      real(kind=DP) :: UT,AQ0,AQ1,AQ2,AQ3,BQ0,BQ1,BQ2,BQ3
      real(kind=DP) :: V11,V12,X1,U11
      real(kind=DP) :: UCRIT
      integer :: J1,N1,KK
      if(.not. bes_glob_inited) then
         call init_bes_globals()
      endif
      ! Error handling
      if(YN.eq.0.0_DP) then
         write(*,*) 'Execution aborted! ERROR: Y=0'
         STOP
      endif
      N1=NN2
!C     ADDITIONAL PARAMETER:
      UCRIT=0.125_DP
      X=YN
      H=R-X
      G1=(4.0_DP)*SQRT(3.0_DP)
      G2=-1.0_DP/3.0_DP
      U12=G1*(R**G2)
      IF (R.LT.(120.0_DP)) THEN
         U11=LOG((8.0_DP)+(200.0_DP/R))
      ELSE IF (R.LT.(333.0_DP)) THEN
         U11=1.5_DP
      ELSE
         U11=U12
      ENDIF
      H1=U11/REAL(N1,DP)
      XX1=0.0_DP
!C     DIR$  NEXTSCALAR
      DO 300  J1=1,N1
         XJ=REAL(-1+J1,DP)
         AA=XJ*H1
         IF(AA.GE.UCRIT) THEN
            GOTO 33
         ELSE
            GOTO 34
         ENDIF
!CPAR$  DOALL
 33      DO 400  KK=1,MG
            U(KK)=AA+(RTM)*H1*AD(KK)
            S3=SINH(U(KK))
            C3=COSH(U(KK))
            S33=S3/(U(KK))
            S1=U(KK)/S3
            C21=SQRT((S33+1.0_DP)*(S33-1.0_DP))
            C1=(C21)/S33
            C6=COS(H*U(KK))
            S6=SIN(H*U(KK))
            V(KK)=(2.0_DP)*ATAN(S1/(1.0_DP+C1))
            VPRM(KK)=(S33 - C3)/(S3*C21)
            T(KK)=-1.0_DP*X*C3*C1 + R*(PIH-V(KK))
            E(KK)=EXP(MAX(T(KK),-70.0_DP))
            E4(KK)=E(KK)*(C6-S6*VPRM(KK))*WD(KK)
 400     END DO
         GO TO 35 
!CPAR$  DOALL
 34      DO 402 KK=1,MG
            U(KK)=AA+(RTM)*H1*AD(KK)
            UT=U(KK)  
            AQ0=(UT**14)*RF(16)
            AQ1=(UT**10)*(RF(12)+RF(14)*UT*UT)+AQ0
            AQ2=(UT**6)*(RF(8)+RF(10)*UT*UT) + AQ1  
            AQ3=(UT**2)*(RF(4)+RF(6)*UT*UT) + AQ2
            A9(KK)=RF(2)+AQ3
            BQ0=(UT**14)*RF(17)
            BQ1=(UT**10)*(RF(13)+RF(15)*UT*UT)+BQ0
            BQ2=(UT**6)*(RF(9)+RF(11)*UT*UT) + BQ1  
            BQ3=(UT**2)*(RF(5)+RF(7)*UT*UT) + BQ2
            B9(KK)=RF(3)+BQ3
            C3=1.0_DP+A9(KK)*(UT*UT)
            S33=1.0_DP+B9(KK)*(UT*UT)
            V11=B9(KK)*(2.0_DP+B9(KK)*UT*UT)
            V12=SQRT(V11)
            VPRM(KK)=(B9(KK)-A9(KK))/(V12*S33)
            S1=(1.0_DP)/(S33)
            C1=U(KK)*(V12/S33)  
            C6=COS(H*U(KK))
            S6=SIN(H*U(KK))
            V(KK)=(2.0_DP)*ATAN(S1/(1.0_DP+C1))
            T(KK)=-1.0_DP*X*C3*C1 + R*(PIH-V(KK))
            E(KK)=EXP(MAX(T(KK),-70.0_DP)) 
            E4(KK)=(C6-S6*VPRM(KK))*E(KK)*WD(KK)
 402     END DO 
35       X1=sum( E4(MG:1:-1) )
         !  X1=ssum(MG,E4,-1)
         XX1=XX1+(RTM)*H1*X1
         IF (T(MG).LT.-55.0_DP) THEN
            GOTO 5
         ENDIF
 300  END DO
 5    RKBES=XX1
      RETURN
      END SUBROUTINE KBES2
!C     
!C     R < Y
      SUBROUTINE KBES3(YN,R,RKBES)  
      use bes_globals
      use commonvariables
      IMPLICIT none
!      integer, PARAMETER :: MG=16
      real(kind=DP) :: YN,R,RKBES
      real(kind=DP), DIMENSION(MG) :: U,V,T,E,E4
      real(kind=DP) :: X,RPIH,S,XC,AL,XCRAL,XX11,C23
      real(kind=DP) :: U11,U22,H1,XX1,XJ,AA,S3,C3,S33,S1
      real(kind=DP) :: C1, T97,C,X1
      integer :: J1,N1,KK
      ! Error handling
      if(YN.eq.0.0_DP) then
         write(*,*) 'Execution aborted! ERROR: Y=0'
         STOP
      endif
      if(NN3.eq.0) then  ! bes_global is not inited
         call init_bes_globals()
      endif
      N1=NN3
      X=YN
      RPIH=R*PIH
      S=R/X
      XC=SQRT((X+R)*(X-R))
      C=(XC)/X 
      AL=(2.0_DP)*ATAN(S/(1.0_DP+C))
      XCRAL=XC + R*AL
!      write(*,*) 'diff=',RPIH-XCRAL
      IF(RPIH-XCRAL.LT.-125.0_DP) THEN
         XX11=0.0_DP
         GO TO 7
      END IF
      C23=110.00_DP/(XC)
      U11=SQRT(C23)  
      IF (R.LT.(40.0_DP)) THEN
         U22=LOG((8.0_DP)+(125.0_DP/X))
      ELSE
         U22=2.0_DP
      ENDIF
      H1=MIN(U11,U22)
      H1=H1/REAL(N1,DP)
      XX1=0.0_DP
      DO 300 J1=1,N1
         XJ=REAL(-1+J1,DP)
         AA=XJ*H1
         DO 400 KK=1,MG
            U(KK)=AA+(RTM)*H1*AD(KK)
            S3=SINH(U(KK))
            C3=COSH(U(KK))
            S33=S3/(U(KK))
            S1=S*(U(KK)/S3)
            C1=SQRT((S33+S)*(S33-S)) / S33
            V(KK)=2.0_DP*ATAN(S1/(1.0_DP+C1))
            T97=-1.0_DP*X*C3*C1-R*V(KK)
            T(KK)=T97+XCRAL
            E(KK)=EXP(MAX(T(KK),-70.0_DP))
            E4(KK)=E(KK)*WD(KK)  
 400     END DO
         X1=sum( E4(MG:1:-1) )
         XX1=XX1+(RTM)*H1*X1
         IF(T(MG).LT.-55.0_DP) THEN
            GO TO 5
         END IF
 300   END DO
 5    XX11=XX1*EXP(RPIH-XCRAL)
 7    RKBES=XX11
      RETURN
      END SUBROUTINE KBES3
!C     

     




!      FUNCTION bessi0_s(x) 
      FUNCTION bessi0(x) 
      use bes_globals
      use commonvariables
!      USE nrtype;USE nrutil,ONLY :poly 
      IMPLICIT NONE 
      REAL(DP),INTENT(IN)::x 
      REAL(DP)::bessi0        ! Returns the modified Bessel function I_0(x)for any real x 
      REAL(DP)::ax 
      REAL(DP),DIMENSION(7)::p =(/1.0_dp,3.5156229_dp,                  &
     &     3.0899424_dp,1.2067492_dp,0.2659732_dp,0.360768e-1_dp,            &
     &     0.45813e-2_dp/)      !Accumulate polynomials in double precision. 
      REAL(DP),DIMENSION(9)::q =(/0.39894228_dp,0.1328592e-1_dp,        &
     &     0.225319e-2_dp,-0.157565e-2_dp,0.916281e-2_dp,                    & 
     &     -0.2057706e-1_dp,0.2635537e-1_dp,-0.1647633e-1_dp,                & 
     &     0.392377e-2_dp/) 
      ax=abs(x) 
      if (ax <3.75_DP) then        !Polynomial  fit. 
         bessi0=poly(real((x/3.75_dp)**2,dp),p) 
      else 
         bessi0=(exp(ax)/sqrt(ax))*poly(real(3.75_dp/ax,dp),q) 
      end if 
      END FUNCTION bessi0


!      FUNCTION bessk0_s(x) 
      FUNCTION bessk0(x) 
      use commonvariables
      use bes_globals
!      USE nrtype;
!      USE nrutil,ONLY :assert,poly 
!      USE nr,ONLY :bessi0 
      IMPLICIT NONE 
      REAL(kind=DP),INTENT(IN)::x 
      REAL(kind=DP)::bessk0 
!Returns the modified Bessel functi n K 0 (x )f r p sitive real x . 
      REAL(kind=DP)::y  !Accumulate polynomials in double precision. 
      REAL(kind=DP),DIMENSION(7)::p =(/-0.57721566_dp,0.42278420_dp,    & 
     &     0.23069756_dp,0.3488590e-1_dp,0.262698e-2_dp,0.10750e-3_dp,       & 
     &     0.74e-5_dp/) 
      REAL(kind=DP),DIMENSION(7)::q =(/1.25331414_dp,-0.7832358e-1_dp,  & 
     &     0.2189568e-1_dp,-0.1062446e-1_dp,0.587872e-2_dp,                  & 
     &     -0.251540e-2_dp,0.53208e-3_dp/) 
      if(x<0.0_DP) then
         write(*,*) 'ERROR: the argument of K_0(X) is <0!!'
         stop
      endif
      !call assert(x >0.0_DP, 'bessk0_s arg' ) 
      if (x <=2.0_DP)then ! Polynomial  fit. 
         y=x*x/4.0_DP 
         bessk0=(-log(x/2.0_DP)*bessi0(x))+poly(y,p) 
      else 
         y=(2.0_DP/x) 
         bessk0=(exp(-x)/sqrt(x))*poly(y,q) 
      end if 
      END FUNCTION bessk0


      ! polynomial evaluation from numerical recipies
      FUNCTION poly(x,coeffs) 
      use commonvariables
      use bes_globals
      implicit none
      REAL(kind=DP),INTENT(IN)::x 
      REAL(kind=DP),DIMENSION(:),INTENT(IN)::coeffs 
      REAL(kind=DP)::poly 
      REAL(kind=DP)::pow 
      REAL(kind=DP),DIMENSION(:),ALLOCATABLE ::vec 
      INTEGER::i,n,nn
      n=size(coeffs) 
      if (n <=0)then 
         poly=0.0_dp 
      else if (n <8)then   ! if we want to parallellize or not 
         poly=coeffs(n) 
         do i=n-1,1,-1 
            poly=x*poly+coeffs(i) 
         end do 
      else 
         allocate(vec(n+1)) 
         pow=x 
         vec(1:n)=coeffs 
         do 
            vec(n+1)=0.0_dp 
            nn=ishft(n+1,-1) 
            vec(1:nn)=vec(1:n:2)+pow*vec(2:n+1:2) 
            if (nn.eq.1) exit 
            pow=pow*pow 
            n=nn 
         end do 
         poly=vec(1) 
         deallocate(vec) 
      end if 
      END FUNCTION poly



      ! I thought about adding a test to make sure x>>0, but
      ! it is probably more efficcient to make sure of that
      ! beforehand
      ! Miller Backwards recursion (from Holgers Then)
      function kbes_rec(r, x,stat)   ! |r| <<1000, x >> 0 !
      use commonvariables
      use bes_globals
      implicit none
      integer::stat
      real(kind=DP),intent(in) :: r,x
      real(kind=DP) :: y, k, d, e, p, q, t,kbes_rec,mr
      integer :: n, m,nn,NMAX
      real(kind=DP) :: prec,tmp
      prec=it_prec
      NMAX=1000
      stat=0
!      write(*,*) 'in Holgers KBES,X,R=',X,R
      ! K_iR(-X)=K_iR(X) but we shouldn't input X<0 anyway
      ! To make a specific test is time-consuming
      p=0.25_DP+r*r
      q=2.0_DP*x-2.0_DP
      t=0.0_DP !/*arbitrary*/
      k=1.0_DP !/*arbitrary*/;
!      for( n=1; t-k<-prec || t-k>prec || n<4; n<<=1) {
      n=1
      do nn=1,NMAX              ! I want to make sure we don't keep on too long
!         if(x.gt.700) then
!            write(*,*) 'n=',n,'nn=',nn
!            write(*,*) 't=',t,' k=',k
!         endif
         if(abs(t-k) < prec) then
            if(n>=4) then
               exit
            endif
         endif
         n=n*2       
         t=k 
         y=1.0_DP               !/*arbitrary*/1; 
         k=1.0_DP
         d=1.0_DP
         tmp=2.0_DP*x-r*PI
         if(tmp.gt.1300.0_DP) then 
            kbes_rec=0.0_DP
            return
         endif
         e=exp( (log(2.0_DP*x/PI) + tmp)/(real(2*n,kind=DP)))
         
         !for( m=n; m; m--) { 
         do m=n,1,-1
            mr=real(m,kind=DP)
            y=(mr-1.0_DP+p/mr)/(q+(mr+1.0_DP)*(2.0_DP-y))
            k=e*(d+y*k)
            d=d*e
         enddo
         k=1.0_DP/k; 
      enddo
      if(.not.((k>-huge(k)).and.(k<huge(k)))) then         
         stat=2 !! ! this is a fatal error, the below is not
         errormsg='the K-bessel routine failed (k large) for x,R='//        &
     &    'x,R='//r2c(x,4)//','//r2c(R,4)//', value=' //r2c(k,10)
!         stop 
      endif
      if(nn.ge.NMAX) then 
         stat=1
         !! The error message can be displayed if wanted
         errormsg='the K-bessel routine failed (too many iterations)'//       &
     &        'x,R='//r2c(x,4)//','//r2c(R,4)//', value=' //r2c(k,10)
      endif
      kbes_rec=k ! exp( pi*r/2) * K_ir( x) !
      end function kbes_rec







      subroutine bessk_pow(R,x,kbess,stat)
      use commonvariables
      use gamma_function
      implicit none
      real(DP)::x,R,kbess
      integer::stat
      real(DP)::f0,fk,fk1,r0,r1,rk,rk1,rk2,c0,ck,ck1,xx,xh
      real(DP)::sigma0,sum,tmp_sin,tmp_cos,tmp_factor,kk,den
      real(DP)::gamma0,exp_Pih_R,xh2,th
      complex(DP)::gamma_tmp
      integer::j,k
      integer,parameter::N_max=200
!      write(*,*) 'R,x=',R,x
      stat=0  !! All is ok
      xh=0.5_DP*x
      xh2=xh*xh
!      gamma_tmp=lngamma(dcmplx(1.0_DP,R))
!      sigma0=imag(gamma_tmp)
!      gamma0=exp(-real(gamma_tmp,DP))  !=   |Gamma(1+iR)|^-1
      sigma0=dimag(lngamma_c(dcmplx(1.0_DP,R)))
      th=R*log(xh)-sigma0
      tmp_sin=dsin(th)
      tmp_cos=dcos(th)      
      tmp_factor=sqrt(PI/(R*sinh(PI*R)))
!      tmp_factor=(PI/sinh(PI*R))*gamma0
      f0=-1.0_DP*tmp_factor*tmp_sin
!      write(*,*) 'rsinhpir=',(R*sinh(PI*R))
!      write(*,*) 'tmp_fak=',tmp_factor      
!      write(*,*) 'tmp_sin=',tmp_sin
      r0=R*tmp_factor*tmp_cos
      r1=R*tmp_factor/(1.0_DP+R**2)*(tmp_cos+R*tmp_sin)
      c0=1.0_DP
      rk1=r0  ! r(k-1)
      fk1=f0  ! f(k-1)
      sum=f0
      exp_Pih_R=exp(Pihalf*R)   
      !k=1
!      kk=real(1.0,DP)
      fk=(fk1+rk1)/(1.0_DP+R*R)
      rk=r1
      ck=xh2
      sum=sum+ck*fk
      fk1=fk
      rk1=r1
      rk2=r0
      ck1=ck

!      write(*,*) 'term=',ck*fk,' sum=',sum
!      write(*,*) 'n_max=',n_max
!      write(*,*) 'prec=',kbes_prec
      do k=2,N_max         
         kk=real(k,DP)
         den=(kk*kk+R*R)
         fk=(kk*fk1+rk1)/den
         rk=((2.0_DP*kk-1.0_DP)*rk1-rk2)/den
         ck=xh2*ck1/kk
         sum=sum+ck*fk
!        write(*,*) 'term=',ck*fk,' sum=',sum,' den=',den
!        write(*,*) 'term/sum=',ck*fk/sum
!        write(*,*) 'term/sum*exp=',ck*fk/sum*exp_Pih_R
        if(abs(ck*fk/sum*exp_Pih_R).lt.kbes_prec) then
            exit
         endif
         
         fk1=fk
         rk2=rk1
         rk1=rk

         ck1=ck
      enddo
      if(k.ge.N_max) stat=1  ! We reached end of loop
!      write(*,*) 'sum=',sum
      kbess=sum*Pi/(2.0_DP*sinh(Pi*R))*exp(0.5_DP*Pi*R)   
!      write(*,*) 'kbes=',kbess
      kbess=sum*exp_Pih_R
!      write(*,*) 'kbes=',kbess
      end subroutine




      subroutine bessk_pow2(R,x,kbess,stat)
      use commonvariables
      use gamma_function
      implicit none
      real(DP)::x,R,kbess
      integer::stat
      real(DP)::f0,fk,fk1,r0,r1,rk,rk1,rk2,c0,ck,ck1,xx,xh,arg1
      real(DP)::sigma0,sum,tmp_sin,tmp_cos,tmp_factor,kk,den
      real(DP)::gamma0,exp_Pih_R,xh2
      complex(DP)::gamma_tmp
      integer::j,k
      integer,parameter::N_max=200
!      write(*,*) 'POW2 (',x,R,')'
      stat=0
      xh=0.5_DP*x
      xh2=xh*xh
      gamma_tmp=lngamma_c(dcmplx(1.0_DP,R))
      sigma0=imag(gamma_tmp)
!      write(*,*) 'lngamma(1+iR)=',gamma_tmp
!      write(*,*) 'arg(gamma(1+iR))=',sigma0
!      write(*,*) 'th=',R*log(xh)-sigma0
      tmp_sin=dsin(R*log(xh)-sigma0)      
      tmp_cos=dcos(R*log(xh)-sigma0)      
      tmp_factor=sqrt(PI/(R*sinh(PI*R)))
      exp_Pih_R=exp(0.5_DP*Pi*R)   
!      f0=-2.0_DP*gamma0*tmp_sin
!      r0=2.0_DP*R*gamma0*tmp_cos
!      r1=2.0_DP*R*gamma0/(1.0_DP+R**2)*(tmp_cos+R*tmp_sin)
      f0=-2.0_DP*tmp_sin
      r0=2.0_DP*R*tmp_cos
      r1=2.0_DP*R/(1.0_DP+R**2)*(tmp_cos+R*tmp_sin)
      c0=1.0_DP
      rk1=r0  ! r(k-1)
      fk1=f0  ! f(k-1)
      sum=f0
      !k=1
      kk=real(1,DP)
      fk=(kk*fk1+rk1)/(kk**2+R**2)
      rk=r1
      ck=xh2
      sum=sum+ck*fk
      fk1=fk
      rk1=r1
      rk2=r0
      ck1=ck
      do k=2,N_max
!         write(*,*) 'sum=',sum
         kk=real(k,DP)
         fk=(kk*fk1+rk1)/(kk**2+R**2)
         rk=((2.0_DP*kk-1.0_DP)*rk1-rk2)/(kk**2+R**2)
         ck=xh2*ck1/kk
         sum=sum+ck*fk
         if(abs(ck*fk/sum*exp_Pih_R*2.0_DP).lt.kbes_prec) then
                                !         write(*,*) 'abs(',ck,'*',fk,',/',sum,')', 'k=',k
            exit
         endif
         fk1=fk
         rk2=rk1
         rk1=rk
         ck1=ck
      enddo
      if(k.ge.N_max) stat=1  ! We reached end of loop
!      write(*,*) gamma0,'*',sum,'*Pi/',(2.0_DP*sinh(Pi*R)),'*',exp_Pih_R
      if(R.gt.15.0_DP) then
         arg1=-real(gamma_tmp,DP)-0.5_DP*Pi*R         
         kbess=Pi*sum*exp(arg1)
!*((1.0_DP+R**2)**(-0.25_DP))*               &
!     &        exp(R*(arg1))*sqrt(Pi/2.0_DP)
      else
         gamma0=exp(-real(gamma_tmp,DP))
         kbess=gamma0*sum*Pi/(2.0_DP*sinh(Pi*R))*exp_Pih_R
      endif
      end subroutine

      
      ! This is also the K-Bessel function evaluated as
      ! a power series 
      ! but intended for large R-values >>1000

      subroutine bessk_pow3(R,x,kbess,stat)
      use gamma_function
      use commonvariables
      implicit none
      real(DP)::x,R,kbess
      integer::stat
      real(DP)::f0,fk,fk1,r0,r1,rk,rk1,rk2,c0,ck,ck1,xx,xh,arg1
      real(DP)::dk,ek,dk1,ek1,qk,kk2
      real(DP)::sigma0,sum,tmp_sin,tmp_cos,tmp_factor,kk,den
      real(DP)::gamma0,exp_Pih_R,xh2,R2
      complex(DP)::gamma_tmp
      integer::j,k
      integer,parameter::N_max=300
      stat=0
      xh=0.5_DP*x
      xh2=xh*xh
      gamma_tmp=lngamma_c(dcmplx(1.0_DP,R))
      sigma0=imag(gamma_tmp)
      tmp_sin=dsin(R*log(xh)-sigma0)      
      tmp_cos=dcos(R*log(xh)-sigma0)      
      tmp_factor=sqrt(PI/(R*sinh(PI*R)))
      exp_Pih_R=exp(0.5_DP*Pi*R)   
!      f0=-2.0_DP*gamma0*tmp_sin
!      r0=2.0_DP*R*gamma0*tmp_cos
!      r1=2.0_DP*R*gamma0/(1.0_DP+R**2)*(tmp_cos+R*tmp_sin)

      R2=R*R

      f0=-2.0_DP*tmp_sin
      r0=2.0_DP*R*tmp_cos
      r1=2.0_DP*R/(1.0_DP+R**2)*(tmp_cos+R*tmp_sin)
      c0=1.0_DP

      rk1=r0  ! r(k-1)
      fk1=f0  ! f(k-1)
      sum=f0
      !k=1
      kk=real(1,DP)
      fk=(fk1+rk1)/(1.0_DP+R2)
      rk=r1
      ck=xh2
      sum=sum+ck*fk
      fk1=fk
      rk1=r1
      rk2=r0
      ck1=ck
      dk1=ck1*fk1
      ek1=ck1*rk1
      qk=rk2/rk1
      do k=2,N_max
         kk=real(k,DP)
         kk2=real(k*k,DP)
         dk=xh2*(dk1+ek1/kk)/(kk2+R2)
         ek=(xh2/kk)*(ek1*(2.0_DP*kk-1.0_DP-qk))/(kk2+R2)
!         fk=(kk*fk1+rk1)/(kk**2+R**2)
         rk=((2.0_DP*kk-1.0_DP)*rk1-rk2)/(kk2+R2)
!         ck=xh2*ck1/kk
         sum=sum+dk
         if(abs(dk/sum).lt.kbes_prec) then
            exit
         endif         
!         fk1=fk
         ek1=ek
         dk1=dk
         rk2=rk1
         rk1=rk
         if(rk1.eq.0.0_DP) then
            qk=0.0_DP
         else
            qk=rk2/rk1
         endif
      enddo            
      if(k.ge.N_max) stat=1  ! We reached end of loop
!      write(*,*) gamma0,'*',sum,'*Pi/',(2.0_DP*sinh(Pi*R)),'*',exp_Pih_R
      if(R.gt.15.0_DP) then
         arg1=-real(gamma_tmp,DP)-0.5_DP*Pi*R         
!         write(*,*) 'arg1=',-real(gamma_tmp,DP),-0.5_DP*Pi*R,'=',arg1
         kbess=Pi*sum*exp(arg1)
      else
         gamma0=exp(-real(gamma_tmp,DP))
         kbess=gamma0*sum*Pi/(2.0_DP*sinh(Pi*R))*exp_Pih_R
      endif
      end subroutine

      subroutine tab_a_uk()
      use commonvariables
      implicit none
      integer:: j,k,NN
      real(DP) kk,kkP1,kkP2j,den
      NN=N_uk
      if(allocated(a_uk)) then
         deallocate(a_uk)
      endif
      allocate(a_uk(0:NN+1,0:NN+1))
      a_uk(0,0)=1.0_DP
      do k=0,NN
         kk=real(k,DP)
         kkP1=real(k+1,DP)
         a_uk(k+1,0)=a_uk(k,0)*(4.0_DP*kk*(kkP1)+1.0_DP)/(8.0_DP*(kkP1))
         do j=1,k
            kkP2j=real(k+2*j,DP)
            den=8.0_DP*(kkP2j+1.0_DP)
            a_uk(k+1,j)=a_uk(k,j)*(4.0_DP*kkP2j*(kkP2j+1.0_DP)+1.0_DP)/   &
     &           den-a_uk(k,j-1)*(4.0_DP*(kkP2j-2.0_DP)*(kkP2j+1.0_DP)+   &
     &           5.0_DP)/den
         enddo
         a_uk(k+1,k+1)=-1.0_DP*((36.0_DP*kk*kkP1+5.0_DP)/(24.0_DP*kkP1))          &
     &        *a_uk(k,k)
 !        do j=0,k
 !           write(*,*) 'a(',k,j,')=',a_uk(k,j)
 !        enddo
      enddo
      end subroutine



      function u_rx( k, t)
      use commonvariables
      implicit none
      integer:: k,j
      real(DP) u_rx,t,t2, tt,sum;
      
      if(.not.(allocated(a_uk))) then
         call tab_a_uk()
      endif
      sum=0.0_DP
      tt=t**k
      if(t.gt.0.0_DP) then
         t2=t*t
      else  ! Remember we actually use another polynomial
         ! for the r>x case when t<0, but fortunately
         ! we just have to switch signs
         t2=-1.0_DP*t*t
      endif
      do j=0,k
         sum=sum+a_uk(k,j)*tt
         tt=tt*t2
      enddo
      u_rx=sum
      end function

      
      ! Asymptotic Hankel series
      function kbes_r( r, x, k_r,stat) ! r >> x > 0 !
      use commonvariables
      implicit none
      real(DP),intent(in):: x, r
      integer:: k_r,stat
      real(DP):: c,R1,R2, s, RR, t1,T2, w,kbes_r,T,X1
      real(DP)::gamma1,factor
      integer:: k;
      stat=0
!      t=0      
      w=sqrt( r*r-x*x)
      gamma1=r/w
      T=r/x
      T1=T+sqrt(T*T-1.0_DP)
      s=pi/4.0_DP-w+r*log(T1)
      factor=sqrt(2.0_DP*Pi)/(r*r-x*x)**(0.25_DP)
      R2=r*r
      t1=1.0_DP  !u_rx(0,X1)
      RR=-1.0_DP/(r*r)
      do k=2,k_r,2
         t1=t1+RR*u_rx( k,gamma1); 
         RR=-1.0_DP*RR/R2
      enddo
      RR=1.0_DP/r
      t2=RR*u_rx(1,gamma1)
      do k=3,k_r,2
         RR=-1.0_DP*RR/R2
         t2=t2+RR*u_rx(k,gamma1)         
      enddo
      kbes_r=factor*(dsin(s)*t1+dcos(s)*t2)
      end function

      ! And Debeyes series
      function kbes_x( r, x, k_r,stat) ! x > r >> 0 !
      use commonvariables
      implicit none
      real(DP),intent(in):: x, r
      integer::stat
      integer,intent(in):: k_r
      real(DP):: c,R1,R2, s, RR, t1,T2, w,kbes_x,T,X1
      real(DP)::gamma1,factor,beta
      integer:: k;
      stat=0
      T=r/x
      w=sqrt((x-r)*(x+r))
      gamma1=r/w
      s=r*dacos(r/x)-w
      factor=sqrt(0.5_DP*Pi/w)

      t1=1.0_DP  !u_rx(0,X1)
      R1=1.0_DP/r
      RR=1.0_DP
!      write(*,*) 'gamma1=',gamma1,u_rx( 1,-gamma1)
      do k=1,k_r
         RR=RR*R1
         t1=t1+RR*u_rx( k,-gamma1)
 !         write(*,*) 't1=',factor*exp(s)*t1
      enddo
      kbes_x=factor*exp(s)*t1
      end function


      !! Airy function
      function Ai(x)            !/* |x| << 10 */
      use commonvariables
      implicit none      
      real(DP)::x,Ai
      real(DP)::f, g,prec, k, t, xxx
      integer::j,NMAX
      NMAX=1000
      prec=it_prec
      xxx=x*x*x;
      k=0.0_DP 
      t=x
      g=t
      do j=1,NMAX 
         k=k+3.0_DP
         t=t*xxx/((k+1.0_DP)*k)
         g=g+t
         if(abs(t).lt.prec) then
            exit
         endif
      enddo
      k=0.0_DP
      t=1.0_DP
      f=t
      do j=1,NMAX 
         k=k+3.0_DP
         t=t*xxx/(k*(k-1.0_DP))
         f=f+t
         if(abs(t).lt.prec) then
            exit
         endif
      enddo
      Ai=0.355028053887817_DP*f-0.258819403792807_DP*g
      end function

      !! Asymptotics in transitional region
      function kbes_t( r, x,stat)    !/* r>0, x>0, |x-r| << pow( x, 1/3.) */
      use commonvariables
      implicit none
      real(DP)::r, x,kbes_t
      integer::stat
      real(DP)::t, tt, X1, XX, XXX, p, q
      stat=0
      t = (2.0_DP/x)**(1.0_DP/3.0_DP)
      tt=t*t
      X1=(x-r)*t
      XX=X1*X1
      XXX=XX*X1;
      p= 1.0_DP                                                           &
     &     + tt*(X1/15.0_DP                                                      &
     &     + tt*(13.0_DP*XX/1260.0_DP                                                &
     &     + tt*((109.0_DP*XXX/56700.0_DP+1.0_DP/900.0_DP)                                    &
     &     + tt*(203743.0_DP*XX*XX/523908000.0_DP+6761.0_DP*             &
     &     X1/7276500.0_DP))))                 
      q = X1                                                               &
     &     + tt*(XX/60.0_DP                                                        &
     &     + tt*((2.0_DP*XXX/1575.0_DP+1.0_DP/140.0_DP)                                          &
     &     + tt*((41.0_DP*XX*XX/283500.0_DP+4.0_DP*X1/1575.0_DP)                                   &
     &     + tt*(6553.0_DP*XXX*XX/327442500.0_DP+1409.0_DP*                    &
     &     XX/1819125.0_DP))))
      kbes_t= pi*t*p*Ai(q)
      end function
      
      
      function kbes_asympt(r,x,stat) 
      use commonvariables
      implicit none
      real(DP)::r,x,kbes_asympt
      real(DP)::lmt
      integer::k_r,k_x,stat
      ! for really karge r use asymptotics only
      !
      stat=1
      k_r=6
      k_x=8
      if(r.lt.3.0_DP) then 
         lmt=2.0_DP*5.0_DP**(0.666666667_DP)*r**(0.33333333333333333_DP)
         k_x=12
         if(x.lt.0.3_DP*r) then
            call bessk_pow(R,x,kbes_asympt,stat)

         elseif(x.lt.10.0_DP+lmt) then
            kbes_asympt=kbes_rec(R,x,stat)
         else
            kbes_asympt=kbes_x(R,x,k_x,stat)

         endif
      elseif(r.lt.6.0_DP) then
         k_x=6
         if(x.lt.0.3_DP*r) then
            call bessk_pow(R,x,kbes_asympt,stat)
!            write(*,*) 'pow=',kbes_asympt
         elseif(x.lt.10.0_DP+3.0_DP*r) then
            kbes_asympt=kbes_rec(R,x,stat)
!            write(*,*) 'kbes_rec=',kbes_asympt
         else
            kbes_asympt=kbes_x(R,x,k_x,stat)
!            write(*,*) 'kbes_x=',kbes_asympt
         endif
      elseif(r.lt.15.0_DP) then
         k_x=14
         if(x.lt.0.3_DP*r) then
            call bessk_pow(R,x,kbes_asympt,stat)
         elseif(x.le.max(10.0_DP+1.2_DP*r,2.0_DP*r)) then
            kbes_asympt=kbes_rec(R,x,stat)
         else
            kbes_asympt=kbes_x(R,x,k_x,stat)
         endif
      elseif(r.lt.50.0_DP) then
         k_x=10
         if(x.lt.0.3333333_DP*r) then
            kbes_asympt=kbes_r(r,x,k_r,stat) ! use iteration in the transitional region
         elseif(x.lt.1.75_DP*r) then
            kbes_asympt=kbes_rec(R,x,stat)
         else
            kbes_asympt=kbes_x(r,x,k_x,stat)
         endif
      elseif(r.lt.150.0_DP) then                   ! We shouldn't be using asymptotics
         if(x.lt.0.3333333_DP*r) then
            kbes_asympt=kbes_r(r,x,k_r,stat) ! use iteration in the transitional region
         elseif(x.lt.1.4_DP*r) then
            kbes_asympt=kbes_rec(R,x,stat)
         else
            kbes_asympt=kbes_x(r,x,k_x,stat)
         endif
      elseif(r.lt.1500.0_DP) then
!         lmt=5.0_DP**(0.666666667_DP)*r**(0.33333333333333333_DP)
         lmt=10.0_DP**(0.666666667_DP)*r**(0.33333333333333333_DP)
         if(x.lt.r-lmt) then
            kbes_asympt=kbes_r(r,x,k_r,stat)
         elseif(x.gt.r+lmt) then
            k_x=8
            kbes_asympt=kbes_x(r,x,k_x,stat)
         else
            kbes_asympt=kbes_t(r,x,stat)
         endif
      else
         lmt=5.0_DP**(0.666666667_DP)*x**(0.33333333333333333_DP)
         if(x.lt.r-2.0_DP*lmt) then            
            kbes_asympt=kbes_r(r,x,k_r,stat)
         elseif(x.gt.r+2.0_DP*lmt) then
            kbes_asympt=kbes_x(r,x,k_x,stat)
         else
            kbes_asympt=kbes_t(r,x,stat)
         endif
      endif
      end function


      subroutine clean_kbessel()
      use whittakerw
      implicit none
      if(allocated(A_uk)) deallocate(A_uk)
      call clean_whittaker
      end subroutine

      end module
      
