!!!! This file contains the following modules:
!
!   Modules used by whittakerW:
!     module bernoulli
!       - Computes Bernoulli numbers via recursion
!     module gamma_function
!       - Computes Gamma function using Stirlings formula
!     module gauss_quad
!       - Initiating parameters used for Gaussian quadrature
!
!   module whittakerW
!       - Computes the Whittaker-W function using stationary phase and gaussian quadrature 
!       provides:
!   WhittW(l,r,x) returns W_l,iR(x) for real l,R,x
!



      !! Compute Bernoulli numbers to be used in the Gamma function
      module bernoulli
      use commonvariables
      implicit none
      integer,save::num_B_r,num_B_mp
      real(DP),dimension(:),allocatable,save::BERN
      logical,save::ber_inited_r      
      contains
      !! Compute Bernouilli numbers in real precision
      subroutine init_ber(N)
      real(DP)::S,R
      integer,optional,intent(in)::N
      integer::m,k,j,nb
      nb=50
      if(PRESENT(N)) then 
         if(N.gt.0) then 
            nb=Nb
         endif
      endif
      if(ber_inited_r.and.(nb.le.num_B_r)) then 
         return
      endif
      num_B_r=nb
!      write(*,*) 'Initiating Bernoulli! num=',num_B_r
      allocate(BERN(1:num_B_r))
      !! Bernoulli Numbers: A(k)=B(2k)/2k
      BERN(1)=1.0D0/12.0D0
      BERN(2)=-1.0D0/120.0D0
      do M=2,Num_B_r
         S=real(m-1,kind=DP)/real(2*M+2,kind=DP)
         DO K=2,M-1
            R=1.0D0
            DO J=2,K
               R=R*real(J+M-K,kind=DP)/real(J,kind=DP)
            enddo
            S=S-R*BERN(K)
         enddo
         BERN(M)=S
!        write(*,'(A)I3(A)',ADVANCE='NO') 'BER(',M,')='
!        call mpwrite(6,S)
      ENDDO
      DO M=3,num_B_r,2
         BERN(M)=0.0D0
      ENDDO
      ber_inited_r=.true.
      end subroutine

      subroutine clean_bernoulli()
      if(allocated(BERN)) deallocate(BERN)
      ber_inited_r=.false.
      end subroutine
      end module

      module gamma_function
      use bernoulli
      use commonvariables
      !! Evaluates Gamma(z) for real or complex z
      interface gamma
      module procedure gamma_c
      module procedure gamma_r
      end interface
      !! Evaluates log(Gamma(z)) for real or complex z
      interface lngamma
      module procedure lngamma_c
      module procedure lngamma_r
      end interface

!      private::DP,PI,ln2PI,ln2PI_2,err,B,pihalf
!      real(DP),parameter::PI=3.14159265358979323846_DP
!      real(DP),parameter::ln2PI=1.837877066409345483560659_DP
!      real(DP),parameter::ln2PI_2=0.9189385332046727417803297_DP
!      real(DP),parameter::Pihalf=1.57079632679489661923132169164_DP
      private err,B
      real(DP),parameter::err=1.0E-16_DP
      integer,parameter::NS=20 !! max number of terms in stirling
      real(DP),save::B(1:24)
      logical::param_inited
      contains

      function gamma_c(z)
      implicit none
      complex(DP),intent(in):: Z
      complex(DP):: gamma_c
      call init_param()
      gamma_c=exp(lngamma_c(z))
      end function
      !! For real positive X
      function gamma_r(X)
      implicit none 
      real(DP),intent(in)::X
      real(DP)::gamma_r
      call init_param()
      if(X.le.0.0_DP) then 
         gamma_r=real(gamma_c(cmplx(X,0.0_DP,kind=DP)),kind=DP)
         return
      endif
      gamma_r=exp(lngamma_r(X))
      end function 

      !! LnGamma(z) for complex z
      function lngamma_c(z)
      use commonvariables
      implicit none
      complex(DP),intent(in):: Z
      real(DP):: a,lnw,u,v,argw,Ims,Res
      complex(DP):: TMP, lngamma_c
      complex(DP):: W ,WW, SUM , STIRLING 
      real(DP),parameter::ln2PI_d2=0.9189385332046727417803297_DP
      INTEGER:: i,j ,N 
      !! Can be better optimized
      call init_param()
      if(real(z,kind=DP).gt.0.0_DP) then
!         N=max(0,anint(10.0_DP-real(z,kind=DP)))
         N=max(anint(0.0D0),anint(10.0_DP-real(z,kind=DP)))
      else
         N=anint(10.0_DP-real(z,kind=DP))
      endif     
      W=Z+cmplx(real(N,kind=DP),0.0_DP,kind=DP)
      SUM=cmplx(0.0,0.0,kind=DP)
      WW=dcmplx(1.0_DP,0.0_DP)/W
      DO I=1,NS
         TMP=dcmplx(B(I),0.0_DP)*WW
         sum=sum+TMP
         if(abs(TMP).lt.err) then 
            exit
         endif
         WW=WW/(W*W)
      enddo
!      STIRLING=(W-0.5_DP)*log(W)-W+ln2PI_2      
      u=real(w,kind=DP)
      v=aimag(w)
      a=sqrt(u*u+v*v)
      lnw=log(a)
      argw=argument(u,v)
      Res=(u-0.5_DP)*lnw-v*argw-u+ln2PI_d2
      Ims=v*(lnw-1.0_DP)+(u-0.5_DP)*argw
      !! Question: Should we truncate the imaginary part to use the principal branch
      !!           Perhaps this is only waste of time here... 
      STIRLING=cmplx(Res,Ims,kind=DP)
      ! this was really for GAMMA(z+N)
      lngamma_c=STIRLING+SUM
       do j=0,N-1
          lngamma_c=lngamma_c-log(z+cmplx(real(j,kind=DP),0.0_DP,         &
     &         kind=DP))
      enddo
      END function
!!!  ! for real positive argument
      function lngamma_r(z)
      implicit none
      real(DP),intent(in)::Z 
      real(DP):: lngamma_r,tmp_r
      real(DP):: W ,WW, SUM , STIRLING
      INTEGER:: i,j ,N 
      real(DP),parameter::ln2PI_d2=0.9189385332046727417803297_DP
      !! Can be better optimized
      call init_param()
      if(z.gt.0.0_DP) then
         N=max(anint(0.0D0),anint(10.0_DP-z))
      else
         N=anint(10.0_DP-z)
      endif
      W=Z+real(N,kind=DP)
      WW=1.0_DP/W
      SUM=0.0_DP
      DO I=1,NS
         TMP_r=B(I)*WW
         sum=sum+TMP_r
         if(abs(TMP_r).le.err) then 
            exit
         endif
         WW=WW/W**2
      enddo

      STIRLING=sum+(W-0.5_DP)*log(W)-W+ln2PI_d2
      ! this was really computed for GAMMA(z+N) and we have to correct taht now
      lngamma_r=STIRLING
      do j=0,N-1                !,0,-1
         lngamma_r=lngamma_r-dlog(z+real(j,kind=DP))
      enddo
      end function 
      
      !! Init Bernoulli numbers and other parameters for the Gamma function
      subroutine init_param()
      use bernoulli
      integer::j,Nber
      Nber=48
      !! Test if inited already or not
      if(param_inited.and.(B(1).ne.0.0_DP)) then 
         return
      endif
      !! We only do this if we need more Bernoullli numbers than original
      call init_ber(Nber)
      !! Remember for Stirling we need B(2k)/2k/(2k+1)
      !!
      do j=1,Nber/2
!         write(*,*) 'BER(',j,')=',BERN(2*j)
         B(j)=BERN(2*j)/real(2*j-1,kind=DP)/real(2*j,kind=DP)         
      enddo
      param_inited=.true.
      end subroutine

      function factorial(n)
      implicit none
      integer j,n,factorial
      if(n.le.1) then
         factorial=1
      elseif(n.lt.30) then
         factorial=N
         do j=N-1,2,-1 
            factorial=factorial*j
         enddo 
                               !      write(*,*) 'factorial(',n,')=',factorial
      else
         write(*,*) 'Dont use this integer function with large args!'
         write(*,*) 'n=',n
         stop
      endif
      end function




!c$$$      !! some helper functions
!c$$$      function is_int(a)
!c$$$      real(DP),intent(in)::a
!c$$$      logical::is_int
!c$$$      is_int=.false.
!c$$$      if(real(real(ceiling(a),kind=DP)-a,kind=DP).eq.0.0_DP)              &
!c$$$     &     is_int=.true.
!c$$$      end function
!c$$$
!c$$$!     ! principal argument of z=x+iy, -Pi<arg(z)<=Pi
!c$$$      function argument(x,y)
!c$$$      implicit none
!c$$$      real(DP),intent(in)::x,y
!c$$$      real(DP)::argument
!c$$$      if(x.gt.0.0_DP) then
!c$$$         argument=datan(y/x)
!c$$$      elseif(x.lt.0.0_DP) then
!c$$$         if(y.lt.0.0_DP) then
!c$$$            argument=datan(y/x)-PI
!c$$$         else
!c$$$            argument=(Pi+datan(y/x))
!c$$$         endif
!c$$$      else !if x=0
!c$$$         if(y.ge.0.0_DP) then
!c$$$            argument=Pihalf
!c$$$         else
!c$$$            argument=-1.0_DP*Pihalf
!c$$$         endif
!c$$$      endif
!c$$$!      argument_r=argument
!c$$$      end function
!c$$$
!c$$$      function argument_c(z)
!c$$$      implicit none
!c$$$      complex(DP),intent(in)::z
!c$$$      real(DP)::argument_c
!c$$$      argument_c=argument(real(z,kind=DP),aimag(z))
!c$$$      end function
!c$$$      


      end module

      





      module gauss_quad
      integer,private,parameter::DP=kind(1.0D0)
      real(DP),dimension(16) :: AD, WD
      real(DP),dimension(20) ::RG,RF
      DATA AD/0.01059906500835006740384582654967_DP,                            &
     &     0.05542497692676742392201158446539_DP,                                       &
     &     0.13436879761216825611953210228761_DP,                                       &
     &     0.24459559164499696610489880515256_DP,                                       &
     &     0.38212375559735625155332823595121_DP,                                       &
     &     0.54198322234277261365758055701642_DP,                                       &
     &     0.71839644922074108676953949853950_DP,                                       &
     &     0.90498749016236255981468066457504_DP,                                       & !  AD(8)
     &     1.09501250983763744018531933542496_DP,                                       &
     &     1.28160355077925891323046050146050_DP,                                       &
     &     1.45801677765722738634241944298358_DP,                                       &
     &     1.61787624440264374844667176404879_DP,                                       &
     &     1.75540440835500303389510119484744_DP,                                       &
     &     1.86563120238783174388046789771239_DP,                                       &
     &     1.94457502307323257607798841553461_DP,                                       &
     &     1.98940093499164993259615417345033_DP/
      DATA WD/0.02715245941175409485178057245602_DP ,                                    &
     &     0.06225352393864789286284383699438_DP ,                                       &
     &     0.09515851168249278480992510760225_DP ,                                       &
     &     0.12462897125553387205247628219202_DP ,                                       &
     &     0.14959598881657673208150173054748_DP ,                                       &
     &     0.16915651939500253818931207903036_DP ,                                       &
     &     0.18260341504492358886676366796922_DP ,                                       &
     &     0.18945061045506849628539672320828_DP ,                                       & !WD(8)
     &     0.18945061045506849628539672320828_DP ,                                       &
     &     0.18260341504492358886676366796922_DP ,                                       &
     &     0.16915651939500253818931207903036_DP ,                                       &
     &     0.14959598881657673208150173054748_DP ,                                       &
     &     0.12462897125553387205247628219202_DP ,                                       &
     &     0.09515851168249278480992510760225_DP ,                                       &
     &     0.06225352393864789286284383699438_DP,                                    &
     &     0.02715245941175409485178057245602_DP/                                    

      DATA RG/1.0_DP,                                                            &
     &     0.5_DP,                                                               &
     &     0.3333333333333333333333333333333_DP,                                 &
     &     0.25_DP,                                                              &
     &     0.2_DP,                                                       &
     &     0.16666666666666666666666666667_DP,                                   &
     &     0.1428571428571428571428571428571_DP,                                 &
     &     0.125_DP,                                                             &
     &     0.1111111111111111111111111111111_DP,                                 &
     &     0.1_DP,                                                               &
     &     0.9090909090909090909090909090909E-1_DP,                              &
     &     0.8333333333333333333333333333333E-1_DP,                                 &
     &     0.7692307692307692307692307692308E-1_DP,                              &
     &     0.7142857142857142857142857142858E-1_DP,                              &
     &     0.6666666666666666666666666666667E-1_DP,                              &
     &     0.6250000000000000000000000000000E-1_DP,                                 &
     &     0.5882352941176470588235294117647E-1_DP,                              &
     &     0.5555555555555555555555555555556E-1_DP,                              &
     &     0.5263157894736842105263157894737E-1_DP,                              &
     &     0.50E-1_DP/                    ! RG(20)
      DATA RF/1.00000000000000000000000000000000e+00_DP,                         &          
     &     5.00000000000000000000000000000000e-01_DP,                         &           
     &     1.66666666666666666666666666666667e-01_DP,                         &            
     &     4.16666666666666666666666666666667e-02_DP,                         &            
     &     8.33333333333333333333333333333333e-03_DP,                         &            
     &     1.38888888888888888888888888888889e-03_DP ,                         &           
     &     1.98412698412698412698412698412698e-04_DP ,                         &           
     &     2.48015873015873015873015873015873e-05_DP ,                         &           
     &     2.75573192239858906525573192239859e-06_DP ,                         &           
     &     2.75573192239858906525573192239859e-07_DP ,                         &           
     &     2.50521083854417187750521083854417e-08_DP ,                         &           
     &     2.08767569878680989792100903212014e-09_DP ,                         &           
     &     1.60590438368216145993923771701549e-10_DP ,                         &           
     &     1.14707455977297247138516979786821e-11_DP ,                         &           
     &     7.64716373181981647590113198578807e-13_DP ,                         &           
     &     4.77947733238738529743820749111754e-14_DP ,                         &           
     &     2.81145725434552076319894558301032e-15_DP ,                         &           
     &     1.56192069685862264622163643500573e-16_DP ,                         &           
     &     8.22063524662432971695598123687228e-18_DP ,                         &           
     &     4.11031762331216485847799061843614e-19_DP/      




      end module

!!!!  Computes the Whittaker W-function W_{k,iR}(X)
!!!!   where R,k are real and X>0.
!!!!   








      module whittakerW
      use commonvariables
      use gamma_function
      use gauss_quad
      implicit none
!      private :: eps,N_max_pow,N_max_as,N_max_recursive
!      PRIVATE
!      integer,private,parameter::DP=kind(1.0D0)
!      real(DP),parameter::Pi=3.14159265358979323846264338328_DP
!      real(DP),parameter::sqrtPi=1.77245385090551602729816748334_DP
!      complex(DP),parameter::cone=cmplx(1.0_DP,0.0_DP,kind=DP)
!      complex(DP),parameter::ctwo=cmplx(2.0_DP,0.0_DP,kind=DP)
!      complex(DP),parameter::chalf=cmplx(0.5_DP,0.0_DP,kind=DP)
!      complex(DP),parameter::czero=cmplx(1.0_DP,0.0_DP,kind=DP)
      real(DP):: eps=1.0E-13_DP
      integer,parameter::N_max_pow=500
      integer,parameter::N_max_as=1500
      integer,parameter::N_max_recursive=2500
    ! Used for Holgers asympt algs
      real(DP),save::c_rx
      integer,save::k_rx
       real(DP),dimension(:,:),allocatable,save::A_uk
      integer,parameter::N_uk=20  ! The max number of terms in u_k
      real(DP),dimension(:,:),allocatable,save::a_k,b_k
      complex(DP),dimension(:),allocatable,save::lambda
      real(DP)::a_set_l,c_set_l
      logical,save::lambda_set
      logical,save::coeff_psi_pow3_is_set
      real(DP),save::a_set
      logical::complex_bes
      real(DP)::rs
      logical::a_is_neg_int,a_is_neghalfint
      !!! Positions and weights for Gaussian quadrature
!      public WhittW,k_bessel,error handling,k_b
      contains 
    


      !! The Whittaker W-function
      !! W(l,iR,x)
      function WhittW(l,R,x)
      implicit none      
      real(DP),intent(in)::R,l,x
      real(DP)::WhittW,W1
      W1=WhittW_norm(l,R,x)
!      write(*,*) 'l,R,x=',l,R,x
!      write(*,*) 'W1=',W1
      WhittW=W1*exp(-R*Pihalf)
      end function
      
      !! The Whittaker normalized W-finction
      !! W~(l,iR,x)=W(l,iR,x)*exp(-R*PI/2)
      function WhittW_norm(l,R,x)
!      use commonvariables
      implicit none      
!      real(DP),dimension(16) :: AD, WD
      real(DP),intent(in)::R,l,x
      real(DP)::W,S,WhittW_norm,T,RR
#ifdef debug
      integer::ierr
#endif
      !! If l is a pos. integer we set a_is_neg_int=.true.
      if(is_int(l).and.l.ge.0.0_DP) then 
         a_is_neg_int=.true.
      else
         a_is_neg_int=.false.
      endif
      if(is_int(-l+0.5_DP).and.l.ge.0.5_DP) then 
         a_is_neghalfint=.true.
      else
         a_is_neghalfint=.false.
      endif

      if(R.lt.0.0_DP) then  ! we have a symmetry in R
         RR=-R
      else
         RR=R
      endif
      T=2.0_DP*RR/X

#ifdef debug
         open(unit=14,FILE='psi_arg.txt',POSITION='APPEND',IOSTAT=ierr)
         open(unit=15,FILE='psi_c.txt',POSITION='APPEND',IOSTAT=ierr)
         if(ierr.gt.0) then
            write(*,*) 'ERROR opening files: ',ierr
         endif
#endif
      S=min(1.0_DP/3.0_DP,(5.0_DP/RR)**(2.0_DP/3.0_DP))
#ifdef debug
         write(*,*) 'T=',T
         write(*,*) 'S=',S
#endif
      if(abs(RR-0.5_DP*x).lt.(RR*S)) then
#ifdef debug
            write(*,*) 'case3'
#endif
         call WhittW_gq3(RR,l,x,W)
      elseif(T.le.1.0_DP) then
#ifdef debug
            write(*,*) 'case2'
#endif
         call WhittW_gq2(RR,l,x,W)
      else
#ifdef debug
         write(*,*) 'case1'
#endif
         call WhittW_gq1(RR,l,x,W)
      endif
#ifdef debug
      close(14)
      close(15)
#endif
      ! Returns the normalized whittaker-function
      !
      WhittW_norm=W*sqrt(x/PI)
      end function 


      !!! Using Gaussian quadrature
    !! Only for the case T>1, R>x!!!

      ! Whittaker W-functions
      ! W(l,iR)*sqrt(Pi/2x)*exp(Pi/2*R)
      subroutine WhittW_gq1(R,l,xx,W_res)
!      use gamma_function
      implicit none      
      real(DP),dimension(16) :: chunks,chunks_h
      real(DP),intent(in)::R,l,xx
      real(DP),intent(out)::W_res
      real(DP)::x,u,u0,u1,a,b,h,cut_off,cut_off1,cut_off2
      real(DP):: sum,delta
      real(DP)::Integrand
      complex(DP)::Psi_c,arg_psi
      integer:: i,j,num_steps
      real(DP)::S,T,start,cosv,coshu,sinhu,temp_factor
      real(DP)::trig_arg,dv_du,v,sinv
      real(DP)::abs_psi,psi_x,psi_y
      real(DP)::sum1,sum_h,test_arg,eta
      real(DP)::c,s1,c1,vt,tt,h2,c23,cc23
      real(DP)::x_coshu_cosv
      integer::num_steps1,num_steps2,num_steps3
      real(DP)::z,AA
      real(DP)::sum_r,sum_l,exp_factor
      integer,parameter::num_pow=10
      real(DP)::RDD,RDZ,SM,SN1,SN2,VX,CXX,C3T
      real(DP)::AQ0,AQ1,AQ2,AQ3,AQ4,BQ0,BQ1,BQ2,BQ3,BQ4,z2AQ4
#ifdef debug
      integer::ierr
         open(unit=12,FILE='whitt_int1',IOSTAT=ierr)
         open(unit=13,FILE='whitt_path1',IOSTAT=ierr)
!         open(unit=14,FILE='whitt_arg')
!         open(unit=15,FILE='hor_ints.txt')
#endif
      IF (R.LT.126.D0) THEN
         ETA=0.D0
      ELSEIF (R .LT. 169.D0) THEN
         ETA=.01D0
      ELSEIF (R .LT. 196.D0) THEN
         ETA=.07D0
      ELSEIF (R .LT. 328.D0) THEN
         ETA=.1D0
      ELSEIF (R .LT. 570.D0) THEN
         ETA=.2D0
      ELSEIF (R .LT. 1044.D0) THEN
         ETA=.3D0
      ELSEIF (R .LT. 2075.D0) THEN
         ETA=.4D0
      ELSEIF (R .LT. 4640.D0) THEN
         ETA=.5D0
      ELSEIF (R .LT. 12620.D0) THEN
         ETA=.6D0
      ELSEIF (R .LT. 23360.D0) THEN
         ETA=.7D0
      ELSEIF (R .LT. 49200.D0) THEN
         ETA=.75D0
      ELSE
         ETA=0.8D0
      ENDIF

      X=0.5_DP*xx  ! to skip all 0.5_DP*x everywhere
      T=R/X
      S=T*log(T+sqrt(T**2-1.0_DP))-sqrt(T**2-1.0_DP)      
      u1=log(T+sqrt(T**2-1.0_DP))
      u0=S/T
#ifdef debug
      write(*,*) 'u0=',u0,' u1=',u1
      write(*,*) 'T=',T
#endif
      delta=dsqrt((R-x)*(R+x))/R  !! = u1-u0
      RDD=1.0_DP/delta
      C23=DSQRT(110.00_DP/DSQRT((R-X)*(R+X)))  

      !if(R.gt.20) then
      !   num_steps1=6
      !else
      !   num_steps1=4
      !endif
      !!! The number of steps to use in the different intervals 
      !!! Can certainly be further optimized for special parameter intervals

!      num_steps2=16
!      num_steps3=16
      if((l.lt.10.0_DP).and.(R.lt.100.0_DP)) then 
         num_steps1=4
         num_steps2=10
         num_steps3=4
      elseif((R.lt.100.0_DP).and.(l.lt.100.0_DP)) then 
         num_steps1=4
         num_steps2=12
         num_steps3=4
      elseif(R.lt.500.0_DP) then 
         num_steps1=4
         num_steps2=24
         num_steps3=12
      elseif(R.lt.10000.0_DP) then
         num_steps1=4
         num_steps2=32
         num_steps3=12
      elseif(R.lt.100000.0_DP) then 
         num_steps1=12
         num_steps2=96
         num_steps3=4         
      elseif(R.lt.500000.0_DP) then 
         num_steps1=12
         num_steps2=128
         num_steps3=4         
      else
         num_steps1=12
         num_steps2=256
         num_steps3=4  
      endif
      if(R.gt.20.0_DP) then
         cut_off1=u1+min(C23,2.0_DP) !dsqrt(110.0_DP/(delta*R)),2.0_DP)
         h=min(dsqrt(110.0_DP/dsqrt((R-x)*(R+x))),2.0_DP)/                 &
     &        REAL(num_steps1,kind=DP) 
      else
         CC23=100.0_DP
         DO I=2,20
            U=U1+REAL(I,kind=DP)          
            C=T*U-S
            S1=C/sinh(u)
            C1=DSQRT((sinh(u)-C)*(sinh(u)+C))/sinh(u)
            VT=(2.0_DP)*ATAN(S1/(1.0_DP+C1))
            TT=-1.0_DP*X*cosh(u)*C1 + R*(PIhalf-VT)
#ifdef debug
               write(*,*) 'TT=',TT
#endif
            IF(TT.LT.(-55.0_DP)) THEN
               CC23=REAL(I,kind=DP)
               exit
            ENDIF
         END DO
         cut_off1=u1+MIN(C23,CC23)
         h=MIN(C23,CC23)/REAL(num_steps1,kind=DP) 
#ifdef debug
            write(*,*) 'h=MIN(',C23,',',CC23,')/',num_steps1 
#endif
      endif
      cut_off2=delta-eta*(delta**3.0_DP)
      h2=cut_off2/real(num_steps2,kind=DP)
      cut_off=cut_off1  !u1+2.5_DP
      num_steps=num_steps1
      start=u1+delta
      h=(cut_off-start)/real(num_steps,kind=DP)
#ifdef debug
        write(*,*) 'num_steps=',num_steps,' h1=',h,'u0=',u0
         write(*,*) 'start=',start,' cut_off=',cut_off,'u1=',u1
         write(*,*) 'cutoff_h=',real(num_steps,kind=DP)*h2
#endif
!c     !! Start with the integral over [u1+delta,cutoff] 
      sum_r=0.0_DP
      do i=1,num_steps
         aa=real(i-1,kind=DP)*h
         if((aa+start-u1).ge.0.5_DP) then
            do j=1,16
               u=start+aa+0.5_DP*h*AD(j)
               coshu=cosh(u)
               sinhu=sinh(u)
               chunks(j)=0.0_DP
               chunks_h(j)=0.0_DP                              
               dv_du=(T*sinhu-(T*u-S)*coshu)
               temp_factor=sqrt((sinhu-(T*u-S))*(sinhu+(T*u-S)))
               dv_du=dv_du/(sinhu*temp_factor)
               sinv=(T*u-S)/sinhu
               v=dasin(sinv)
               cosv=temp_factor/sinhu
               x_coshu_cosv=x*coshu*cosv
               test_arg=R*(Pihalf-v)-x_coshu_cosv               
               if(l.ne.0.0_DP) then
                  arg_psi= cmplx(x+x_coshu_cosv,x*(T*u-S),kind=DP)            
                  if(l.eq.0.5_DP)then
                     psi_c=sqrt(arg_psi)
                  else
                     psi_c=psi_integrand(-l,arg_psi)
#ifdef debug
                     write(*,*) 'Psi(-l,1/2',arg_psi,')=',psi_c

#endif
                  endif
                  psi_x=real(psi_c,kind=DP)
                  psi_y=aimag(psi_c)               
                  abs_psi=abs(psi_c)
                  trig_arg=argument(psi_x,psi_y)
                  trig_arg=trig_arg+x*S              
               else
                  abs_psi=1.0_DP
                  trig_arg=x*S               
               endif
                             !  This quantity will be tested after the inner loop
                             ! to see if we want to break out from the outer loop
               Integrand=(dcos(trig_arg)-dsin(trig_arg)*dv_du)
               Integrand=Integrand*exp(test_arg)*abs_psi
#ifdef debug
                  if(ierr.eq.0) then
                     write(14,2770) arg_psi
                     write(15,2770) psi_c
                  endif
#endif
               chunks(j)=WD(j)*Integrand                        
            enddo
         else  !! using power series expansion to avoid cancellation: Idea by Dennis Hejhal
            do j=1,16
!               z=aa+0.5_DP*h*AD(j)
               z=(start-u1)+aa+0.5_DP*h*AD(j)
               u=u1+z
               AQ0=(Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z)*(RF(16)*               &
     &              (delta+RG(17)*Z)+RF(18)*(delta+RG(19)*Z)*Z*Z)
               AQ1=(Z*Z*Z*Z*Z*Z*Z*Z*Z*Z)*(RF(12)*(delta+RG(13)*Z)+      &
     &              RF(14)*(delta+RG(15)*Z)*Z*Z)+AQ0
               AQ2=(Z*Z*Z*Z*Z*Z)*(RF(8)*(delta+RG(9)*Z)+                &
     &              RF(10)*(delta+RG(11)*Z)*Z*Z)+AQ1
               AQ3=(Z*Z)*(RF(4)*(delta+RG(5)*Z)+                        &
     &              RF(6)*(delta+RG(7)*Z)*Z*Z)+AQ2
               AQ4=RF(2)*(delta+RG(3)*Z)+AQ3
!               A9=AQ4*(RDD)
               BQ0=(Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z)*(RF(15)*                &
     &              (delta+RG(16)*Z)+RF(17)*(delta+RG(18)*Z)*Z*Z)
               BQ1=(Z*Z*Z*Z*Z*Z*Z*Z*Z*Z)*(RF(11)*(delta+RG(12)*Z)+                &
     &              RF(13)*(delta+RG(14)*Z)*Z*Z)+BQ0
               BQ2=(Z*Z*Z*Z*Z*Z)*(RF(7)*(delta+RG(8)*Z)+                &
     &              RF(9)*(delta+RG(10)*Z)*Z*Z)+BQ1
               BQ3=(Z*Z)*(RF(3)*(delta+RG(4)*Z)+                         &
     &              RF(5)*(delta+RG(6)*Z)*Z*Z)+BQ2
               BQ4=(delta+RG(2)*Z)+BQ3
!               B9=BQ4*(RDD)
               z2AQ4=z*z*AQ4
               rdz=rdd*z
               C3T=1.0_DP+z*BQ4
               SM=1.0_DP+rdz
               SN1=z2AQ4*RDD+1.0_DP+rdz
               SN2=(2.0_DP*SM)+z2AQ4*RDD
               CXX=dsqrt(AQ4*RDD)*dsqrt(SN2)
               sinv=SM/SN1
               cosv=z*CXX/SN1
               VX=-1.0_DP*BQ4*RDD-(BQ4-AQ4)*z*RDD*RDD
               dv_du=VX/(CXX*SN1)               
               v=2.0_DP*datan(SM/SN1/(1.0_DP+cosv))
               ! meeting u=u1+z
               x_coshu_cosv=x*cosh(u)*cosv
               if(l.ne.0.0_DP) then
                  arg_psi= cmplx(x+x_coshu_cosv,x*(T*u-S),kind=DP)            
                  if(l.eq.0.5_DP)then
                     psi_c=sqrt(arg_psi)
                  else
                     psi_c=psi_integrand(-l,arg_psi)
                  endif
                  psi_x=real(psi_c,kind=DP)
                  psi_y=aimag(psi_c)               
                  abs_psi=abs(psi_c) 
                  trig_arg=argument(psi_x,psi_y)
                  trig_arg=trig_arg+x*S
               else
                  abs_psi=1.0_DP
                  trig_arg=x*S                
               endif
               Integrand=(dcos(trig_arg)-dsin(trig_arg)*                   &
     &              (VX/(CXX*SN1)))*abs_psi*                               &
     &              dexp(R*(-c3t*z*CXX/SN1-v+Pihalf))
               chunks(j)=Integrand*WD(j)
#ifdef debug
                  write(12,2775) u,Integrand
                  write(13,2775) u,v
                  if(ierr.eq.0) then
                     write(14,2770) arg_psi
                     write(15,2770) psi_c
                  endif
#endif
            enddo
            ! quantity to check if we break out of outer loop
            test_arg=R*(Pihalf-v-C3T*cosv)
         endif
         sum=0.0_DP
         do j=16,1,-1
            sum=sum+chunks(j)
         enddo
         sum_r=sum_r+sum*0.5_DP*h
#ifdef debug
         write(*,*) 'sum_r=',sum_r
         write(*,*) 'test_arg=',test_arg
#endif
         if(test_arg.lt.-55.0_DP) then
         endif
         a=a+h
         b=b+h
      enddo
      ! Next we go backwards from u1+delta to u1-delta against u0
      !! u1+delta
      num_steps=num_steps2
      start=u1+delta
      cut_off=u1-cut_off2
!      h=abs(cut_off-start)/real(num_steps,kind=DP)
      h2=abs(cut_off-start)/real(num_steps,kind=DP)
#ifdef debug
         write(*,*) 'num_steps=',num_steps,' h1=',h,'h2=',h2
         write(*,*) 'start=',start
         write(*,*) 'stop=',cut_off
#endif
      sum_l=0.0_DP
      start=u1+delta
      do i=1,num_steps
         aa=real(1-i,kind=DP)*h2
!         if((aa+start-u1).le.(-0.25_DP)) then
         if(abs(aa+start-u1).ge.(0.2_DP)) then
            do j=1,16
               u=start+aa-0.5_DP*h2*AD(j)
               coshu=cosh(u)
               sinhu=sinh(u)               
               chunks(j)=0.0_DP
               sinv=(T*u-S)/sinhu
               dv_du=(T*sinhu-(T*u-S)*coshu)
               temp_factor=sqrt((sinhu-(T*u-S))*(sinhu+(T*u-S)))
               dv_du=dv_du/(sinhu*temp_factor)
            ! u is < u1  here if(u.lt.u1) then
               if(u.le.u1) then
                  v=PI-dasin(sinv)
                  dv_du=-1.0_DP*dv_du
                  cosv=-1.0_DP*temp_factor/sinhu
               else
                  v=dasin(sinv)
                  cosv=temp_factor/sinhu
               endif
               x_coshu_cosv=x*coshu*cosv
               test_arg=R*(Pihalf-v)-x_coshu_cosv
               exp_factor=exp(test_arg)
               if(l.ne.0.0_DP) then
                  arg_psi= cmplx(x+x_coshu_cosv,x*(T*u-S),kind=DP)            
                  if(l.eq.0.5_DP) then
                     psi_c=sqrt(arg_psi)
                  else
                     psi_c=psi_integrand(-l,arg_psi)
                  endif
                  abs_psi=abs(psi_c) 
                  psi_x=real(psi_c,kind=DP)
                  psi_y=aimag(psi_c)
                  trig_arg=argument(psi_x,psi_y)
                  trig_arg=trig_arg+x*S
               else
                  abs_psi=1.0_DP
                  trig_arg=x*S
               endif               
               Integrand=(dcos(trig_arg)-dsin(trig_arg)*dv_du)*abs_psi
               Integrand=Integrand*exp_factor
               chunks(j)=Integrand*WD(j)
            enddo            
         else
               ! Do power series to avoid cancellation
            do j=1,16
               u=start+aa-0.5_DP*h2*AD(j)
               z=u-u1
               AQ0=(Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z)*(RF(16)*                &
     &              (delta+z/17.0_DP)+RF(18)*(delta+z/19.0_DP)*Z*Z)
               AQ1=(Z*Z*Z*Z*Z*Z*Z*Z*Z*Z)*(RF(12)*(delta+z/13.0_DP)+                &
     &              RF(14)*(delta+Z/15.0_DP)*Z*Z)+AQ0
               AQ2=(Z*Z*Z*Z*Z*Z)*(RF(8)*(delta+Z/9.0_DP)+                &
     &              RF(10)*(delta+Z/11.0_DP)*Z*Z)+AQ1
               AQ3=(Z*Z)*(RF(4)*(delta+Z/5.0_DP)+                       &
     &              RF(6)*(delta+Z/7.0_DP)*Z*Z)+AQ2
               AQ4=0.5_DP*(delta+Z/3.0_DP)+AQ3
!               A9=AQ4*(RDD)
               BQ0=(Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z*Z)*(RF(15)*                &
     &              (delta+RG(16)*Z)+RF(17)*(delta+RG(18)*Z)*Z*Z)
               BQ1=(Z*Z*Z*Z*Z*Z*Z*Z*Z*Z)*(RF(11)*(delta+RG(12)*Z)+                &
     &              RF(13)*(delta+RG(14)*Z)*Z*Z)+BQ0
               BQ2=(Z*Z*Z*Z*Z*Z)*(RF(7)*(delta+RG(8)*Z)+                &
     &              RF(9)*(delta+RG(10)*Z)*Z*Z)+BQ1
               BQ3=(Z*Z)*((delta+RG(4)*Z)/6.0_DP+                       &
     &              (delta+RG(6)*Z)*Z*Z/120.0_DP)+BQ2
               BQ4=(delta+0.5_DP*Z)+BQ3
!               B9=BQ4*(RDD)
                                            
               z2AQ4=z*z*AQ4
               C3T=1.0_DP+Z*BQ4
               rdz=rdd*z
               SM=1.0_DP+(RDD*z)
               SN1=SM+z2AQ4*RDD !z2A9
               SN2=(2.0_DP*SM)+z2AQ4*RDD !z2A9
               CXX=sqrt(AQ4*RDD)*sqrt(SN2)
               VX=-1.0_DP*BQ4*RDD-(BQ4-AQ4)*RDD*z*RDD
               dv_du=VX/(CXX*SN1)
               sinv=SM/SN1
               cosv=z*(CXX/SN1)
               if(u.le.u1) then
                  v=PI-2.0_DP*datan(sinv/(1.0_DP-cosv))
               else
                  v=2.0_DP*datan(sinv/(1.0_DP+cosv))
               endif

               test_arg=R*(Pihalf-v-c3t*cosv)
               x_coshu_cosv=x*cosh(u)*cosv               
               if(l.ne.0.0_DP) then
                  arg_psi= cmplx(x+x_coshu_cosv,x*(T*u-S),kind=DP)            
                  if(l.eq.0.5_DP) then               
                     psi_c=sqrt(arg_psi)
                  else
                     psi_c=psi_integrand(-l,arg_psi)
                  endif
                  abs_psi=abs(psi_c) 
                  psi_x=real(psi_c,kind=DP)
                  psi_y=aimag(psi_c)
                  trig_arg=argument(psi_x,psi_y)
                  trig_arg=trig_arg+x*S
               else
                  abs_psi=1.0_DP 
                  trig_arg=x*S                                
               endif
                                !  This quantity will be tested after the inner loop
                                ! to see if we want to break out from the outer loop
               Integrand=(dcos(trig_arg)-dsin(trig_arg)*dv_du)*           &
     &              abs_psi
               Integrand=Integrand*dexp(R*(Pihalf-v-c3t*z*CXX/SN1))
               chunks(j)=Integrand*WD(j)
            enddo
         endif
         sum=0.0_DP
         do j=1,16
            sum=sum+chunks(j)
         enddo
         sum_l=sum_l+sum*0.5_DP*h2

 !        test_arg=test_arg+l*log(x)
#ifdef debug
            write(*,*) 'sum_l=',sum_l !+sum_r
            write(*,*) 'test_arg=',test_arg
         if((l.eq.0.0_DP).and.(test_arg.lt.-75.0_DP)) then            
            write(*,*) 'u=',u
         endif  
#endif
         if((l.eq.0.0_DP).and.(test_arg.lt.-75.0_DP)) then            
            exit
         endif
      enddo
      ! PARTIII : Horizontal part
      sum_h=0.0_DP         
      if((R.lt.75.0_DP))  then
         num_steps=num_steps3
         start=0.0_DP
         cut_off=u0
#ifdef debug
         write(*,*) 'start=',start,' stop=',cut_off
         write(*,*) 'h=',h,' num_steps=',num_steps
         write(*,*) '4+6anint(u0)=',4+6*ANINT(u0)
#endif
         h=(cut_off-start)/real(num_steps,kind=DP)
         a=start
         b=a+h
         sum_h=0.0_DP
         do i=1,num_steps
            aa=real(i-1,kind=DP)*h
            do j=1,16
               u=aa+0.5_DP*h*AD(j)
               coshu=cosh(u)
               sinhu=sinh(u)
               sinv=0.0_DP      !(T*u-S)/sinhu
               chunks(j)=0.0_DP
               chunks_h(j)=0.0_DP               
                                ! The argument is a negative real number
                                ! meaning the psi will be a pure imaginary number
                                ! if l=0.5, else it might be any complex number
                                ! However by a "TRICK" I can choose different 
                                ! argument at u>0 and u<0 to get the usual
                                ! symmetrization and get a single real integral
               if(l.ne.0.0_DP) then
                  if(l.eq.0.5_DP) then
                  !! note that coshu>=1 always so the argument
                  !! is <=0 if x>=0
                     abs_psi=sqrt(x*(coshu-1.0_DP))
                     trig_arg=Pihalf+R*u
                  else
                     !!! Note that the argument is real but might be negative
                     arg_psi=cmplx(x-x*coshu,0.0_DP,kind=DP)  
                     psi_c=psi_integrand(-l,arg_psi)
                     psi_x=real(psi_c,kind=DP)
                     psi_y=aimag(psi_c)
                     abs_psi=abs(psi_c)
!                     psi_x=psi_r(-l,0.5_DP,x-x*coshu)
!                     trig_arg=argument(psi_x,0.0_DP)
                     trig_arg=argument(psi_x,psi_y)
                     trig_arg=trig_arg+R*u
                  endif
               else
                  abs_psi=1.0_DP
                  trig_arg=R*u               
               endif
               !! Note that here usually |psi| is already small
               Integrand=abs_psi*dexp(max(x*coshu-PIhalf*R,-70.0_DP))        &
     &              *dcos(trig_arg)
               chunks_h(j)=WD(j)*Integrand

#ifdef debug
               if(ierr.eq.0) then
                  write(14,2770) arg_psi
                  write(15,2770) psi_c
               endif
#endif            
            enddo
            sum=0.0_DP
            do j=1,16
               sum=sum+chunks_h(j)
            enddo
            sum_h=sum_h+sum*0.5_DP*h
#ifdef debug
               write(*,*) 'sum_h(',i,')=',sum_h !+sum_l+sum_h
#endif
         enddo
      endif
      sum1=sum_r+sum_l+sum_h
      W_res=sum1
#ifdef debug
         write(*,*) 'sum_r=',sum_r
         write(*,*) 'sum_l=',sum_l
         write(*,*) 'sum_h=',sum_h
         write(*,*) 'sum=',sum1
         close(12)
         close(13)
         close(15)
 2770 FORMAT(F20.12,1X,F20.12)
 2775 FORMAT(F20.17,1X,F25.17)
#endif

      end subroutine WhittW_gq1


 
      !! For case T<=1 , x>R
      !! Psi(0.5+I*R-l,1.0+2*I*R,x)*exp(-Pi/2*R)
      ! Whittaker W-functions
      ! W(l,iR)*sqrt(Pi/2x)
      subroutine WhittW_gq2(R,l,xx,W_res)
!      use gamma_function
!      use commonvariables
      implicit none      
      real(DP),intent(in)::R,l,xx
      real(DP),intent(out)::W_res
      real(DP),dimension(16) :: chunks
      real(DP)::x,u,h,hh
      real(DP):: sum
      complex(DP)::psi_c,arg_psi
      integer:: i,j,num_steps
      real(DP)::T,cosv,coshu,sinhu
      real(DP)::trig_arg,integrand,dv_du,v,sinv,tmp1,abs_psi
      real(DP)::AA,psi_x,psi_y,RPIH,test_arg,trig_factor
      real(DP)::XC,ALFA,XCRAL,U11,C23,U22
      real(DP)::sum1,x_coshu_cosv,Tu,Ru,x_sinhu
#ifdef debug     
         open(unit=13,FILE='whitt_path2')
         open(unit=13,FILE='whitt_int2')
#endif
      X=0.5_DP*xx               ! to skip all 0.5_DP*x everywhere
      T=R/X
      if(T.gt.1.0_DP) then
         write(*,*) 'you are calling the wrong function here, T>1!!'
         stop
      endif
      !!! This part can be further optimized for other regions of k, z and R
      if(R.gt.20.0_DP) then
         num_steps=4
      else
         num_steps=8    
      endif
      T=R/X
      RPIH=R*PIhalf
      XC=SQRT((X+R)*(X-R))
!      C=(XC)/X 
      ALFA=2.0_DP*ATAN(R/(X+XC))
      XCRAL=XC+R*ALFA
!      IF(RPIH-XCRAL.LT.-125.0_DP) THEN
      IF(RPIH-XCRAL.LT.-125.0_DP) THEN
         sum1=0.0_DP
         W_res=0.0_DP
         return
      END IF
      C23=110.00_DP/(XC)
      U11=SQRT(C23)  
      IF (R.LT.(40.0_DP)) THEN
         U22=LOG((8.0_DP)+(125.0_DP/X))
      ELSE
         U22=2.0_DP
      ENDIF
      h=MIN(U11,U22)
      h=h/REAL(num_steps,kind=DP)
#ifdef debug     
         write(*,*) 'h=',h
#endif
      hh=0.5_DP*h
      sum1=0.0_DP      
      do i=1,num_steps
         aa=real(i-1,kind=DP)*h
         do j=1,16
            chunks(j)=0.0_DP
            u=aa+hh*AD(j)
            coshu=cosh(u)
            sinhu=sinh(u)
            Tu=T*u
            Ru=R*u
            x_sinhu=x*sinhu
            sinv=Tu/sinhu
            tmp1=sinhu*sqrt((x_sinhu-Ru)*(x_sinhu+Ru))
            dv_du=R*(sinhu-u*coshu)/tmp1

            tmp1=sqrt((sinhu-Tu)*(sinhu+Tu))
            cosv=tmp1/sinhu
            v=2.0_DP*datan(sinv/(1.0_DP+cosv))
            x_coshu_cosv=x*coshu*cosv
            if(l.ne.0.0_DP) then
               
               arg_psi=cmplx(x+x_coshu_cosv,x*Tu,kind=DP)
               if(l.eq.0.5_DP) then
                  psi_c=sqrt(arg_psi)
               else
                  psi_c=psi_integrand(-l,arg_psi)
               endif
               abs_psi=abs(psi_c) 
               psi_x=real(psi_c,kind=DP)
               psi_y=aimag(psi_c)
               trig_arg=argument(psi_x,psi_y)
               trig_factor=(dcos(trig_arg)-dv_du*dsin(trig_arg))
!            elseif(complex_bes) then
            else
               abs_psi=1.0_DP 
               trig_arg=0.0_DP
               trig_factor=1.0_DP           
            endif
            test_arg=-1.0_DP*v*R-x_coshu_cosv+xcral
            Integrand=exp(R*(Pihalf-v)-x_coshu_cosv)*abs_psi                   &
     &           *trig_factor
            chunks(j)=Integrand*WD(j)
#ifdef debug     
               write(14,2770) arg_psi
               write(15,2770) psi_c
#endif
         enddo
         sum=0.0_DP
         do j=16,1,-1
            sum=sum+chunks(j)
         enddo
#ifdef debug     
            write(*,*) 'partial sum=',sum
#endif
         sum1=sum1+sum*hh         
         if(test_arg.lt.(-55.0_DP)) then
            !write(*,*) 'u_stop=',u
            exit
         endif
      enddo
      W_res=sum1 
#ifdef debug     
         write(*,*) 'sum=',sum1
         close(12)
         close(13)
 2770    FORMAT(F20.12,1X,F20.12)
#endif
      end subroutine WhittW_gq2





     !! For case  T~1 
 !! Psi(0.5+I*R-l,1.0+2*I*R,x)*exp(-Pi/2*R)
      ! W(l,R,x)*sqrt(Pi/2x)
      subroutine WhittW_gq3(R,l,xx,W_res)
      implicit none      
      real(DP),dimension(16) :: chunks
      real(DP),intent(in)::R,l,xx
      real(DP),intent(out)::W_res
      real(DP)::x,u,h,psi_x,psi_y,abs_psi
      real(DP):: sum,delta
      complex(DP)::Psi_c,arg_psi
      integer:: i,j,num_steps
      real(DP)::cosv,coshu,sinhu
      real(DP)::trig_arg,integrand,dv_du,v,sinv,tmp1
      real(DP)::sum1,x_coshu_cosv
      real(DP)::AA
      real(DP)::test_arg,S33,V11,V12,G1,G2,U12,U11
      real(DP)::AQ0,AQ1,AQ2,AQ3,BQ0,BQ1,BQ2,BQ3,B9,A9
      X=0.5_DP*xx  
      delta=R-x
      sum1 =0.0_DP
!      T=R/X  ~~1  here and is treated as exactly=1
      !! The number of steps can be further optimized

      if(R.lt.10.0_DP) then
         num_steps=8     
      elseif(R.lt.100.0_DP) then
         num_steps=12
      elseif(r.lt.1000.0_DP) then 
         num_steps=16
      else
         num_steps=32
      endif
      G1=(4.0_DP)*dsqrt(3.0_DP)
      G2=-1.0_DP/3.0_DP
      U12=G1*(R**G2)
      IF (R.LT.(120.0_DP)) THEN
         U11=LOG((8.0_DP)+(200.0_DP/R))
      ELSE IF (R.LT.(333.0_DP)) THEN
         U11=1.5_DP
      ELSE
         U11=U12
      ENDIF
      H=U11/REAL(num_steps,kind=DP)

#ifdef debug
      write(*,*) 'num_steps=',num_steps,' h=',h
#endif
      do i=1,num_steps                  
         aa=real(i-1,kind=DP)*h
         if(aa.ge.(0.125_DP)) then
            do j=1,16
                                !            u=0.5_DP*((b-a)*AD(j)+(b+a))           
               u=aa+0.5_DP*h*AD(j)
               coshu=cosh(u)
               sinhu=sinh(u)            
               sinv=u/sinhu
               tmp1=sinhu*sqrt((sinhu-u)*(sinhu+u))
               dv_du=(sinhu-u*coshu)/(tmp1)
!               cosv=dsqrt((sinhu-u)*(sinhu+u))/sinhu
               cosv=sqrt(1.0_DP-(u/sinhu)**2)
               x_coshu_cosv=x*coshu*cosv
        !               v=2.0_DP*datan(sinv/(1.0_DP+cosv))
               v=2.0_DP*argument(1.0_DP+cosv,sinv)
               ! Evaluation of Psi(u) in different cases
               if(l.ne.0.0_DP) then
                  arg_psi=cmplx(x+x_coshu_cosv,x*u,kind=DP)                  
                  if(l.eq.0.5_DP) then
                     psi_c=sqrt(arg_psi)
                  else
                     psi_c=psi_integrand(-l,arg_psi)
                  endif
                  psi_x=real(psi_c,kind=DP)
                  psi_y=aimag(psi_c)
                  abs_psi=abs(psi_c) 
                  trig_arg=argument(psi_x,psi_y)
                  trig_arg=trig_arg+delta*u 
!               elseif(complex_bes) then
               else
                  abs_psi=1.0_DP
                  trig_arg=delta*u 
               endif
                                !!!!Testing for K_Bessel
               test_arg=R*(Pihalf-v)-x_coshu_cosv
               Integrand=(dcos(trig_arg)-dv_du*dsin(trig_arg))              &
     &              *exp(max(R*Pihalf-R*v-x_coshu_cosv,-70.0_DP))*       &
     &              abs_psi
               chunks(j)=Integrand*WD(j)            
#ifdef debug
                  write(14,2770) arg_psi
                  write(15,2770) psi_c
# endif
            enddo
         else  ! Use power-series to avoid cancellation
            do j=1,16
               u=aa+0.5_DP*h*AD(j)
               AQ0=(U*U*U*U*U*U*U*U*U*U*U*U*U*U)*RF(16)
               AQ1=(U*U*U*U*U*U*U*U*U*U)*(RF(12)+RF(14)*U*U)+AQ0
               AQ2=(U*U*U*U*U*U)*(RF(8)+RF(10)*U*U) + AQ1  
               AQ3=(U**2)*(RF(4)+RF(6)*U*U) + AQ2
               A9=RF(2)+AQ3
               BQ0=(U*U*U*U*U*U*U*U*U*U*U*U*U*U)*RF(17)
               BQ1=(U*U*U*U*U*U*U*U*U*U)*(RF(13)+RF(15)*U*U)+BQ0
               BQ2=(U*U*U*U*U*U)*(RF(9)+RF(11)*U*U) + BQ1  
               BQ3=(U**2)*(RF(5)+RF(7)*U*U) + BQ2
               B9=RF(3)+BQ3               
               coshu=1.0_DP+A9*(U*U)
               S33=1.0_DP+B9*(U*U)
               V11=B9*(2.0_DP+B9*U*U)
               V12=dsqrt(V11)
               dv_du=(B9-A9)/(V12*S33)
               sinv=(1.0_DP)/(S33)
               cosv=U*(V12/S33)  
               
               V=(2.0_DP)*datan(sinv/(1.0_DP+cosv))
!               v=dacos(cosv)
               x_coshu_cosv=x*coshu*cosv
               ! Evaluation of Psi(u) in different cases
               if(l.ne.0.0_DP) then
                  arg_psi=cmplx(x+x_coshu_cosv,x*u,kind=DP)
                  if(l.eq.0.5_DP) then               
                     psi_c=sqrt(arg_psi)
                  else
                     psi_c=psi_integrand(-l,arg_psi)
                  endif
                  psi_x=real(psi_c,kind=DP)
                  psi_y=aimag(psi_c)
                  abs_psi=abs(psi_c) 
                  trig_arg=argument(psi_x,psi_y)
                  trig_arg=trig_arg+delta*u               
               else
                  abs_psi=1.0_DP 
                  trig_arg=delta*u
               endif
               test_arg=R*(Pihalf-v)-x_coshu_cosv
               Integrand=(dcos(trig_arg)-dv_du*dsin(trig_arg))              &
     &              *dexp(max(R*(Pihalf-v)-x_coshu_cosv,-70.0_DP))       &
     &              *abs_psi
               chunks(j)=Integrand*WD(j)
#ifdef debug
                  write(14,2770) arg_psi
                  write(15,2770) psi_c
#endif
            enddo
         endif
         sum=0.0_DP
         do j=16,1,-1
            sum=chunks(j)+sum  
         enddo
         sum1=sum*0.5_DP*h+sum1
         if(test_arg.lt.-55.0_DP) then
            exit
         endif
      enddo
      W_res=sum1

#ifdef debug
         close(13)
         close(12)
 2770    FORMAT(F20.12,1X,F20.12)
#endif
      end subroutine Whittw_gq3


!!!!!!!!!!!!!!!!!!!!!!!!!!!
!!!!!!  Evaluation of the Psi-functions inside the integrands, i.e. Psi(a,1/2,z) with real a and complex z
!!!!!!!!!!!!!!!!!!!!!!!!!!!

    

      !!! This is Psi(a,1/2,z)
      !!! with a complex z and real a!
      function psi_integrand(a,z)
      implicit none
      real(DP)::a
      complex(DP)::z
      real(DP)::conv_r,asymp_r,r,x
      integer::info
      complex(DP)::psi_integrand,psi,PSI2
      info=1
      conv_r=4.0_DP
      asymp_r=20.0_DP
      !!! Must determine which method to use
      r=abs(z)
      x=real(z,kind=DP)
      !! If a is a negative integer (checked from the start we use that fact) 
!      if(a_is_neg_int.and.(r.gt.2.0)) then 
!         call psi_int_recursive(a,z,psi,info,.true.)
!      endif
!      call psi_int_recursive(a,z,psi,info)
!      psi_integrand=psi
!      return
      if(r.lt.conv_r) then
        call psi_int_pow(a,z,psi,info)
      elseif(x.lt.0.0_DP) then
         if(((r.lt.5.0_DP*conv_r).and.(a.lt.1.0_DP)).or.                            &
     &      ((r.lt.3.0_DP*conv_r).and.(a.lt.2.0_DP))) then
            call psi_int_pow(a,z,psi,info)
         endif
      endif
      if((info.eq.1).and.(r.ge.asymp_r)) then 
         call psi_int_aspow(a,z,psi,info)
      endif
      !info=0 means succesful
      ! otherwise we use the "fall-back" algorithm. i.e. recursive 
      if(info.ne.0) then
         call psi_int_recursive(a,z,psi,info)
      endif

!      call psi_int_recursive(a,z,psi,info)
      if(info.ne.0) then 
         write(*,*) 'ERROR: info=',info
         write(*,*) 'arg=',z
         write(*,*) 'l/2=',a
         write(*,*) 'psi=',psi
         write(*,*) '|psi|=',abs(psi)
         write(*,*) 'r=',r
         call psi_int_recursive(a,z,psi2,info)
         write(*,*) 'l/2=',a
         write(*,*) '|psi_REC-PASI|=',abs(psi2-PSI)
         stop
      endif
      psi_integrand=psi
      end function 


      
!     !!! Power series representation of Psi(a,1/2,z)
!     !!! Use Psi(a,1/2,z)=sqrt(Pi)*(Phi(a,1/2,z)/GAMMA(a+0.5)
!     !!!                  -2*sqrt(z)*Phi(a+0.5,1.5,z)/GAMMA(a)
!     !!! and
!     !!!     Psi(a,1/2,-z)=e(-z)*sqrt(Pi)*(Phi(1/2-a,1/2,z)/GAMMA(a+0.5)
!     !!!                  -2*sqrt(-z)*Phi(1-a,1.5,z)/GAMMA(a)  
!     !!!
!     !!! We use the formula so that the argument has Real part <0
!   
      subroutine psi_int_pow(a_r,z_in,psi,info)
      implicit none
      real(DP),intent(in)::a_r
      complex(DP),intent(in)::z_in
      complex(DP),intent(out)::psi
      integer,intent(inout)::info
      complex(DP)::z,tmp1,A0,A1,A2,tmp2,gc1,gc2,c_ett
      real(DP)::jj
      integer::j
      integer,parameter::N_max=1000
      real(DP)::aa1,aa2,c1,c2,Ri,g1,g2
      info=0
      tmp1=cmplx(0.0_DP,0.0_DP,kind=DP)
      tmp2=cmplx(0.0_DP,0.0_DP,kind=DP)
      c_ett=cmplx(1.0_DP,0.0_DP,kind=DP)
      z=z_in
      aa1=a_r
      aa2=0.5_DP+a_r
      c1=0.5_DP
      c2=1.5_DP
      if(is_int(a_r+0.5_DP).and.(a_r.le.-0.5_DP)) then !! a+1/2 is a non-pos. integer
         !! If so, the first term vanishes
      else
!c     !!!! Phi(a,1/2,z)
         if(aa1.eq.c1) then
            A1=exp(z)           !! remember Phi(a,a,z)=e(z)
         else
            A0=c_ett
            A1=A0+z*cmplx(aa1/c1,0.0_DP,kind=DP)
!c     $!         write(*,*) 'A1=',A1,'=1+',z,'*',a,'/',c
            aa1=aa1-1.0_DP
            c1=c1-1.0_DP
            do j=2,N_max
               jj=real(j,kind=DP)
!               Ri=(aa1+jj-1.0_DP)/(jj*(c1+jj-1.0_DP))
               Ri=(aa1+jj)/(jj*(c1+jj))
               A2=A1+(A1-A0)*z*cmplx(Ri,0.0_DP,kind=DP)
               A0=A1
               A1=A2
               if(abs(A0-A1)/abs(A1).lt.eps) then
                  exit
               endif
               
            enddo
            if(j.ge.N_max) then
               info=-j
               return
            endif
         endif
!         write(*,*) 'j1=',j
!c     !! A1=Phi(a,1/2,z)
         tmp1=A1
      endif
      if(a_is_neg_int) then !! a is a non-pos. integer
                                !! If so, the second term vanishes
      else         
         if(aa2.eq.c2) then
            A1=exp(z)           !! remember Phi(a,a,z)=e(z)
         else
            A0=c_ett
            A1=A0+z*cmplx(aa2/c2,0.0_DP,kind=DP)
            aa2=aa2-1.0_DP
            c2=c2-1.0_DP
            do j=2,N_max
               jj=real(j,kind=DP)
               Ri=(aa2+jj)/(jj*(c2+jj))
               A2=A1+z*(A1-A0)*cmplx(Ri,0.0_DP,kind=DP)
               A0=A1
               A1=A2
               if(abs((A0-A1)).lt.eps) then
                  exit
               endif
            enddo
            if(j.ge.N_max) then
               info=-j
               return
            endif
!c     !! A1=Phi(a+0.5,3/2,z)  !! or Phi(1-a,3/2,-z)
         endif
         tmp2=A1
      endif

      !! Treat a few special cases
      if(abs(a_r).eq.0.25_DP) then
         if(a_r.gt.0.0_DP) then 
         !!g1=gamma(a_r+0.5)=gamma(0.75)
            g1=1.225416702465177645129098_DP
         !!g2=gamma(a_r)=gamma(0.25)
            g2=3.625609908221908311930685_DP
         else
         !!g1=gamma(a_r+0.5)=gamma(0.25)
            g1=3.625609908221908311930685_DP
         !!g2=gamma(a_r)=gamma(-0.25)            
            g2=-4.901666809860710580516393_DP
         endif
         gc1=cmplx(g1,0.0_DP,kind=DP)
         gc2=cmplx(2.0_DP/g2,0.0_DP,kind=DP)
         psi=tmp1/gc1-sqrt(z_in)*tmp2*gc2
      else
         if(a_is_neghalfint) then !! a+1/2 is a non-pos. integer
            psi=sqrt(z_in)*tmp2*cmplx(2.0_DP/gamma_r(a_r),0.0_DP,                 &
     &           kind=DP)
         elseif(a_is_neg_int) then !! a is a non-pos. integer
            psi=tmp1/cmplx(gamma_r(a_r+0.5_DP),0.0_DP,kind=DP)
         else
            psi=tmp1/cmplx(gamma_r(a_r+0.5_DP),0.0_DP,kind=DP)-                    &
     &           sqrt(z_in)*tmp2*cmplx(2.0_DP/gamma_r(a_r),0.0_DP,                 &
     &           kind=DP)
         endif
      endif
      psi=cmplx(sqrtPi,0.0_DP,kind=DP)*psi      
      end subroutine psi_int_pow


!c      !! Psi(a,1/2,z) with asymptotic powerseries in 1/z
!c     !!! Psi(a,c,x)=Sum_{n=0}^{N}(-1)^n (a)_n * (a-c+1)_n / n! *x^(-a-n)
!c     !!!            +O(|x|^(-a-N-1))
!c     !!!
!c     !!! 
      !! Note, if a<0  then we have to go at least up to |a|+1 otherwise we are not decreasing at all!
      subroutine psi_int_aspow(a,z,psi,info)
      implicit none
      integer::j
      integer::info,ab
      real(DP),intent(in)::a
      complex(DP),intent(out)::psi
      complex(DP),intent(in)::z
      complex(DP)::Psi1,T1,factor,a_c_m_c_c,nn_cone
      complex(DP)::a_c,cone,nn_c
      info=0
      a_c=cmplx(a,0.0_DP,kind=DP)
      a_c_m_c_c=cmplx(a-0.5_DP,0.0_DP,kind=DP) ! a-c
      cone=cmplx(1.0_DP,0.0_DP,kind=DP)
      T1=-a_c*(a_c_m_c_c+cone)/z
      Psi1=cone+T1
      ab=0
      if(a.lt.-1.0_DP) ab=-nint(a)+1
      loop: do j=1,N_max_as
         nn_c=cmplx(j,0.0_DP,kind=DP)
         nn_cone=cmplx(1+j,0.0_DP,kind=DP)
         factor=(a_c+nn_c)*(a_c_m_c_c+nn_cone)/((nn_cone)*z)
         ! remember this is an asymptotic expansion
         ! we must cut of the series when it start to grow
         T1=-cone*T1*factor
         Psi1=Psi1+T1
         if((j.ge.ab).and.((abs(T1/Psi1).lt.eps).or.(abs(factor).ge.        &
     &        1.0_DP))) then             
            exit loop
         endif
      enddo loop
      if(j.ge.N_max_as) info=-j
      psi=(z**(-a_c))*Psi1
      end subroutine


      ! Here x=z= is complex
      ! I use only complex arithmetic
      ! This works much better than mixing...
      !! Modified, for c=0.5
      !!
      subroutine psi_int_recursive(a,z,psi,info)
      implicit none
      real(DP),intent(in)::a
      complex(DP),intent(in)::z
      complex(DP),intent(out)::psi
      integer::info
      complex(DP)::Sn0,Sn1,Rn0,Rn1,a_c
!      complex(DP)::an,bn,den
!      complex(DP),dimension(:),allocatable::lambda
      complex(DP)::psi0,jjc,a1,a2,a3
      integer :: N,j,n_start
      real(DP)::x,jj,r
      info=0
      r=abs(z)    
      x=real(z,kind=DP)
!      y=imag(z)
      call get_lambda(a,0.5_DP)
      !!! If a= neg. integer then the recursion terminates after n=1-a
      if(a_is_neg_int) then 
         n_start=anint(1.0_DP-a)         
      elseif(r.gt.30.0_DP) then
         n_start=14
      elseif(r.gt.25.0_DP) then
         n_start=16
      elseif(r.gt.20.0_DP) then
         n_start=18
      elseif(r.gt.15.0_DP) then
         n_start=22
      elseif(r.gt.12.0_DP) then
         n_start=25
      elseif(r.gt.9.0_DP) then
         n_start=30
      elseif(r.gt.7.0_DP) then
         n_start=36
      elseif(r.gt.6.0_DP) then
         n_start=40
      elseif(r.gt.5.0_DP) then           
         n_start=45
      else
         n_start=100
      endif
      if(a.gt.0.0_DP) then
         if(a.le.4.0_DP) then
            n_start=ceiling(a)*n_start
         else
            n_start=4*n_start
         endif
         if(x.lt.0.0_DP) then
            n_start=2*n_start
         endif
      endif
      psi0=cmplx(1000.0_DP,0.0_DP,kind=DP)
      psi=czero
      a_c=cmplx(a,0.0_DP,kind=DP)
      a1=cone-a_c
      a2=chalf-ctwo*a_c
      a3=chalf+a_c
      do N=n_start,N_max_recursive,2
         Sn0=czero
         Rn0=czero
         do j=N,1,-1
            jjc=cmplx(real(j,kind=DP),0.0_DP,kind=DP)
            Rn1=(a1-jjc)/((a2-ctwo*jjc-z)+                                 &
     &           Rn0*(a3+jjc))
            Sn1=Rn1*(lambda(j)+Sn0)
            Rn0=Rn1
            Sn0=Sn1
         enddo
         psi=cone/(z**(a_c)*(cone+Sn0))
         if((abs(psi0-psi)/abs(psi).lt.eps).and.(N.gt.n_start)) then
            exit
         endif
         psi0=psi
      enddo
!      write(*,*) 'N=',N
!      write(*,*) 'N_start=',n_start
      if(N.ge.N_max_recursive) then
         info=-N
         !!! Also give the error in the convergence scheme as the real part
         psi=cmplx(abs(psi0-psi),0.0_DP,kind=DP)
       !  write(*,*) 'Max recursive reached!'
      endif         
      end subroutine psi_int_recursive









  
      !! First set up all the lambdas
      !! Observe that when computing the Whittaker function
      !! a=-weight/2 and c=1/2 in all cases
      !! hence the lambdas are the same...
      subroutine get_lambda(a,c)
      implicit none
      integer::j
      real(DP),intent(in)::a,c
      if(lambda_set.and.(allocated(lambda))) then
         if((a.eq.a_set_l).and.(c.eq.c_set_l)) then
            return
         endif  
      endif
      a_set_l=a
      c_set_l=c
      if(allocated(lambda)) then
         deallocate(lambda)
      endif
      allocate(lambda(0:N_max_recursive))
      lambda(0)=cmplx(1.0D0,0.0_DP,kind=DP)
      do j=1,N_max_recursive 
         lambda(j)=-lambda(j-1)*cmplx(                                          &
     &        (c-a-real(j,kind=DP))/real(j,kind=DP),0.0_DP,kind=DP)

      enddo
      lambda_set=.true.
      end subroutine get_lambda



      !! Hyperbolic functions of complex argument
      function cosh_c(z)
      implicit none
      complex(DP),intent(in)::z
      complex(DP)::cosh_c
      cosh_c=exp(z)+exp(-z)
      cosh_c=cosh_c*chalf
      end function

      function sinh_c(z)
      implicit none
      complex(DP),intent(in)::z
      complex(DP)::sinh_c
      sinh_c=exp(z)-exp(-z)
      sinh_c=sinh_c*chalf
      end function

      subroutine clean_whittaker()
      if(allocated(A_uk)) deallocate(A_uk)
      if(allocated(A_k)) deallocate(A_k)
      if(allocated(B_k)) deallocate(B_k)
      if(allocated(lambda)) deallocate(lambda)
      call clean_bernoulli()
      end subroutine

      end module



   
