/*
********************************************
* Decomposition.mg                         *
*                                          *
* Computes the decomposition of a          *
* finite dimensional commutative algebra   *
* into local factors.                      *
*                                          *
* Gabor Wiese, version of 31/12/2007       *
********************************************
*/

// ******* decomposition over base field *************

intrinsic Decomposition ( M :: Mtrx ) -> Tup
{Given a matrix M, computes a decomposition of the standard
vector space such that the minimal polynomial of M is a prime
power on each summand. The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.}
  L := <>;
  F := CoefficientRing(M);
  mipo := Factorisation(MinimalPolynomial(M));
  for fac in mipo do
    W := Kernel(Evaluate(fac[1]^(fac[2]),ChangeTypeToAlgebra(M)));
    T := BaseChangeMatrices(W);
    Append(~L, T);
  end for;

  return L;
end intrinsic;

intrinsic Decomposition ( L :: SeqEnum ) -> Tup
{Given a sequence L of commuting matrices, computes a 
decomposition of the standard vector space such that 
the minimal polynomial of each l in L is a prime power
on each summand. The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.}
  output := <>;
  dec1 := Decomposition(L[1] );
  if #L eq 1 then
    return dec1;
  else
    for d in dec1 do
      L2 := [ BaseChange(L[i], d) : i in [2..#L] ];
      dec2 := Decomposition(L2);
      for a in dec2 do        
        Append(~output, BaseChange(d,a));
      end for;
    end for;
    return output;
  end if;
end intrinsic;

intrinsic Decomposition ( A :: AlgMat ) -> Tup
{Given a commutative matrix algebra, computes a 
decomposition of the standard vector space such that 
each the minimal polynomial of each a in A is a prime power
on each summand. The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.}
  return Decomposition(Basis(A));
end intrinsic;


intrinsic AlgebraDecomposition ( A :: AlgMat ) -> SeqEnum
{Given a matrix algebra A over a finite field k, 
returns a list of all local factors of A.}
  L := Basis(A);
  dec := Decomposition(L);
  return [MatrixAlgebra([BaseChange(l,D) : l in L]) : D in dec];
end intrinsic;



// ******* decomposition over residue field *************

intrinsic DecompositionOverResidueField ( M :: Mtrx : DegBound := 0 ) -> Tup
{Given a matrix M, computes a decomposition of the standard
vector space such that M acts as multiplication by a scalar 
on each summand.
For this, each local factor of the algebra is base changed to its residue field.
The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.}
  L := <>;
  F := CoefficientRing(M);
  mipo := Factorisation(MinimalPolynomial(M));
  for fac in mipo do
    if (DegBound eq 0) or  (Degree(fac[1]) le DegBound) then
      K := ext<F | Degree(fac[1]) >;
      Pol := PolynomialRing(K);
      facK := Factorisation(Pol!(fac[1]));

      for i := 1 to #facK do
        W := Kernel(Evaluate(facK[i][1]^(fac[2]),ChangeTypeToAlgebra(ChangeRing(M,K))));
        T := BaseChangeMatrices(W);
        Append(~L, T);
      end for;
    end if;
  end for;

  return L;
end intrinsic;

intrinsic DecompositionUpToConjugation ( M :: Mtrx : DegBound := 0 ) -> Tup
{Given a matrix M, computes a decomposition of the standard
vector space such that M acts as multiplication by a scalar 
on each summand.
For this, each local factor of the algebra is base changed to its residue field.
The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.
Summands conjugate under the absolute Galois group 
only appear once.}
  L := <>;
  F := CoefficientRing(M);
  mipo := Factorisation(MinimalPolynomial(M));
  for fac in mipo do
    if (DegBound eq 0) or  (Degree(fac[1]) le DegBound) then
      K := ext<F | Degree(fac[1]) >;
      Pol := PolynomialRing(K);
      facK := Factorisation(Pol!(fac[1]));
      W := Kernel(Evaluate(facK[1][1]^(fac[2]),ChangeTypeToAlgebra(ChangeRing(M,K))));
      T := BaseChangeMatrices(W);
      Append(~L, T);
    end if;
  end for;

  return L;
end intrinsic;

intrinsic DecompositionOverResidueField ( L :: SeqEnum : DegBound := 0 ) -> Tup
{Given a sequence L of commuting matrices, computes a 
decomposition of the standard vector space such that 
each l in L acts as multiplication by a scalar 
on each summand.
For this, each local factor of the algebra is base changed to its residue field.
The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.}
  output := <>;
  dec1 := DecompositionOverResidueField(L[1] : DegBound := DegBound);
  if #L eq 1 then
    return dec1;
  else
    for d in dec1 do
      L2 := [ BaseChange(L[i], d) : i in [2..#L] ];
      dec2 := DecompositionOverResidueField(L2 : DegBound := DegBound);
      for a in dec2 do        
        Append(~output, BaseChange(d,a));
      end for;
    end for;
    return output;
  end if;
end intrinsic;

intrinsic DecompositionOverResidueField ( A :: AlgMat : DegBound := 0 ) -> Tup
{Given a commutative matrix algebra, computes a 
decomposition of the standard vector space such that 
each a in A acts as multiplication by a scalar
on each summand.
For this, each local factor of the algebra is base changed to its residue field.
The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.}
  return DecompositionOverResidueField(SetToSequence(Generators(A)) : DegBound := DegBound);
end intrinsic;


intrinsic DecompositionUpToConjugation ( L :: SeqEnum : DegBound := 0) -> Tup
{Given a sequence L of commuting matrices, computes a 
decomposition of the standard vector space such that 
each l in L acts as multiplication by a scalar
on each summand.
For this, each local factor of the algebra is base changed to its residue field.
The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.
Summands conjugate under the absolute Galois group 
only appear once.}
  output := <>;
  dec1 := DecompositionUpToConjugation(L[1] : DegBound := DegBound);
  if #L eq 1 then
    return dec1;
  else
    for d in dec1 do
      L2 := [ BaseChange(L[i], d) : i in [2..#L] ];
      dec2 := DecompositionUpToConjugation(L2 : DegBound := DegBound);
      for a in dec2 do        
        Append(~output, BaseChange(d,a));
      end for;
    end for;
    return output;
  end if;
end intrinsic;


intrinsic DecompositionUpToConjugation ( A :: AlgMat : DegBound := 0 ) -> Tup
{Given a commutative matrix algebra, computes a 
decomposition of the standard vector space such that 
each a in A acts as multiplication by a scalar
on each summand.
For this, each local factor of the algebra is base changed to its residue field.
The output is a tuple consisting of
base change tuples <C,D> corresponding to the summands.
Summands conjugate under the absolute Galois group 
only appear once.}
  return DecompositionUpToConjugation(SetToSequence(Generators(A)) : DegBound := DegBound);
end intrinsic;


intrinsic AlgebraDecompositionUpToConjugation ( A :: AlgMat : DegBound := 0 ) -> SeqEnum
{Given a matrix algebra A over a finite field k, 
returns a list consisting for each local factor B of A one local factor of B tensor K
where K is the residue field of B.
Identical with ChangeToResidueField.}
  L := SetToSequence(Generators(A));
  dec := DecompositionUpToConjugation(L : DegBound := DegBound);
  return [MatrixAlgebra([BaseChange(l,D) : l in L]) : D in dec];
end intrinsic;

intrinsic ChangeToResidueField ( A :: AlgMat : DegBound := 0) -> SeqEnum
{Given a matrix algebra A over a finite field k, 
returns a list consisting for each local factor B of A one local factor of B tensor K
where K is the residue field of B.
Identical with AlgebraDecompositionUpToConjugation.}
  return AlgebraDecompositionUpToConjugation(A : DegBound := DegBound );
end intrinsic;


intrinsic AlgebraDecompositionOverResidueField ( A :: AlgMat : DegBound := 0 ) -> SeqEnum
{Given a matrix algebra A over a finite field k, 
returns a list of all local factors of A after base change
to their residue fields.}
  L := SetToSequence(Generators(A));
  dec := DecompositionOverResidueField(L : DegBound := DegBound);
  return [MatrixAlgebra([BaseChange(l,D) : l in L]) : D in dec];
end intrinsic;


// ************************* Lower triangular representation ******************

intrinsic IdealTorsion ( m :: AlgMat ) -> ModTupFld
{Given an ideal m in a matrix algebra of degree d over a field F, calculate
the sub vector space of F^d consisting of those elements killed
by every element of m.}
  local g,i,Vi;

  if Degree(m) eq 0 then return VectorSpace(GF(2),0); end if;

  g := SetToSequence(Generators(m));
  Vi := VectorSpace(CoefficientRing(m),Degree(m));
  for i := 1 to #g do
    Vi := Vi meet (Kernel (g[i]));     
  end for;

  return Vi;
end intrinsic;


intrinsic CommonLowerTriangular ( A :: AlgMat ) -> AlgMat
{Given a local commutative matrix algebra A this function returns
an isomorphic matrix algebra whose matrices are all lower
triangular.}
  if Dimension(A) eq 0 then return A; end if;
  
  d := Degree(A);
  C := A!0; // the base change matrix to be determined
  L := [];  // store the dimensions of the primary spaces in here
  r := 1;   // row number of matrix C currently treated

  // get maximal ideals
  max := MaximalIdeals(A);
  error if #max ne 1, "Error. The algebra is not local. Number of factors:", #max;

  // for the maximal ideal m calculate the power m^i st m^i = m^(i+1)
  // and a basis of V[ m^i ], which becomes part of the basis to
  // be calculated

  n := max[1];
  V := IdealTorsion (n);
  b := Basis(V);

  m := n * max[1];
  while m ne n do
    W := IdealTorsion(m);
    b := ExtendBasis (b, W);
    n := m;
    m := m * max[1];
  end while;

  for j := 1 to #b do
    C[j] := b[j];
  end for;  

  D := C^(-1);
  B := Basis(A);
  E := [BaseChange(b,<C,D>) : b in B];
  
  return MatrixAlgebra(E);

end intrinsic;


