/*
********************************************
* MatrixAlgebras.mg                        *
*                                          *
* Various functions concerning             *
* commutative matrix algebras.             *
*                                          *
* Gabor Wiese, version of 31/12/2007       *
********************************************
*/

// ********* various functions ********************

intrinsic ChangeTypeToAlgebra ( M :: Mtrx ) -> Mtrx
{Given a square matrix M, return M as an element of
the corresponding matrix algebra.}
  d := Nrows(M);
  e := Ncols(M);
  require (d eq e) : "Argument 1 must be a square matrix.";
  F := CoefficientRing(M);
  return MatrixAlgebra(F,d)!M;
end intrinsic;

// ********* base change functions *****************

intrinsic BaseChangeMatrices ( V :: ModTupFld ) -> Tup
{Computes a tuple <C,D> consisting of the standard 
base change matrices for the basis of the vector
space V.}
  C := BasisMatrix(V);
  M := MatrixAlgebra(CoefficientRing(V),Dimension(V));
  D := Transpose(Solution(Transpose(C),M!1));
  return <C,D>;
end intrinsic;

intrinsic BaseChange ( M :: Mtrx, T :: Tup ) -> Mtrx
{Given a matrix M and a tuple T = <C,D> of base change
matrices (for a subspace), computes the matrix of M wrt. 
the basis corresponding to T.}
  F := CoefficientRing(T[1]);
  return T[1] * ChangeRing(M,F) * T[2];
end intrinsic;

intrinsic BaseChange ( S :: Tup, T :: Tup ) -> Tup
{Computes the composition of the base change matrices in T,
followed by those in S.}
  F := CoefficientRing(T[1]);
  return <T[1] * ChangeRing(S[1],F), ChangeRing(S[2],F) * T[2]>;
end intrinsic;

intrinsic BaseChange ( M :: AlgMat, T :: Tup ) -> AlgMat
{Given a matrix algebra M and a tuple T = <C,D> of base 
change matrices (for a subspace), computes the matrix algebra 
of M wrt. the basis corresponding to T.}
  F := CoefficientRing(T[1]);
  alg := MatrixAlgebra(F,Nrows(T[1]));
  if Dimension(M) eq 0 then
    return sub< alg | >;
  else
    gen := [alg!(T[1] * ChangeRing(g,F) * T[2]) : g in Generators(M)];
    return sub< alg | gen >;
  end if;
end intrinsic;


// ******* matrix algebra creation *********************

intrinsic MatrixAlgebra ( L :: SeqEnum ) -> AlgMat
{Given a list of matrices, return the matrix
algebra generated by L.}
  require L ne [] : "Argument 1 must not be the empty list.";
  require Nrows(L[1]) eq Ncols(L[1]) : 
    "Matrices in Argument 1 must be square.";
  F := BaseRing(L[1]);
  d := Nrows(L[1]);
  a := MatrixAlgebra(F,d);
  return sub< a | [a!l : l in L] >;
end intrinsic;

intrinsic Transpose ( A :: AlgMat ) -> AlgMat
{Given a matrix algebra A, creates the transposed matrix algebra.}
  local L,M,i,B;

  L := SetToSequence(Generators(A));
  M := [];
  for i := 1 to #L do
    M[i] := Transpose(L[i]);
  end for;

  B := MatrixAlgebra(M);
  return B;
end intrinsic;


// ************************ Regular representation ***************************

intrinsic RegularRepresentation ( A :: AlgMat ) -> AlgMat
{Computes the regular representation of the commutative matrix algebra A.}
  if Dimension(A) eq 0 then return A; end if;

  B := Basis(A);
  dim := #B;
  d := Degree(A);
  R := BaseRing(A);

  M := MatrixAlgebra(R,dim);
  L := [];

  Mt := RMatrixSpace(R,d,d);
  At := RMatrixSpaceWithBasis([Mt!b : b in B]);

  for b in B do
    l := M!0;
    for i := 1 to dim do
      l[i] := Vector(Coordinates(At,At!(b*B[i])));
    end for;
    Append (~L, l);
  end for;

  return MatrixAlgebra(L);
end intrinsic;


// ********************** Jordan form ***************************************

// because the usual one crashes for 0x0 matrices
_JordanForm := function ( A )
  if NumberOfRows (A) ge 1 then
    return JordanForm(A);
  else
    return A,A,<>;
  end if;
end function;

intrinsic JordanDecomposition ( M :: Mtrx ) -> Mtrx, Mtrx
{Calculates the decomposition M = D + N with D diagonalisable
and N nilpotent.}
  local i,j,z,D,N,F,n,alg,J,C,S,d;

  F := BaseRing(M);
  n := NumberOfColumns(M);
  alg := MatrixAlgebra (F,n);
  N := alg!0;
  J,C,S := _JordanForm (M);
  z := 1;
  for i := 1 to #S do
    d := Degree (S[i][1]);
    for j := 1 to S[i][2] - 1 do
      N := N + MatrixUnit (alg, z+d-1,z+d);
      z := z + d;
    end for; 
    z := z + d;
  end for;

  N := C^(-1) * N * C;
  D := M - N;
  return D,N;
end intrinsic;

