// BigOofNSquareSpaceTables.c: creates Tables with O(N^2) size
////////////////////////////////////////////////////////////////////////////////
//
//   Copyright (C) 2008   Georg S. Weber
//
//   This file is part of the Essen Modular package programs.
//
//   The Essen Modular package programs are
//   free software:
//   you can redistribute them and/or modify them
//   under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   The Essen Modular package programs are
//   distributed in the hope that they will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License along
//   with the Essen Modular package programs,
//   see the text file COPYING in this directory.
//   If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////
//
//   history / version (the version is YYYY-MM-DD of the last change)
//   2008-02-22: creation (gsw) of first version
//   2008-03-05: bulk of coding (gsw)
//   2008-03-06: multiplication mod N now for x in [0 .. N] (gsw)
//
////////////////////////////////////////////////////////////////////////////////

#include "./BigOofNSquareSpaceTables.h"
#include "./BigONSpaceTables.h"


//local variable
static EF_sng   currentNinUse = 0x0;        //used also as a flag that
                                            // tables are in use

//local tables
static EF_sng * tableForMultiplicationModN;

static EF_sng * arrayOfRepresentativeIndices;


//functions
inline uint8_t BigOofNSquareSpaceTablesConstruction(EF_sng N)
{
    EF_dbl i, j;        //counter variables
    
    EF_sng sizeOfP1ZNZplusOne = 0x0;    // 0x1 + size of P^1(Z/NZ)
    
    EF_sng * listReps;              //list of representatives of P^1(Z/NZ)
                                    // preceded by (0:0)
    
    EF_sng u, v;        //variables for a representative (u:v)
    
//    EF_sng J;           //inverse (mod N) of j
    
    
    if(0x0 != currentNinUse)
    {
        // caller shall destruct old Tables first
        return(0x1);
    }
    
    if( (N < 0x2) || (N > 65535))
    {
        //invalid N
        return(0x2);
    }
    
    //malloc
    tableForMultiplicationModN = (EF_sng *) malloc( ( N + 0x1 )
                                                    *
                                                    ( N + 0x1 )
                                                    *
                                                    sizeof(EF_sng)
                                                  );
    
    if(NULL == tableForMultiplicationModN)
    {
        return(0x3);
    }
    
    //create table for multiplication ---
    //please note that this is far from being an array,
    //and not all the entries are initialized, i.e. there is quite
    //a bit of storage space being wasted ...
    //note that we explicitly want i in [0 .. N], not only [0 .. N-1]
    //(because we want to negate x mod N, x in [0 .. N-1] by just
    // setting -x := N -x. But this is a priori only in [0 .. N].
    // so we work with x in [0 .. N] instead, and enlarge the
    // multiplication (and addition) lookup-tables accordingly
    for(i = 0x0; i <= (EF_dbl)N; i++)
    {
        for(j = 0x0; j <= (EF_dbl)N; j++)
        {
            tableForMultiplicationModN[ i * j ] = (EF_sng)( (i * j)%N );
        }
    }
    
    //malloc
    arrayOfRepresentativeIndices = (EF_sng *) calloc( N * N, sizeof(EF_sng));
    
    if(NULL == arrayOfRepresentativeIndices)
    {
        free(tableForMultiplicationModN);
        return(0x3);
    }
    
    //create array of representatives
///for the time being, we *rely* on someelse having done
///(and care for) the construction of the O(N) tables!
///TODO: get rid of this dependency, possibly using
///      tableForMultiplicationModN as a temporary array
///      first before initializing it for own purposes
//or(i = 0x0; i < (EF_dbl)N; i++)
//
//   for(j = 0x0; j < (EF_dbl)N; j++)
//   {
//       arrayOfRepresentativeIndices[N * i + j] =
//           indexOfRepresentative( (EF_sng)i, (EF_sng)j );
//   }
//
    sizeOfP1ZNZplusOne = listOfRepresentatives( &listReps );
    
    for( i = 0x1; i < sizeOfP1ZNZplusOne; i++ )
    {
        u = listReps[(i<<0x1)      ];
        v = listReps[(i<<0x1) + 0x1];
        
        for( j = 0x1; j < N; j++)
        {
            if(0x0 != inverseModN(j))
            {
                arrayOfRepresentativeIndices[N
                                             *
                                             tableForMultiplicationModN[u * j]
                                             +
                                             tableForMultiplicationModN[v * j]
                                            ] = i;
            }
        }
    }
    
    currentNinUse = N;  //set flag (and store N locally for reference purposes)
    
    return(0x0);
}

inline void BigOofNSquareSpaceTablesDestruction(void)
{
    if(0x0 != currentNinUse)
    {
        //free memory
        free(arrayOfRepresentativeIndices);
        free(tableForMultiplicationModN);
        
        currentNinUse = 0x0;
    }
    
    return;
}

inline EF_sng multiplyModN(EF_sng a, EF_sng b)
{
//    if(0x0 != currentNinUse)
//    {
        return(tableForMultiplicationModN[ a * b ]);
//    }
//    else
//    {
//        return(0x0);
//    }
}

inline EF_sng indexOfRepresentativeNSquareSpace(EF_sng u, EF_sng v)
{
//    if( (0x0 != currentNinUse)
//        &&
//        (0x0 <= u)
//        &&
//        (0x0 <= v)
//      )
//    {
//        u = u%currentNinUse;
//        v = v%currentNinUse;
//
        return( arrayOfRepresentativeIndices[u * currentNinUse + v] );
//    }
//    else
//    {
//        return(0x0);
//    }
}

inline EF_sng arrayOfRepresentativeIndicesNSquareSpace(EF_sng ** ptrptr_list)
{
//    if(0x0 != currentNinUse)
//    {
        *ptrptr_list = arrayOfRepresentativeIndices;
        
        return(currentNinUse * currentNinUse);
//    }
//    else
//    {
//        *ptrptr_list = (EF_sng *)NULL;
//        
//        return(0x0);
//    }
}

//
// Given 0<= a, b, ka, kb < N with gcd(a,b,N) = 1 and such that we know
// ka = k*a mod N, kb = k*b mod N for some k,
// retrieve the unique k with 0 <= k < N with this property
//
inline EF_sng retrieveKusingMulTable(EF_sng a, EF_sng b, EF_sng ka, EF_sng kb)
{
    //no sanity checks for the time being
    
    EF_sng inverse;     //a or b need not be invertible, but may
    EF_sng i;           //counter variable
    EF_sng ia = 0x0;    //test variable for a
    EF_sng ib = 0x0;    //test variable for b
    
    //printf("%u %u %u %u", a, b, ka, kb);
    if( 0x0 != (inverse = inverseModN(a)) )
    {
        return( tableForMultiplicationModN[ka * inverse] );
    }
    else if( 0x0 != (inverse = inverseModN(b)) )
    {
        return( tableForMultiplicationModN[kb * inverse] );
    }
    else if( (0x0 == ka) && (0x0 == kb) )
    {
        return(0x0);
    }
    else
    {
        //TODO: create look-up-table of size O(N) in composite case
        
        //brute force for the time being
        for( i = 0x1; i < currentNinUse; i++ )
        {
            if( (ia += a) >= currentNinUse ) ia -= currentNinUse;
            if( (ib += b) >= currentNinUse ) ib -= currentNinUse;
            //now   ia == i * a (mod N)   and   ib == i * b (mod N)
            if( (ia == ka) && (ib == kb) )  break;
        }
        return(i);
        //incidentally, on invalid input, we do get
        // k == currentNinUse
        // as output, which also is invalid, and thus could be used as a test
    }
}

