// PrimeFactors.c: creates a table with prime factorization data
////////////////////////////////////////////////////////////////////////////////
//
//   Copyright (C) 2008   Georg S. Weber
//
//   This file is part of the Essen Modular package programs.
//
//   The Essen Modular package programs are
//   free software:
//   you can redistribute them and/or modify them
//   under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   The Essen Modular package programs are
//   distributed in the hope that they will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License along
//   with the Essen Modular package programs,
//   see the text file COPYING in this directory.
//   If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////
//
//   history/version (the version is YYYY-MM-DD of the last change)
//   2008-03-22: creation (gsw)
//   2008-03-23: modified (gsw)
//
////////////////////////////////////////////////////////////////////////////////


#include"./PrimeFactors.h"

//typedefs
typedef struct {EF_sng prime; EF_sng fraction;} PF_tableEntry;


//local variables
static EF_sng currentUpperBound         = 0x0;      //used also as a flag that
                                                    // table is in use
static PF_tableEntry * tableOfPrimesAndFractions;


//functions
uint8_t PrimeFactorsTableConstruction(EF_sng upperBound)
{
    EF_sng i, j, fraction;     //index variables
    EF_sng currentUpperBoundSquareRoot = (EF_sng)(floor(sqrt(upperBound))+0x1);
    
    if(0x0 != currentUpperBound)
    {
        // caller shall destruct old table first
        return(0x1);
    }
    
    if(upperBound < 0x1)
    {
        //invalid upperBound
        return(0x2);
    }
    
    //malloc
    tableOfPrimesAndFractions = (PF_tableEntry *) malloc( (upperBound + 0x1)
                                                          *
                                                          sizeof(PF_tableEntry)
                                                        );
    
    if(NULL == tableOfPrimesAndFractions)
    {
        //not enough memory available
        return(0x3);
    }
    
    
    currentUpperBound = upperBound;
    
    //set first values
    tableOfPrimesAndFractions[0x0].prime    = 0x1; //unused (invalid) entry
    tableOfPrimesAndFractions[0x0].fraction = 0x0;
    tableOfPrimesAndFractions[0x1].prime    = 0x1; //stop marker (not a prime)
    tableOfPrimesAndFractions[0x1].fraction = 0x1;
    
    //clear other table entries
    for( i = 0x2; i <= currentUpperBound; i++)
    {
        tableOfPrimesAndFractions[i].prime    = 0x0;
    }
    
    //set entries
    for( i = 0x2; i < currentUpperBoundSquareRoot; i++)
    {
        if( 0x0 != tableOfPrimesAndFractions[i].prime )
        {
            // i is a composite number and
            // we already found out the smallest prime factor of i before
            continue;
        }
        
        //else i itself is a prime (being not greater than sqrt(upperBound))
        tableOfPrimesAndFractions[i].prime = i;
        tableOfPrimesAndFractions[i].fraction = 0x1;    // 0x1 == i / i
        
        //up to currentBound, run through all multiples j of this prime i
        j = i<<0x1;         //       j == 2 * i
        fraction = 0x2;     //fraction == j / i;
        while( j <= currentUpperBound )
        {
            if(0x0 != tableOfPrimesAndFractions[j].prime)
            {
                //we already found out the smallest prime factor of j before
                // (i.e. j is divisible by another prime smaller than i)
                ;   //do nothing
            }
            else
            {
                //i is being the smallest prime factor of j
                tableOfPrimesAndFractions[j].prime = i;
                tableOfPrimesAndFractions[j].fraction = fraction;
            }
            
            j += i;
            fraction++;
        }
    }
    // the remaining untouched entries belong to primes only
    // (and the index variable i already has the correct start value)
    for(        ; i <= currentUpperBound; i++)
    {
        if( 0x0 != tableOfPrimesAndFractions[i].prime )
        {
            // i is a composite number and
            // we already found out the smallest prime factor of i before
            continue;
        }
        else
        {
            // i is a prime
            tableOfPrimesAndFractions[i].prime = i;
            tableOfPrimesAndFractions[i].fraction = 0x1;    // 0x1 == i / i
        }
    }
    
    return(0x0);
}


void PrimeFactorsTableDestruction(void)
{
    if(0x0 != currentUpperBound)
    {
        //free memory
        free(tableOfPrimesAndFractions);
            
        currentUpperBound = 0x0;
    }
    return;
}

int8_t PrimeFactorization(EF_sng input, EF_sng output[][0x2])
{
    int8_t retVal       = -0x1;
    
    if( (input < 0x1) || (input > currentUpperBound) )
    {
        //error
        return(retVal);
    }
    
    //now the input number has at least one prime factor
    
    retVal = 0x0;   //use as index variable
    
    while( 0x1 != input)
    {
        output[retVal][0x0] = tableOfPrimesAndFractions[input].prime;  //prime
        
        output[retVal][0x1] = 0x0;      //multiplicity (not known yet)
        
        while(tableOfPrimesAndFractions[input].prime == output[retVal][0x0])
        {
            //increase multiplicity of this prime factor
            ++(output[retVal][0x1]);
            
            //input := input/prime
            input = tableOfPrimesAndFractions[input].fraction;
        }
        
        //process next larger prime factor of input (if any)
        retVal++;
    }
    
    //return the number of prime factors, i.e. length(output)
    return(retVal);
}

