// EratosthenesFermat.c: given int x>0, for 0<=y<x, return int y^-1 mod x, or 0
////////////////////////////////////////////////////////////////////////////////
//
//   Copyright (C) 2008   Georg S. Weber
//
//   This file is part of the Essen Modular package programs.
//
//   The Essen Modular package programs are
//   free software:
//   you can redistribute them and/or modify them
//   under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   The Essen Modular package programs are
//   distributed in the hope that they will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License along
//   with the Essen Modular package programs,
//   see the text file COPYING in this directory.
//   If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////
//
//   history / version (the version is YYYY-MM-DD of the last change)
//   2008-02-03: creation (gsw)
//   2008-02-08: minor fixes (gsw)
//   2008-03-10: corrected some comments (when y|x) about the algorithm (gsw)
//   2008-03-14: modified (gsw)
//   2008-03-23: make use of "PrimeFactors" (gsw)
//
////////////////////////////////////////////////////////////////////////////////

#include "./EratosthenesFermat.h"

#include "./PrimeFactors.h"     //used in newer versions of EratosthenesFermatP


//
// function   : EratosthenesFermat
//
// param     x: integer (EF_sng), with respect to which inverses are computed
// param array: array of integers (EF_sng) of length x, which will be filled
//              ATTENTION: No buffer overflow (array length) check is done!
//                         It is the caller's responsibility to take care!!!
//              The content of array[y] after calculation will be:
//                y^{-1} mod x (between 0 and x-1)    if gcd(x,y) == 1, or
//                0                                   if gcd(x,y) > 1 
// returns    : void
//
// last change: 2008-02-03, creation (gsw)
//
#ifndef ERATOSTHENES_FERMAT_OLD_VERSION
void EratosthenesFermat(EF_sng x, EF_sng * array)
{
    EratosthenesFermatP(x, array, 0x1);
    
    return;
}
#else
void EratosthenesFermat(EF_sng x, EF_sng * array)
{
    EF_sng y;                   // main index variable
    EF_sng i;                   // secondary index variable
    EF_sng now_visited,
           next_to_visit;       // temporary variables for orbit indices
    EF_sng backup_inverse;      // temporary backup for a value

    EF_sng  sieve_x =   x;      // bound for sieving (will be lowered)
    
    // small values check
    if(0x0 >= x)
    {
        // do nothing
        // (don't touch an array of length 0 or even negative length ...)
        return;
    }
    else if(0x1 == x)
    {
        array[0x0] = 0x0;
        //we're done
        return;
    }
    else
    {
        ;   //continue
    }
    
    // preset array values
    array[0x0] = 0x0; // 0 certainly is not invertible (since we know x > 0)
    
    array[0x1] = 0x0; // 1 certainly is (and we know x > 1, so 1 != 0), but
                      // we need 0x0 here as a stop marker in the while-loops
    
    for(y = 0x2; y < x; y++)
    {
        array[y] = 0x1; // this is never the correct value for y != 1, so we
                        // can and do use this as a marker for a "virgin" entry
    }

    // run through all entries which are still in virgin state
    for(y = 0x2; y < x; y++)
    {
        if(0x1 != array[y])
        {
            ; // continue (we have been here before)
        }
        else
        {
            //so y < x  and  0x1 == array[y], we have hit a virgin entry
            
            //sieving away the numbers not coprime to x
            if(0x0 != sieve_x)
            {
                if(0x0 == sieve_x%y)
                {
                    // so y is a prime, and a prime factor of x
                    
                    //sieve
                    for(i = y; i < x; i += y)
                    {
                        //for all these multiples of y, certainly gcd(x,y) > 1
                        array[i] = 0x0;
                    }
                    
                    while(0x0 == sieve_x%y)
                    {
                        // remove this prime factor completely
                        sieve_x = sieve_x/y;
                    }
                    
                    if(0x1 == sieve_x)
                    {
                        // this means we removed all prime factors
                        // stop any sieving from now on
                        sieve_x = 0x0;
                    }
                    else
                    {
                        ;   // continue
                    }
                }
                else
                {
                    ;   // continue
                }
            }
            else
            {
                ;   // continue
            }
            
            // main working loops relying on Fermat's "Little Theorem"
            //
            // of which we make use too, according to the Eratosthenes principle
            // saying: "Doing something like factorisation for all integers
            //          below a certain bound all at once can be done *much*
            //          cheaper, than by doing it for each number separately."
            
            // thanks to the sieving done before, we know that gcd(x,y) == 1
            // (we passed through all possible common prime divisors of y and x)
            // so we "only" need to find the inverse of y (mod x)
            
            now_visited = 0x0;  // start value, chosen as a stop marker
    
            // Let's visit the array entries y, y*y, y*y*y, y*y*y*y, ...
            // (i.e. the orbit under multiplication by y --- all these entries
            //  must be invertible mod x)
            // until we hit a non-virgin entry (whose inverse we thus already
            // know!), which we ultimately must. 
            // One way to see this is the pigeon hole principle, since we
            // modify each entry visited to be a value != 0x1).
            // One other way to see this is Fermat's "Little Theorem", which
            // also gives an upper bound on the steps until we finally would
            // arrive at array[1] (the unity, where we did put the value 0x0
            // as a stop marker, and whose inverse we do know of course).
            // (This way we also see that we may not arrive in a circle
            //  back at array[y], before one of the other two cases occurs)
            // In practice, we may hit much earlier some other entry whose
            // inverse already has been calculated at some step before, but
            // in the end, the totally running is absolutely insensitive to
            // this, since we visit each y with gcd(y,x) == 1 exactly once
            // during each of the two while-loops below, at some step.
    
            next_to_visit = y;  // current starting point
            
            //the loop is executed at least once if still array[y] == 0x1,
            //or not at all if y was a prime factor of x (then array[y] == 0x0)
            while(0x1 == array[next_to_visit])
            {
                // we want to remember, where came from (backwards linked list),
                // except for the very first (last?) entry at array[y]
                // (which is set temporarily to the value 0x0)
                array[next_to_visit] = now_visited;
                
                // now "visit" the "next to visit" entry
                now_visited = next_to_visit;
                
                // calculate the successor "next to visit" entry
                //  for the multiplication result we need double the digits, but
                //  then we cut down to numbers smaller than x, which is EF_sng
                next_to_visit = (EF_sng)( ( ((EF_dbl) y)
                                            *
                                            ((EF_dbl) now_visited)
                                          )%((EF_dbl) x)
                                        );
                //  (maybe the compiler is so smart that we don't need each and
                //   every cast in the above statement, but maybe not)
            }
            
            // we have to check explicitly whether we arrived at unity,
            // since we had set array[1] == 0x0 as a stop marker
            if(0x1 == next_to_visit)
            {
                // this just means: y is the inverse (mod x) of "now_visited"!
                
                // carefully go back one step
                next_to_visit = now_visited;
                now_visited = array[next_to_visit];
                // (and don't forget to put the correct value into the array)
                array[next_to_visit] = y;
            }
            else
            {
                ; // continue
            }
            
            // now we have: array[next_to_visit] != 1 (and != 0)
            // this means that the content of "array[next_to_visit]"
            // has been set to the inverse (mod x) of "next_to_visit"
            // at some step before
            // which implies for every y' in the orbit visited just now its
            // inverse mod x is only just one single other pair of a
            // multiplication / division (mod x) away
            
            // to be more precise (all equations are mod x):
            //
            //                   1 == array[next_to_visit] * next_to_visit
            //                     == array[next_to_visit] * y * now_visited
            // so we have:
            //
            //    now_visited^{-1} == array[next_to_visit] * y
            //
            // and we just need to put this value into array[now_visited]
            
            // then we revisit the orbit step by step backwards thanks to
            // the linked list and repeatedly put values in the array
            // until we reach the beginning
            
            // (possibly we had y*y == 1, so don't go into the next loop)

            while(0x0 != now_visited) // we'll hit the stop marker at array[y]
                                      // (or y had been a prime factor of x)
            //
            // we alternatively also could have tested the loop condition thus:
            //     "while(y != next_to_visit)"
            // but testing against zero is slightly faster in machine language
            {
                // we have to backup the inverse since the value of
                // "next_to_visit" will be destroyed
                backup_inverse = (EF_sng)( ( ((EF_dbl)( array[next_to_visit] ))
                                             *
                                             ((EF_dbl) y)
                                           )%((EF_dbl) x)
                                         );
                
                next_to_visit = now_visited;
                
                now_visited = array[next_to_visit];
                
                array[next_to_visit] = backup_inverse;
            }
            
            // now the whole (forward) orbit of y has been correctly filled
            // (doing right here the "backward orbit" of y also is possible,
            //  but does not really save time overall)
            
            // end of "main working loops"

        }//else part of if(0x1 != array[y])
    
    }//for(y = 2; y < x; y++)
    
    array[1] = 0x1; // at last, fill in the correct value at unity ;-)

    // with regard to the running time:
    // clearly linear in x, with very few operations each time
    // (if x itself is a prime, then every 1 < y < x is visited twice in the
    //  while-loops, since they are all invertible, but still there are
    //  very few operations quite near the machine language)
    //
    // with regard to space: negligible apart from the array,
    //                       which has size x times EF_sng
    //
    
    return;
}
#endif //ERATOSTHENES_FERMAT_OLD_VERSION

//
// function   : EratosthenesFermatP_old: OBSOLETE
//
// param     x: integer (EF_sng), with respect to which inverses are computed
// param array: array of integers (EF_sng) of length x, which will be filled
//              ATTENTION: No buffer overflow (array length) check is done!
//                         It is the caller's responsibility to take care!!!
//              The content of array[y] after calculation will be:
//               P* y^{-1} mod x (between 0 and x-1)    if gcd(x,y) == 1, or
//                  0                                   if gcd(x,y) > 1 
// param     P: a prime P such that P mod x is invertible mod x
// returns    : void
//
// last change: 2008-03-12, creation (gsw)
//
inline void EratosthenesFermatP_old(EF_sng x, EF_sng * array, EF_dbl P)
{
    EF_sng y;                   // main index variable
    EF_sng i;                   // secondary index variable
    EF_sng now_visited,
           next_to_visit;       // temporary variables for orbit indices
    EF_sng backup_inverse;      // temporary backup for a value

    //EF_sng sieve_x =   x;       // bound for sieving (will be lowered)
    
    EF_sng Pmodx;               //P modulo x (after we have checked x != 0 ...)
    
    EF_sng x_half   = (x+0x1)>>0x1;      //upper bound for the main for-loop
    
    EF_sng primeFactors[MAX_PRIME_FACTORS][0x2];
    int8_t primeFactorsLength;


    // small values check
    if(0x0 >= x)
    {
        // do nothing
        // (don't touch an array of length 0 or even negative length ...)
        return;
    }
    else if(0x1 == x)
    {
        array[0x0] = 0x0;
        //we're done
        return;
    }
    else if(0x2 == x)
    {
        array[0x0] = 0x0;
        array[0x1] = 0x1;
        //we're done
        return;
    }
    else
    {
        ;   //continue
    }
    
    Pmodx = (EF_sng)(P%((EF_dbl)x));
    // preset array values
    array[0x0] = 0x0; // 0 certainly is not invertible (since we know x > 0)
    
//    array[0x1] = 0x0; // 1 certainly is (and we know x > 1, so 1 != 0), but
//                      // we need 0x0 here as a stop marker in the while-loops
    
    for(y = 0x1; y < x; y++)
    {
        array[y] =  Pmodx;// this is never the correct value for y != 1, so we
                          // use this as a marker for a "virgin" entry
    }
    
//    // run through all entries which are still in virgin state
//    for(y = 0x2; y < x; y++)
    //we do know the inverse of -1 mod x, and so its product with P mod x
    array[x - 0x1] = x - Pmodx;
    
    if( (0x0 == (x&0x1)) && (x != 0x2) )    //x is even and greater equal to 4
    {
        array[x_half] = 0x0;    //x_half is not reached
    }
    //zero out the other y with gcd(y,x) > 1
    //get prime factorization of x
    if( 0x0 > (primeFactorsLength = PrimeFactorization(x, primeFactors)) )
    {
        //error in calling PrimeFactors --- the caller has to initialize that
        return;
    }
    //run through all prime factors of x
    for( i = 0x0; i < (EF_sng)primeFactorsLength; i++)
    {
        //zero out the ith prime factor of x, and all its multiples up to x
        for(y = primeFactors[i][0x0]; y < x; y += primeFactors[i][0x0])
        {
            array[y] = 0x0;
        }
    }
    
    //now we may check whether gcd(P,x) > 1
    if( 0x0 == array[Pmodx])
    {
        return;     //we don't have an error return value, so do nothing else
    }
    array[0x1] = 0x0; // 1 certainly is (and we know x > 1, so 1 != 0), but
                      // we need 0x0 here as a stop marker in the while-loops
    
    // run through all entries which are still in virgin state,
    // considering always x - y together with y
    // (the bound x_half is preset such that for all x, we run up to y < x/2,
    //  which is OK for odd x, and for even x >= 4, y := x/2 had been zeroed
    //  out, so that's OK for even x, too)
    // the orbits of y, and -y mod x, cannot interfere, because we preset the
    // inverse of -1 mod x, so we don't "run through" -1 --- hence we're safe
    for(y = 0x2; y < x_half; y++)
    {
        if( Pmodx != array[y] )
        {
            ; // continue (we have been here before)
        }
        else
        {
            //so y < x  and  0x1 == array[y], we have hit a virgin entry
            
//            //sieving away the numbers not coprime to x
//            if(0x0 != sieve_x)
//            {
//                if(0x0 == sieve_x%y)
//                {
//                    // so y is a prime, and a prime factor of x
//                    
//                    //sieve
//                    for(i = y; i < x; i += y)
//                    {
//                        //for all these multiples of y, certainly gcd(x,y) > 1
//                        array[i] = 0x0;
//                    }
//                    
//                    while(0x0 == sieve_x%y)
//                    {
//                        // remove this prime factor completely
//                        sieve_x = sieve_x/y;
//                    }
//                    
//                    if(0x1 == sieve_x)
//                    {
//                        // this means we removed all prime factors
//                        // stop any sieving from now on
//                        sieve_x = 0x0;
//                    }
//                    else
//                    {
//                        ;   // continue
//                    }
//                }
//                else
//                {
//                    ;   // continue
//                }
//            }
//            else
//            {
//                ;   // continue
//            }
            
            // main working loops relying on Fermat's "Little Theorem"
            //
            // of which we make use too, according to the Eratosthenes principle
            // saying: "Doing something like factorisation for all integers
            //          below a certain bound all at once can be done *much*
            //          cheaper, than by doing it for each number separately."
            
            // thanks to the sieving done before, we know that gcd(x,y) == 1
            // (we passed through all possible common prime divisors of y and x)
            // so we "only" need to find the inverse of y (mod x)
            
            now_visited = 0x0;  // start value, chosen as a stop marker
    
            // Let's visit the array entries y, y*y, y*y*y, y*y*y*y, ...
            // (i.e. the orbit under multiplication by y --- all these entries
            //  must be invertible mod x)
            // until we hit a non-virgin entry (whose inverse we thus already
            // know!), which we ultimately must. 
            // One way to see this is the pigeon hole principle, since we
            // modify each entry visited to be a value != 0x1).
            // One other way to see this is Fermat's "Little Theorem", which
            // also gives an upper bound on the steps until we finally would
            // arrive at array[1] (the unity, where we did put the value 0x0
            // as a stop marker, and whose inverse we do know of course).
            // (This way we also see that we may not arrive in a circle
            //  back at array[y], before one of the other two cases occurs)
            // In practice, we may hit much earlier some other entry whose
            // inverse already has been calculated at some step before, but
            // in the end, the totally running is absolutely insensitive to
            // this, since we visit each y with gcd(y,x) == 1 exactly once
            // during each of the two while-loops below, at some step.
    
            next_to_visit = y;  // current starting point
            
            //the loop is executed at least once if still array[y] == Px,
            //or not at all if y was a prime factor of x (then array[y] == 0x0)
            while( Pmodx == array[next_to_visit] )
            {
                // we want to remember, where came from (backwards linked list),
                // except for the very first (last?) entry at array[y]
                // (which is set temporarily to the value 0x0)
                array[next_to_visit] = now_visited;
                
                // now "visit" the "next to visit" entry
                now_visited = next_to_visit;
                
                // calculate the successor "next to visit" entry
                //  for the multiplication result we need double the digits, but
                //  then we cut down to numbers smaller than x, which is EF_sng
                next_to_visit = (EF_sng)( ( ((EF_dbl) y)
                                            *
                                            ((EF_dbl) now_visited)
                                          )%((EF_dbl) x)
                                        );
                //  (maybe the compiler is so smart that we don't need each and
                //   every cast in the above statement, but maybe not)
            }
            
            // we have to check explicitly whether we arrived at unity,
            // since we had set array[1] == 0x0 as a stop marker
            if(0x1 == next_to_visit)
            {
                // this just means: y is the inverse (mod x) of "now_visited"!
                
                // carefully go back one step
                next_to_visit = now_visited;
                now_visited = array[next_to_visit];
                // (and don't forget to put the correct value into the array)
                array[next_to_visit] = (EF_sng)((P*(EF_dbl)y)%(EF_dbl)x);
    //the inverse of the negative is just the negative of the inverse
    array[x - next_to_visit]    = x - array[next_to_visit];
            }
            else
            {
                ; // continue
            }
            
            // now we have: array[next_to_visit] != 1 (and != 0)
            // this means that the content of "array[next_to_visit]"
            // has been set to the inverse (mod x) of "next_to_visit"
            // at some step before
            // which implies for every y' in the orbit visited just now its
            // inverse mod x is only just one single other pair of a
            // multiplication / division (mod x) away
            
            // to be more precise (all equations are mod x):
            //
            //                   1 == array[next_to_visit] * next_to_visit
            //                     == array[next_to_visit] * y * now_visited
            // so we have:
            //
            //    now_visited^{-1} == array[next_to_visit] * y
            //
            // and we just need to put this value into array[now_visited]
            
            // then we revisit the orbit step by step backwards thanks to
            // the linked list and repeatedly put values in the array
            // until we reach the beginning
            
            // (possibly we had y*y == 1, so don't go into the next loop)

            while(0x0 != now_visited) // we'll hit the stop marker at array[y]
                                      // (or y had been a prime factor of x)
            //
            // we alternatively also could have tested the loop condition thus:
            //     "while(y != next_to_visit)"
            // but testing against zero is slightly faster in machine language
            {
                // we have to backup the inverse since the value of
                // "next_to_visit" will be destroyed
                backup_inverse = (EF_sng)( ( ((EF_dbl)( array[next_to_visit] ))
                                             *
                                             ((EF_dbl) y)
                                           )%((EF_dbl) x)
                                         );
                
                next_to_visit = now_visited;
                
                now_visited = array[next_to_visit];
                
                array[next_to_visit] = backup_inverse;
    //the inverse of the negative is just the negative of the inverse
    array[x - next_to_visit]    = x - backup_inverse;
            }
            
            // now the whole (forward) orbit of y has been correctly filled
            // (doing right here the "backward orbit" of y also is possible,
            //  but does not really save time overall)
            
            // end of "main working loops"

        }//else part of if(0x1 != array[y])
    
    }//for(y = 2; y < x_half; y++)
//    }//for(y = 2; y < x; y++)
    
    array[0x1] = Pmodx;   // at last, fill in the correct value at unity ;-)

    // with regard to the running time:
    // clearly linear in x, with very few operations each time
    // (if x itself is a prime, then every 1 < y < x is visited twice in the
    //  while-loops, since they are all invertible, but still there are
    //  very few operations quite near the machine language)
    //
    // with regard to space: negligible apart from the array,
    //                       which has size x times EF_sng
    //
    
    return;
}





//
// function   : EratosthenesFermatP
//
// param     x: integer (EF_sng), with respect to which inverses are computed
// param array: array of integers (EF_sng) of length x, which will be filled
//              ATTENTION: No buffer overflow (array length) check is done!
//                         It is the caller's responsibility to take care!!!
//              The content of array[y] after calculation will be:
//               P* y^{-1} mod x (between 0 and x-1)    if gcd(x,y) == 1, or
//                  0                                   if gcd(x,y) > 1 
// param     P: a prime P such that P mod x is invertible mod x (not checked)
// returns    : void
//
// last change: 2008-03-12, creation (gsw)
//            : 2008-03-23, use PrimeFactors; consider negatives (gsw)
//
inline void EratosthenesFermatP(EF_sng x, EF_sng * array, EF_dbl P)
{
    EF_sng y;                   // main index variable
    EF_sng i;                   // secondary index variable
    EF_sng now_visited,
           next_to_visit;       // temporary variables for orbit indices
    EF_sng backup_inverse;      // temporary backup for a value
    
    EF_sng Pmodx;               //P modulo x (after we have checked x != 0 ...)
    
    EF_sng x_half   = (x+0x1)/0x2;      //upper bound for the main for-loop
    
    EF_sng primeFactors[MAX_PRIME_FACTORS][0x2];
    int8_t primeFactorsLength;
    
    EF_sng  intermediate_bound = 30;   //upper bound for "small" y
    
    // small values check
    if( 0x2 >= x )
    {
        if(0x2 == x)
        {
//            if( 0x0 != (P&0x1) )    //if x == 2, P shall be odd
//            {
                array[0x0] = 0x0;
                array[0x1] = 0x1;
                //we're done
                ;
//            }
//            else
//            {
//                // do nothing
//                ;
//            }
        }
        else if(0x1 == x)
        {
            array[0x0] = 0x0;
            //we're done
            ;
        }
        else
        {
            // do nothing
            // (don't touch an array of length 0 or even negative length ...)
            ;
        }
    }
    else
    {
    //no indent
    
    // we now know:   x != 0    so we now may divide by x
    Pmodx = (EF_sng)(P%((EF_dbl)x));
//    if( 0x0 == Pmodx )
//    {
//        //the following code relies on P and thus Pmodx being invertible mod x
//        // (we'll check that later, but the case 0x00 == Pmodx has to be 
//        //  checked right now)
//        return;     //we don't have an error return value, so do just nothing
//    }
    
    // preset array values
    for(y = 0x1; y < x; y++)
    {
        array[y] =  Pmodx;// this is never the correct value for y != 1, so we
                          // use this as a marker for a "virgin" entry
    }
    
    array[0x0] = 0x0; // 0 certainly is not invertible (since we know x > 0)
    
    //zero out the other y with gcd(y,x) > 1
    //get prime factorization of x
    primeFactorsLength = PrimeFactorization(x, primeFactors);
//    if( 0x0 > primeFactorsLength )
//    {
//        //error in calling PrimeFactors --- the caller has to initialize that
//        //printf("primeFactorsLength: %d\n", primeFactorsLength);
//        return;
//    }
    //run through all prime factors of x
    for( i = 0x0; i < (EF_sng)primeFactorsLength; i++)
    {
        //zero out the ith prime factor of x, and all its multiples up to x
        for(y = primeFactors[i][0x0]; y < x; y += primeFactors[i][0x0])
        {
            array[y] = 0x0;
        }
    }
    
    //now we may check whether gcd(P,x) > 1   --- take care if Pmodx == 1
//    if( 0x0 == array[Pmodx])
//    {
//        return;     //we don't have an error return value, so do nothing else
//    }
    
    array[0x1] = 0x0; // 1 is invertible (and we know x > 1, so 1 != 0), but
                      // we need 0x0 here as a stop marker in the while-loops
                      // (do this only after checking whether gcd(P,x) > 1)
    
    //we do know the inverse of -1 mod x, and so its product with P mod x
    array[x - 0x1] = x - Pmodx;
    
    // run through all entries which are still in virgin state,
    // considering always x - y together with y
    // (the bound x_half is preset such that for all x, we run up to y < x/2,
    //  which is OK for odd x, and for even x >= 4, y := x/2 had been zeroed
    //  out, so that's OK for even x, too)
    // the orbits of y, and -y mod x, cannot interfere, because we preset the
    // inverse of -1 mod x, so we don't "run through" -1 --- hence we're safe
    
    //do extra code for small y, especially for y = 0x2
    // (see the commented general code below)
    for(y = 0x2; y < 0x3; y++)
    {
        if( Pmodx != array[y] )
        {
            ;   // we have been here before
        }
        else
        {
        now_visited = 0x0;  // start value, chosen as (another) stop marker
        
        next_to_visit = y;  // current starting point
        
        //the loop is executed at least once since Pmodx == array[y]
        while( Pmodx == array[next_to_visit] )
        {
            array[next_to_visit] = now_visited;
            
            // now "visit" the "next to visit" entry
            now_visited = next_to_visit;
            
            //take advantage of y being small, for now:  y == 0x2
            if(x <= ( next_to_visit = (now_visited<<0x1) ))
            {
                next_to_visit -= x;
            }
            //next_to_visit = (EF_sng)( ( ((EF_dbl) y)
            //                            *
            //                            ((EF_dbl) now_visited)
            //                          )%((EF_dbl) x)
            //                        );
            //  (maybe the compiler is so smart that we don't need each and
            //   every cast in the above statement, but maybe not)
        }
        
        // we have to check explicitly whether we arrived at unity,
        // since we had set array[0x1] == 0x0 as a stop marker
        if(0x1 == next_to_visit)
        {
            // this just means: y is the inverse (mod x) of "now_visited"!
            
            // carefully go back one step
            next_to_visit = now_visited;
            now_visited = array[next_to_visit];
            // (and don't forget to put the correct values into the array)
            //take advantage of y being small, for now:  y == 0x2
            if(x <= ( array[next_to_visit] = (Pmodx<<0x1) ))
            {
                array[next_to_visit] -= x;
            }
            
            //array[next_to_visit]    = (EF_sng)( ((EF_dbl)Pmodx * (EF_dbl)y)
            //                                    %(EF_dbl)x
            //                                  );
            //the inverse of the negative is just the negative of the inverse
            array[x - next_to_visit]    = x - array[next_to_visit];
        }
        else
        {
            ; // else we just go on
        }
        
        while(0x0 != now_visited) // we'll hit the stop marker at array[y]
        {
            // we have to backup the inverse since the value of
            // "next_to_visit" will be destroyed

            //take advantage of y being small, for now:  y == 0x2
            if(x <= ( backup_inverse = ((array[next_to_visit])<<0x1) ))
            {
                backup_inverse -= x;
            }
            //backup_inverse = (EF_sng)( ( ((EF_dbl)( array[next_to_visit] ))
            //                             *
            //                             ((EF_dbl) y)
            //                           )%((EF_dbl) x)
            //                         );
            
            next_to_visit = now_visited;
            
            now_visited = array[next_to_visit];
            
            array[next_to_visit]        = backup_inverse;
            //the inverse of the negative is just the negative of the inverse
            array[x - next_to_visit]    = x - backup_inverse;
        }
        
        // end of "main working loops"
        
        }//else-part of if( Pmodx != array[y] )
    
    }//for(y = 0x2; y < 0x3; y++)
    
    //do extra code for other small y, but now also care for small x
    if(intermediate_bound > x_half)
    {
        intermediate_bound = x_half;
    }
    //the start value of y is already the correct one
    for(      ; y < intermediate_bound; y++)
    {
        if( Pmodx != array[y] )
        {
            ;   // we have been here before
        }
        else
        {
        now_visited = 0x0;  // start value, chosen as (another) stop marker
        
        next_to_visit = y;  // current starting point
        
        //the loop is executed at least once since Pmodx == array[y]
        while( Pmodx == array[next_to_visit] )
        {
            array[next_to_visit] = now_visited;
            
            // now "visit" the "next to visit" entry
            now_visited = next_to_visit;
            
            //take advantage of y being small
            next_to_visit = y * now_visited;
            while( x <= next_to_visit )
            {
                next_to_visit -= x;
            }
            //next_to_visit = (EF_sng)( ( ((EF_dbl) y)
            //                            *
            //                            ((EF_dbl) now_visited)
            //                          )%((EF_dbl) x)
            //                        );
            //  (maybe the compiler is so smart that we don't need each and
            //   every cast in the above statement, but maybe not)
        }
        
        // we have to check explicitly whether we arrived at unity,
        // since we had set array[0x1] == 0x0 as a stop marker
        if(0x1 == next_to_visit)
        {
            // this just means: y is the inverse (mod x) of "now_visited"!
            
            // carefully go back one step
            next_to_visit = now_visited;
            now_visited = array[next_to_visit];
            // (and don't forget to put the correct values into the array)
            //take advantage of y being small
            array[next_to_visit] = Pmodx * y;
            while( x <= array[next_to_visit] )
            {
                array[next_to_visit] -= x;
            }
            
            //array[next_to_visit]    = (EF_sng)( ((EF_dbl)Pmodx * (EF_dbl)y)
            //                                    %(EF_dbl)x
            //                                  );
            //the inverse of the negative is just the negative of the inverse
            array[x - next_to_visit]    = x - array[next_to_visit];
        }
        else
        {
            ; // else we just go on
        }
        
        while(0x0 != now_visited) // we'll hit the stop marker at array[y]
        {
            // we have to backup the inverse since the value of
            // "next_to_visit" will be destroyed

            //take advantage of y being small
            backup_inverse = array[next_to_visit] * y;
            while( x <= backup_inverse )
            {
                backup_inverse -= x;
            }
            //backup_inverse = (EF_sng)( ( ((EF_dbl)( array[next_to_visit] ))
            //                             *
            //                             ((EF_dbl) y)
            //                           )%((EF_dbl) x)
            //                         );
            
            next_to_visit = now_visited;
            
            now_visited = array[next_to_visit];
            
            array[next_to_visit]        = backup_inverse;
            //the inverse of the negative is just the negative of the inverse
            array[x - next_to_visit]    = x - backup_inverse;
        }
        
        // end of "main working loops"
        
        }//else-part of if( Pmodx != array[y] )
    
    }//for(       ; y < intermediate_bound; y++)
    
    //do general (larger) y, the start value of y is already the correct one
    for(       ; y < x_half; y++)
    {
        if( Pmodx != array[y] )
        {
            ;   // we have been here before
        }
        else
        {
        //no indent
        
        // so y < x_half  and  Pmodx == array[y], we have hit a virgin entry
        // (and we also know that gcd(x,y) == 1 holds by the above)
        
        // main working loops relying on Fermat's "Little Theorem"
        //
        // of which we make use according to the Eratosthenes principle
        // saying: "Doing something like factorisation for all integers
        //          below a certain bound all at once can be done *much*
        //          cheaper, than by doing it for each number separately."
        
        // We "only" need to find the inverse of y (mod x)
        
        // Let's visit the array entries y, y*y, y*y*y, y*y*y*y, ...
        // (i.e. the orbit under multiplication by y --- all these entries
        //  must be invertible mod x)
        // until we hit a non-virgin entry (whose inverse we thus already
        // know!), which we ultimately must (possibly array[0x1]).
        // We do so before we could arrive in a circle back at array[y]
        // or any other member of the orbit just visited.
        // All this follows from the fact that gcd(y,x) == 1,
        // by basic properties of the abelian group (Z/xZ)^*.
        // Fermat's "Little Theorem" gives an upper bound on the steps until
        // we finally would arrive at array[0x1]
        // (the unity, where we did put the value 0x0 as a stop marker, and
        //  whose inverse we do know of course).
        // In practice, we may hit much earlier some other entry whose
        // inverse already has been calculated at some step before.
        // But in the end, the totally running time is absolutely insensitive
        // to this, since for each y with gcd(y,x) == 1, we "visit" either y
        // or its negative mod x exactly once during the first while-loop
        // below, at some step.

        now_visited = 0x0;  // start value, chosen as (another) stop marker
        
        next_to_visit = y;  // current starting point
        
        //the loop is executed at least once since Pmodx == array[y]
        while( Pmodx == array[next_to_visit] )
        {
            // we want to remember, where came from (backwards linked list),
            // except for the very first (last?) entry at array[y]
            // (which is set temporarily to the value 0x0)
            array[next_to_visit] = now_visited;
            
            // now "visit" the "next to visit" entry
            now_visited = next_to_visit;
            
            // calculate the successor "next to visit" entry
            //  for the multiplication result we need double the digits, but
            //  then we cut down to numbers smaller than x, which is EF_sng
            next_to_visit = (EF_sng)( ( ((EF_dbl) y)
                                        *
                                        ((EF_dbl) now_visited)
                                      )%((EF_dbl) x)
                                    );
            //  (maybe the compiler is so smart that we don't need each and
            //   every cast in the above statement, but maybe not)
        }
        
        // we have to check explicitly whether we arrived at unity,
        // since we had set array[0x1] == 0x0 as a stop marker
        if(0x1 == next_to_visit)
        {
            // this just means: y is the inverse (mod x) of "now_visited"!
            
            // carefully go back one step
            next_to_visit = now_visited;
            now_visited = array[next_to_visit];
            // (and don't forget to put the correct values into the array)
            array[next_to_visit]        = (EF_sng)( ((EF_dbl)Pmodx * (EF_dbl)y)
                                                    %(EF_dbl)x
                                                  );
            //the inverse of the negative is just the negative of the inverse
            array[x - next_to_visit]    = x - array[next_to_visit];
        }
        else
        {
            ; // else we just go on
        }
        
        // now we have: array[next_to_visit] != Pmodx (and != 0x0)
        // this means that the content of "array[next_to_visit]"
        // has been set to the inverse (mod x) of "next_to_visit"
        // at some step before
        // which implies for every y' in the orbit visited just now its
        // inverse mod x is only just one single other pair of a
        // multiplication / division (mod x) away
        
        // to be more precise (all equations are mod x taken out the factor P):
        //
        //                   1 == array[next_to_visit] * next_to_visit
        //                     == array[next_to_visit] * y * now_visited
        // so we have:
        //
        //    now_visited^{-1} == array[next_to_visit] * y
        //
        // and we just need to put this value into array[now_visited]
        
        // then we revisit the orbit step by step backwards thanks to
        // the linked list and repeatedly put values in the array
        // until we reach the beginning
        // (doing so, we're able to also set all the values at the negatives)
        
        // (possibly we had y*y == 1, so don't go into the next loop)

        while(0x0 != now_visited) // we'll hit the stop marker at array[y]
        // we alternatively also could have tested the loop condition thus:
        //     "while(y != next_to_visit)"
        // but testing against zero is slightly faster in machine language
        {
            // we have to backup the inverse since the value of
            // "next_to_visit" will be destroyed
            backup_inverse = (EF_sng)( ( ((EF_dbl)( array[next_to_visit] ))
                                         *
                                         ((EF_dbl) y)
                                       )%((EF_dbl) x)
                                     );
            
            next_to_visit = now_visited;
            
            now_visited = array[next_to_visit];
            
            array[next_to_visit]        = backup_inverse;
            //the inverse of the negative is just the negative of the inverse
            array[x - next_to_visit]    = x - backup_inverse;
        }
        
        // now the whole (forward) orbit of y has been correctly filled
        // (doing right here the "backward orbit" of y also is possible,
        //  but does not really save time overall)
        
        // end of "main working loops"
        
        }//else-part of if( Pmodx != array[y] )
    
    }//for(       ; y < x_half; y++)
    
    array[0x1] = Pmodx;   // at last, fill in the correct value at unity ;-)

    // with regard to the running time:
    // clearly linear in x, with very few operations each time
    // (if x itself is a prime, then exactly half of the  1 < y < x are
    //  visited each twice in the while-loops (their negatives "don't count"),
    //  since they are all invertible, but still there are
    //  very few operations quite near the machine language)
    //
    // with regard to space: negligible apart from the array,
    //                       which has size x times EF_sng
    //
    
    }//else-part of if( 0x2 >= x )
    
    return;
}

