// EssenHeckeTp.c: given level N > 1, and an odd prime p, (p,N)=1, calculate T_p
////////////////////////////////////////////////////////////////////////////////
//
//   Copyright (C) 2008   Georg S. Weber
//
//   This file is part of the Essen Modular package programs.
//
//   The Essen Modular package programs are
//   free software:
//   you can redistribute them and/or modify them
//   under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   The Essen Modular package programs are
//   distributed in the hope that they will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License along
//   with the Essen Modular package programs,
//   see the text file COPYING in this directory.
//   If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////
//
//   history / version (the version is YYYY-MM-DD of the last change)
//   2008-02-11: creation (gsw) of first version
//   2008-03-04: major rewrite (gsw)
//   2008-03-13: minor modification (Warning on gcc3.4.4 / cygwin) (gsw)
//
////////////////////////////////////////////////////////////////////////////////

#include "./EssenHeckeTp.h"
#include "./BigONSpaceTables.h"
#include "./BigOofNSquareSpaceTables.h"
#include "./ApplyHeilbronnMatrices.h"
#include "./PrimeFactors.h"

//
// function       : main
//
// param         N: level (natural number > 1)
// param         p: odd prime not dividing N
// params u_i, v_i: 0 <= u_i,v_i < N and
//                   gcd(u_i, v_i, N) == 1
//                   (u_i : v_i) denotes the ith input Manin symbol
//                  if no u_i, v_i's are given, then we output the whole matrix
//                  otherwise the images of the (u_i : v_i) are output
//
// output       : on stdout (possibly piped to whatever the caller wanted)
//              
// returns      : we use exit()
//
// changes      : 2008-02-11, creation (gsw)
//                2008-02-22, modified (gsw)
//

int main(int argc, char * argv[])
{
    uint32_t N;                     //level (natural number > 1)
    
    uint64_t p;                     //odd prime not dividing N
    
    EF_sng p_squareroot;            //floor(sqrt(p)) + 0x01
    
    EF_sng * workingArray;          //for massive p, we need much memory here
    
    EF_sng sizeOfP1ZNZplusOne = 0x0;    // 0x1 + size of P^1(Z/NZ)
    
    EF_sng * listReps;              //list of representatives of P^1(Z/NZ)
                                    // preceded by (0:0)
    
    EF_sng quantityOfInput;         //the number of input Manin symbols
    
    EF_sng * inputArray;            //input array of Manin symbols (serialized)
    
    EF_dbl * outputArray;           //array for coefficients of output symbols
    
    EF_sng i, j;                    //counter variables
    
    uint32_t u, v;                  //temp variables for Manin symbol input
    
    EF_sng * indexArray;            // N*N array of indices of representatives

    uint64_t NBound;                // N * (N + 1)

    EF_dbl * NworkingArray;         // N*(N+1) array for intermediate coeffs
    
    
    //start input parsing
    if( (argc < 0x3) || (0x0 != (argc - 0x1)%0x2) )
    {
        printf("Error: Wrong number of arguments!\n");
        exit(1);
    }
    
    //first argument: the level
    N = strtol(argv[0x1], (char **)NULL, 10);
    
    if(N < 0x2)
    {
        printf("Error: The level N must be greater than 1!\n");
        exit(1);
    }
    else if( N > 65535)
    {
        printf("Error: Levels N > 65535 are currently not supported!\n");
        exit(1);
    }
    
    //second argument: the odd prime p
    p = strtoll(argv[0x2], (char **)NULL, 10);
    
    if( (p <= 0x1) || (0x0 == p%0x2) )
    {
        printf("Error: The prime p must be odd!\n");
        //well, "9" passes, and is not prime ...
        //let's not bother about this, hoping the caller takes care
        // (below, we check for N and p to be coprime, and if they are,
        //  the algorithms run smoothly through to the end ---
        //  more precisely, the HeilbronnManin family is well-defined
        //  for any odd number m coprime to N, but we do produce the whole
        //  set only under the assumption that m itself is being a prime p,
        //  otherwise some of the matrices will be missing/omitted ...
        //  so in case "p" is odd, but not a prime, the result is garbage)
        exit(1);
    }
    
    p_squareroot = (EF_sng)(floor(sqrt(p)) + 0x1);
    
    workingArray = (EF_sng *) malloc(p_squareroot * sizeof(EF_sng));
    
    if(NULL == workingArray)
    {
        printf("Error: Not enough memory available (p too big)!\n");
        
        exit(1);
    }
    
    if( N <= p_squareroot )
    {
        if( 0x0 != PrimeFactorsTableConstruction(p_squareroot) )
        {
            printf("Error: Not enough memory available (p too big)!\n");
        
            free(workingArray);
            exit(1);
        }
    }
    else
    {
        if( 0x0 != PrimeFactorsTableConstruction(N) )
        {
            printf("Error: Not enough memory available (p too big)!\n");
        
            free(workingArray);
            exit(1);
        }
    }
    
    //preparations
    switch( BigONSpaceTablesConstruction( (EF_sng)N ) )
    {
        case 0x0:
        {
            //success
            //get list
            sizeOfP1ZNZplusOne = listOfRepresentatives( &listReps );
    
            break;
        }
        case 0x1:
        {
            printf("Error: Tables already created --- internal error!\n");
            break;
        }
        case 0x2:
        {
            printf("Error: N must be greater or equal to 2!\n");
            break;
        }
        case 0x5:
        {
            printf("Error: For the time being, N must be a prime power!\n");
            break;
        }
        default:
        {
            printf("Error: Not enough memory for BigONSpaceTables!\n");
            break;
        }
    }//end of switch( BigONSpaceTablesConstruction( (EF_sng)N )
    if(0x0 == sizeOfP1ZNZplusOne)
    {
        //in this case we did run into an error creating the BigONSpaceTables
        PrimeFactorsTableDestruction();
        free(workingArray);
        exit(1);
    }
    
    if(0x0 == inverseModN( (EF_sng)( p % (uint64_t)N ) ))
    {
        printf("Error: The level N and p must be coprime!\n");
        
        BigONSpaceTablesDestruction();
        PrimeFactorsTableDestruction();
        free(workingArray);
        exit(1);
    }
    
    if(0x3 == argc)
    {
        //no input Manin symbols given, i.e. calculate just for all of them
        //and output the whole matrix
        quantityOfInput = sizeOfP1ZNZplusOne - 0x1;
    }
    else
    {
        //argc >= 0x5
        quantityOfInput = ((EF_sng)argc - 0x3)/0x2;
    }
    
    //for every Manin symbol (u:v), there are two entries to store: u and v
    inputArray = (EF_sng *) malloc(0x2 * quantityOfInput * sizeof(EF_sng));
    
    if(NULL == inputArray)
    {
        printf("Error: Not enough memory available for input array?!\n");
        
        BigONSpaceTablesDestruction();
        PrimeFactorsTableDestruction();
        free(workingArray);
        exit(1);
    }
    
    outputArray = (EF_dbl *) calloc(quantityOfInput * sizeOfP1ZNZplusOne,
                                    sizeof(EF_dbl)
                                   );
    //usage of calloc instead of malloc so the storage is initialized to zero
    
    if(NULL == outputArray)
    {
        printf("Error: Not enough memory for output array! (N too big)\n");
        
        free(inputArray);
        BigONSpaceTablesDestruction();
        PrimeFactorsTableDestruction();
        free(workingArray);
        exit(1);
    }
    
    //prepare O(N^2) workingArray
    if (0x0 != BigOofNSquareSpaceTablesConstruction( (EF_sng)N ))
    {
        free(outputArray);
        free(inputArray);
        BigONSpaceTablesDestruction();
        PrimeFactorsTableDestruction();
        free(workingArray);
        exit(1);
    }
    else
    {
        //get created array of indices of representatives
        (void)arrayOfRepresentativeIndicesNSquareSpace( &indexArray );
    }
    
    
    if(0x3 == argc)
    {
        //in this case, the input array must be filled "by hand"
        for(i = 0x0; i < quantityOfInput; i++)
        {
            inputArray[(i<<0x1)      ] = listReps[(i<<0x1) + 0x2];
            inputArray[(i<<0x1) + 0x1] = listReps[(i<<0x1) + 0x3];
        }
    }
    else
    {
        //continue parsing
        for(i = 0x0; i < quantityOfInput; i++)
        {
            u = strtol(argv[(i<<0x1) + 0x3], (char **)NULL, 10);
            v = strtol(argv[(i<<0x1) + 0x4], (char **)NULL, 10);
            
            if( (u < 0x0) || ( u >= N) || (v < 0x0) || ( v >= N) )
            {
                printf("Error: The Manin symbols must be reduced mod N!\n");

                BigOofNSquareSpaceTablesDestruction();
                free(outputArray);
                free(inputArray);
                BigONSpaceTablesDestruction();
                PrimeFactorsTableDestruction();
                free(workingArray);
                exit(1);
            }
            else if( 0x0 == indexOfRepresentativeNSquareSpace( (EF_sng)u,
                                                               (EF_sng)v
                                                             )
                   )
            {
                printf("Error: (u, v) is not a valid Manin symbol!\n");
                printf("(The level N and gcd(u, v) must be coprime.)\n");

                BigOofNSquareSpaceTablesDestruction();
                free(outputArray);
                free(inputArray);
                BigONSpaceTablesDestruction();
                PrimeFactorsTableDestruction();
                free(workingArray);
                exit(1);
            }
            else
            {
                inputArray[(i<<0x1)      ] = (EF_sng)u;
                inputArray[(i<<0x1) + 0x1] = (EF_sng)v;
            }
        }
    }
    
    
    NBound = N * (0x1 + (0x2*N));
    
    NworkingArray = (EF_dbl *) calloc( NBound, sizeof(EF_dbl) );
    //usage of calloc instead of malloc so the storage is initialized to zero
    
    if(NULL == NworkingArray)
    {
        printf("Error: Not enough memory for Nworking array! (N too big)\n");
        
        BigOofNSquareSpaceTablesDestruction();
        free(outputArray);
        free(inputArray);
        BigONSpaceTablesDestruction();
        PrimeFactorsTableDestruction();
        free(workingArray);
        exit(1);
    }
    
    
    // input parsing, preparations, etc. are done
    //perform calculations
    ApplyHeilbronnMatrices( (EF_sng) N,
                            NULL,
                            NULL,
                            sizeOfP1ZNZplusOne,
                            listReps,
                            NULL,
                            indexArray,
                            (EF_dbl) NBound,
                            NworkingArray,
                            (EF_dbl) p,
                            p_squareroot,
                            workingArray,
                            quantityOfInput,
                            inputArray,
                            outputArray,
                            BASMAJI_SEQ_O_OF_NSQUARE
                           );
    
    //output the result
    for(i = 0x0; i < quantityOfInput; i++)
    {
        printf("\n");
        //skip the (0:0) entry
        for(j = 0x1; j < sizeOfP1ZNZplusOne; j++)
        {
            printf("%llu  ", (long long)outputArray[i*sizeOfP1ZNZplusOne + j]);
            printf(" %lu",   (long)listReps[(j<<0x1)      ]);
            printf(" %lu\n", (long)listReps[(j<<0x1) + 0x1]);
        }
    }
    
    //free malloc'ed memory and quit
    free(NworkingArray);
    BigOofNSquareSpaceTablesDestruction();
    free(outputArray);
    free(inputArray);
    BigONSpaceTablesDestruction();
    PrimeFactorsTableDestruction();
    free(workingArray);

    exit(0);
}

