// EssenModular.m: Magma interface of (some) the Essen Modular package programs
////////////////////////////////////////////////////////////////////////////////
//
//   Copyright (C) 2008   Georg S. Weber
//
//   This file is part of the Essen Modular package programs.
//
//   The Essen Modular package programs are
//   free software:
//   you can redistribute them and/or modify them
//   under the terms of the GNU General Public License as published by
//   the Free Software Foundation, either version 3 of the License, or
//   (at your option) any later version.
//
//   The Essen Modular package programs are
//   distributed in the hope that they will be useful,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//   GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License along
//   with the Essen Modular package programs,
//   see the text file COPYING in this directory.
//   If not, see <http://www.gnu.org/licenses/>.
//
////////////////////////////////////////////////////////////////////////////////
//
//   history / version (the version is YYYY-MM-DD of the last change)
//   2008-03-03: creation (gsw) of first version
//   2008-03-06: modified (gsw)
//   2008-03-12: modified to use Pipe() instead of System() (gsw)
//   2008-03-13: added check for N and p being coprime (gsw)
//
////////////////////////////////////////////////////////////////////////////////


intrinsic EssenHeckeOperator(M::ModSym, p::RngIntElt) -> AlgMatElt
{Calculated using the Essen Modular package programs:
A matrix representing the pth Hecke operator (p an odd prime) w.r.t. Basis(M).}


    //currently, the path to the C program(s) has to be adapted "by hand"
    Command := "./EssenHeckeTp";
    
    
    if Dimension(M) eq 0 then
        //nothing to do, so return immediately
        return MatrixAlgebra(BaseRing(Parent(M)),0)!0;
    end if;
    
    //from now on, we know:   Dimension(M) ge 1
    
    if not(IsAmbientSpace(M)) then
        //the following code relies on M being an ambient space, so go "up"
        // (the usual old and shamelessly copied trick)
        A := EssenHeckeOperator(AmbientSpace(M),p);
        F := BaseRing(Parent(A));
        B := Matrix(F, BasisMatrix(VectorSpace(M)));
        return MatrixAlgebra(F, Nrows(B)) ! Solution(B, B*A);
    end if;
    
    //from now on, we know:  IsAmbientSpace(M) eq true
    
    N := Level(M);
    
    //sanity checks
    require Weight(M) eq 2:
        "Only Ms with weight k eq 2 are currently supported";
    require not(IsMultiChar(M)):
        "Ms of MultiChar type currently are not supported";
    require IsTrivial(DirichletCharacter(M)):
        "Only Ms with trivial character eps are currently supported";
    require N ge 2:
        "Only Ms with Level ge 2 are currently supported";
    require N le 65535:
        "Only Ms with Level le 65535 are currently supported";
//    require IsPrimePower(N):
//        "Only Ms whose Level is a prime power are currently supported";
//    require IsAmbientSpace(M):
//        "Only Ms being their own ambient space are currently supported";
//    require Type(BaseField(M)) eq FldRat:
//        "Only Ms with the Rationals as BaseField are currently supported";
    require p lt 2^63:
        "Only values of p less than 2^63 are currently supported";
    require IsPrime(p) and not(p eq 2):
        "Only values of p with p being an odd prime are currently supported";
    require GCD(N, p) eq 1:
        "Only coprime values of N (the Level of M) and p are currently supported";
    
    //build the rest of the command string S (see below)
    Ssymbs := "";   // "S" for "string" of symbols
    
    //since we know Dimension(M) ge 1 (see above), now something does happen
    for i in [1 .. Dimension(M)] do
        ms := ManinSymbol(Basis(M)[i]);
        require #ms eq 1:
            "Oops. We require that each base element of M, a modular symbol,\
             corresponds to a single Manin symbol.\
             This always should be the case, at least for ambient spaces\
             (a case we made sure that we are in),\
             so what did happen here?";
        ms_u := IntegerRing()!ms[1][2][1];
        ms_v := IntegerRing()!ms[1][2][2];
        Ssymbs := Ssymbs * " " * IntegerToString(ms_u);
        Ssymbs := Ssymbs * " " * IntegerToString(ms_v);
    end for;
    
    //S is the input string to the shell command
    S := " " * IntegerToString(N) * " " * IntegerToString(p) * Ssymbs;
    
    //output useful for debugging
    //print S;
    
    //call the corresponding C program of the Essen Modular package
    //System(Command * S * " >./EssenHeckeOperatorTemporaryOutputFile.txt");
    
    //using Pipe() should be even faster and no temp file would be needed, 
    // but we might want to have all possible debug info
    
    //read in the created temp file
    //outString := Read("./EssenHeckeOperatorTemporaryOutputFile.txt");
    
    outString := Pipe(Command * S, "");
    
    //and convert it to a sequence of ints 
    outList := StringToIntegerSequence(outString);
    
    check, sizeOfOutputRow := IsDivisibleBy(#outList, 3 * Dimension(M));
    
    require check:
        "Sorry, internal error w.r.t. the C interface!";
    
    require not(sizeOfOutputRow le 0):
        "Sorry, zero output from the C interface! (Segmentation fault?!)";
    
    // de-serialize the outList, piecing together the result
    // (and do remember that Magma conventions and C conventions about
    //  array indexing are not quite the same)
    matrixRows := [];
    
    for i in [0 .. Dimension(M)-1] do
        outSymbol := M!0;
        for j in [1 .. sizeOfOutputRow] do
            outSymbol +:= BaseField(M)!outList[ 3*(sizeOfOutputRow*i + j)-2 ] \
                          *                                                   \
                          ConvertFromManinSymbol(M,                           \
                                  <1, [outList[ 3*(sizeOfOutputRow*i + j)-1 ],\
                                       outList[ 3*(sizeOfOutputRow*i + j)   ] \
                                      ]                                       \
                                  >             );
        end for;//j in [1 .. sizeOfOutputRow]
        Append(~matrixRows, Eltseq(outSymbol));
        //each Eltseq(outSymbol) is a sequence of length Dimension(M)
    end for;//i in [0 .. Dimension(M)-1]
    //TODO: the amount of data tranfered could be reduced to almost on third,
    //      if the coefficients would be separated from the Manin symbols,
    //      because those need to be tranfered only once (or not at all,
    //       if we knew their exakt enumeration used on the C side ...)
    //REMARK: for levels N large w.r.t. the prime p,
    //            BaseField(M)!outList[ sizeOfRows * i + 3*j-2 ]
    //        might often be zero, so ConvertFromManinSymbol would be
    //        unnecessary, but since we're interested in massive p,
    //        this is not an issue we care about for the time being
    
    //we return a square matrix, as we should
    return Matrix(matrixRows);

end intrinsic;

