/*
********************************************
* Basis.mg                                 *
*                                          *
* Computing a basis of a                   *
* p-adic algebra.                          *
*                                          *
* Gabor Wiese, version of 24/01/2014       *
********************************************
*/

intrinsic pAdicBasis ( L :: SeqEnum, p :: RngIntElt ) -> SeqEnum
{Input: L a list of commuting matrices with integer entries; p a prime.
Output: A subset of L such that the reductions mod p form an F_p-basis of the matrix algebra generated by the reductions; hence, by Nakayama's lemma, this subset is a Z_p-basis for the Z_p-algebra generated by the matrices in L.}
  require L ne [] : "First argument must not be the empty list.";
  require Nrows(L[1]) eq Ncols(L[1]) : "Matrices must be square.";
  require IsPrime(p) : "Second argument must be a prime number.";
  R := BaseRing(L[1]);
  require R eq Integers() : "The base ring of matrices must be the integers.";

  d := Nrows(L[1]);
  km := KMatrixSpace(GF(p),d,d);
  ka := MatrixAlgebra(GF(p),d);

  dim := Dimension(MatrixAlgebra([ka!b : b in L]));
  bas := [];

  n := 1;
  while #bas lt dim do
    T := L[n];
    if IsIndependent([km!a : a in bas] cat [km!T]) then
      bas := bas cat [T];
    end if;
    n := n+1;
  end while;

  return bas;
end intrinsic;

intrinsic pAdicBasis ( L :: SeqEnum ) -> SeqEnum
{Input: L a list of commuting matrices with entries in a p-adic ring.
Output: A subset of L such that the reductions mod p form an F_p-basis of the matrix algebra generated by the reductions; hence, by Nakayama's lemma, this subset is a Z_p-basis for the Z_p-algebra generated by the matrices in L.}
  require L ne [] : "First argument must not be the empty list.";
  require Nrows(L[1]) eq Ncols(L[1]) : "Matrices must be square.";
  R := BaseRing(L[1]);
  require Type(R) eq RngPad : "The base ring of matrices must be a p-adic ring.";

  d := Nrows(L[1]);
  p := Prime(R);
  km := KMatrixSpace(GF(p),d,d);
  ka := MatrixAlgebra(GF(p),d);

  dim := Dimension(MatrixAlgebra([ka!b : b in L]));
  bas := [];

  n := 1;
  while #bas lt dim do
    T := L[n];
    if IsIndependent([km!a : a in bas] cat [km!T]) then
      bas := bas cat [T];
    end if;
    n := n+1;
  end while;

  return bas;
end intrinsic;

